/*
 * Decompiled with CFR 0.152.
 */
package android.arch.paging;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import java.util.AbstractList;
import java.util.ArrayList;
import java.util.List;

final class PagedStorage<T>
extends AbstractList<T> {
    private static final List PLACEHOLDER_LIST = new ArrayList();
    private int mLeadingNullCount;
    private final ArrayList<List<T>> mPages;
    private int mTrailingNullCount;
    private int mPositionOffset;
    private int mStorageCount;
    private int mPageSize;
    private int mNumberPrepended;
    private int mNumberAppended;

    PagedStorage() {
        this.mLeadingNullCount = 0;
        this.mPages = new ArrayList();
        this.mTrailingNullCount = 0;
        this.mPositionOffset = 0;
        this.mStorageCount = 0;
        this.mPageSize = 1;
        this.mNumberPrepended = 0;
        this.mNumberAppended = 0;
    }

    PagedStorage(int leadingNulls, List<T> page, int trailingNulls) {
        this();
        super.init(leadingNulls, page, trailingNulls, 0);
    }

    private PagedStorage(PagedStorage<T> other) {
        this.mLeadingNullCount = other.mLeadingNullCount;
        this.mPages = new ArrayList<List<T>>(other.mPages);
        this.mTrailingNullCount = other.mTrailingNullCount;
        this.mPositionOffset = other.mPositionOffset;
        this.mStorageCount = other.mStorageCount;
        this.mPageSize = other.mPageSize;
        this.mNumberPrepended = other.mNumberPrepended;
        this.mNumberAppended = other.mNumberAppended;
    }

    PagedStorage<T> snapshot() {
        return new PagedStorage<T>(this);
    }

    private void init(int leadingNulls, List<T> page, int trailingNulls, int positionOffset) {
        this.mLeadingNullCount = leadingNulls;
        this.mPages.clear();
        this.mPages.add(page);
        this.mTrailingNullCount = trailingNulls;
        this.mPositionOffset = positionOffset;
        this.mStorageCount = page.size();
        this.mPageSize = page.size();
        this.mNumberPrepended = 0;
        this.mNumberAppended = 0;
    }

    void init(int leadingNulls, @NonNull List<T> page, int trailingNulls, int positionOffset, @NonNull Callback callback) {
        this.init(leadingNulls, page, trailingNulls, positionOffset);
        callback.onInitialized(this.size());
    }

    @Override
    public T get(int i) {
        int pageInternalIndex;
        int localPageIndex;
        if (i < 0 || i >= this.size()) {
            throw new IndexOutOfBoundsException("Index: " + i + ", Size: " + this.size());
        }
        int localIndex = i - this.mLeadingNullCount;
        if (localIndex < 0 || localIndex >= this.mStorageCount) {
            return null;
        }
        if (this.isTiled()) {
            localPageIndex = localIndex / this.mPageSize;
            pageInternalIndex = localIndex % this.mPageSize;
        } else {
            int pageSize;
            pageInternalIndex = localIndex;
            int localPageCount = this.mPages.size();
            for (localPageIndex = 0; localPageIndex < localPageCount && (pageSize = this.mPages.get(localPageIndex).size()) <= pageInternalIndex; pageInternalIndex -= pageSize, ++localPageIndex) {
            }
        }
        List<T> page = this.mPages.get(localPageIndex);
        if (page == null || page.size() == 0) {
            return null;
        }
        return page.get(pageInternalIndex);
    }

    boolean isTiled() {
        return this.mPageSize > 0;
    }

    int getLeadingNullCount() {
        return this.mLeadingNullCount;
    }

    int getTrailingNullCount() {
        return this.mTrailingNullCount;
    }

    int getStorageCount() {
        return this.mStorageCount;
    }

    int getNumberAppended() {
        return this.mNumberAppended;
    }

    int getNumberPrepended() {
        return this.mNumberPrepended;
    }

    int getPageCount() {
        return this.mPages.size();
    }

    int getPositionOffset() {
        return this.mPositionOffset;
    }

    @Override
    public int size() {
        return this.mLeadingNullCount + this.mStorageCount + this.mTrailingNullCount;
    }

    int computeLeadingNulls() {
        List<T> page;
        int total = this.mLeadingNullCount;
        int pageCount = this.mPages.size();
        for (int i = 0; i < pageCount && ((page = this.mPages.get(i)) == null || page == PLACEHOLDER_LIST); ++i) {
            total += this.mPageSize;
        }
        return total;
    }

    int computeTrailingNulls() {
        List<T> page;
        int total = this.mTrailingNullCount;
        for (int i = this.mPages.size() - 1; i >= 0 && ((page = this.mPages.get(i)) == null || page == PLACEHOLDER_LIST); --i) {
            total += this.mPageSize;
        }
        return total;
    }

    T getFirstLoadedItem() {
        return this.mPages.get(0).get(0);
    }

    T getLastLoadedItem() {
        List<T> page = this.mPages.get(this.mPages.size() - 1);
        return page.get(page.size() - 1);
    }

    void prependPage(@NonNull List<T> page, @NonNull Callback callback) {
        int count = page.size();
        if (count == 0) {
            return;
        }
        if (this.mPageSize > 0 && count != this.mPageSize) {
            this.mPageSize = this.mPages.size() == 1 && count > this.mPageSize ? count : -1;
        }
        this.mPages.add(0, page);
        this.mStorageCount += count;
        int changedCount = Math.min(this.mLeadingNullCount, count);
        int addedCount = count - changedCount;
        if (changedCount != 0) {
            this.mLeadingNullCount -= changedCount;
        }
        this.mPositionOffset -= addedCount;
        this.mNumberPrepended += count;
        callback.onPagePrepended(this.mLeadingNullCount, changedCount, addedCount);
    }

    void appendPage(@NonNull List<T> page, @NonNull Callback callback) {
        int count = page.size();
        if (count == 0) {
            return;
        }
        if (this.mPageSize > 0 && (this.mPages.get(this.mPages.size() - 1).size() != this.mPageSize || count > this.mPageSize)) {
            this.mPageSize = -1;
        }
        this.mPages.add(page);
        this.mStorageCount += count;
        int changedCount = Math.min(this.mTrailingNullCount, count);
        int addedCount = count - changedCount;
        if (changedCount != 0) {
            this.mTrailingNullCount -= changedCount;
        }
        this.mNumberAppended += count;
        callback.onPageAppended(this.mLeadingNullCount + this.mStorageCount - count, changedCount, addedCount);
    }

    void initAndSplit(int leadingNulls, @NonNull List<T> multiPageList, int trailingNulls, int positionOffset, int pageSize, @NonNull Callback callback) {
        int pageCount = (multiPageList.size() + (pageSize - 1)) / pageSize;
        for (int i = 0; i < pageCount; ++i) {
            int beginInclusive = i * pageSize;
            int endExclusive = Math.min(multiPageList.size(), (i + 1) * pageSize);
            List<T> sublist = multiPageList.subList(beginInclusive, endExclusive);
            if (i == 0) {
                int initialTrailingNulls = trailingNulls + multiPageList.size() - sublist.size();
                this.init(leadingNulls, sublist, initialTrailingNulls, positionOffset);
                continue;
            }
            int insertPosition = leadingNulls + beginInclusive;
            this.insertPage(insertPosition, sublist, null);
        }
        callback.onInitialized(this.size());
    }

    public void insertPage(int position, @NonNull List<T> page, @Nullable Callback callback) {
        int newPageSize = page.size();
        if (newPageSize != this.mPageSize) {
            boolean onlyEndPagePresent;
            int size = this.size();
            boolean addingLastPage = position == size - size % this.mPageSize && newPageSize < this.mPageSize;
            boolean bl = onlyEndPagePresent = this.mTrailingNullCount == 0 && this.mPages.size() == 1 && newPageSize > this.mPageSize;
            if (!onlyEndPagePresent && !addingLastPage) {
                throw new IllegalArgumentException("page introduces incorrect tiling");
            }
            if (onlyEndPagePresent) {
                this.mPageSize = newPageSize;
            }
        }
        int pageIndex = position / this.mPageSize;
        this.allocatePageRange(pageIndex, pageIndex);
        int localPageIndex = pageIndex - this.mLeadingNullCount / this.mPageSize;
        List<T> oldPage = this.mPages.get(localPageIndex);
        if (oldPage != null && oldPage != PLACEHOLDER_LIST) {
            throw new IllegalArgumentException("Invalid position " + position + ": data already loaded");
        }
        this.mPages.set(localPageIndex, page);
        if (callback != null) {
            callback.onPageInserted(position, page.size());
        }
    }

    private void allocatePageRange(int minimumPage, int maximumPage) {
        int newStorageAllocated;
        int leadingNullPages = this.mLeadingNullCount / this.mPageSize;
        if (minimumPage < leadingNullPages) {
            for (int i = 0; i < leadingNullPages - minimumPage; ++i) {
                this.mPages.add(0, null);
            }
            newStorageAllocated = (leadingNullPages - minimumPage) * this.mPageSize;
            this.mStorageCount += newStorageAllocated;
            this.mLeadingNullCount -= newStorageAllocated;
            leadingNullPages = minimumPage;
        }
        if (maximumPage >= leadingNullPages + this.mPages.size()) {
            newStorageAllocated = Math.min(this.mTrailingNullCount, (maximumPage + 1 - (leadingNullPages + this.mPages.size())) * this.mPageSize);
            for (int i = this.mPages.size(); i <= maximumPage - leadingNullPages; ++i) {
                this.mPages.add(this.mPages.size(), null);
            }
            this.mStorageCount += newStorageAllocated;
            this.mTrailingNullCount -= newStorageAllocated;
        }
    }

    public void allocatePlaceholders(int index, int prefetchDistance, int pageSize, Callback callback) {
        if (pageSize != this.mPageSize) {
            if (pageSize < this.mPageSize) {
                throw new IllegalArgumentException("Page size cannot be reduced");
            }
            if (this.mPages.size() != 1 || this.mTrailingNullCount != 0) {
                throw new IllegalArgumentException("Page size can change only if last page is only one present");
            }
            this.mPageSize = pageSize;
        }
        int maxPageCount = (this.size() + this.mPageSize - 1) / this.mPageSize;
        int minimumPage = Math.max((index - prefetchDistance) / this.mPageSize, 0);
        int maximumPage = Math.min((index + prefetchDistance) / this.mPageSize, maxPageCount - 1);
        this.allocatePageRange(minimumPage, maximumPage);
        int leadingNullPages = this.mLeadingNullCount / this.mPageSize;
        for (int pageIndex = minimumPage; pageIndex <= maximumPage; ++pageIndex) {
            int localPageIndex = pageIndex - leadingNullPages;
            if (this.mPages.get(localPageIndex) != null) continue;
            this.mPages.set(localPageIndex, PLACEHOLDER_LIST);
            callback.onPagePlaceholderInserted(pageIndex);
        }
    }

    public boolean hasPage(int pageSize, int index) {
        int leadingNullPages = this.mLeadingNullCount / pageSize;
        if (index < leadingNullPages || index >= leadingNullPages + this.mPages.size()) {
            return false;
        }
        List<T> page = this.mPages.get(index - leadingNullPages);
        return page != null && page != PLACEHOLDER_LIST;
    }

    @Override
    public String toString() {
        StringBuilder ret = new StringBuilder("leading " + this.mLeadingNullCount + ", storage " + this.mStorageCount + ", trailing " + this.getTrailingNullCount());
        for (int i = 0; i < this.mPages.size(); ++i) {
            ret.append(" ").append(this.mPages.get(i));
        }
        return ret.toString();
    }

    static interface Callback {
        public void onInitialized(int var1);

        public void onPagePrepended(int var1, int var2, int var3);

        public void onPageAppended(int var1, int var2, int var3);

        public void onPagePlaceholderInserted(int var1);

        public void onPageInserted(int var1, int var2);
    }
}

