package aQute.lib.osgi;

import java.io.*;
import java.lang.annotation.*;
import java.lang.reflect.*;
import java.nio.*;
import java.util.*;

public class Clazz {
    public static enum QUERY {
        IMPLEMENTS, EXTENDS, IMPORTS, NAMED, ANY, VERSION, CONCRETE, ABSTRACT, PUBLIC, ANNOTATION, RUNTIMEANNOTATIONS, CLASSANNOTATIONS
    };

    public static EnumSet<QUERY> HAS_ARGUMENT = EnumSet.of(QUERY.IMPLEMENTS,
                                                      QUERY.EXTENDS,
                                                      QUERY.IMPORTS,
                                                      QUERY.NAMED,
                                                      QUERY.VERSION,
                                                      QUERY.ANNOTATION);

    static protected class Assoc {
        Assoc(byte tag, int a, int b) {
            this.tag = tag;
            this.a = a;
            this.b = b;
        }

        byte tag;
        int  a;
        int  b;
    }

    final static byte                SkipTable[] = { 0, // 0 non existent
            -1, // 1 CONSTANT_utf8 UTF 8, handled in
            // method
            -1, // 2
            4, // 3 CONSTANT_Integer
            4, // 4 CONSTANT_Float
            8, // 5 CONSTANT_Long (index +=2!)
            8, // 6 CONSTANT_Double (index +=2!)
            -1, // 7 CONSTANT_Class
            2, // 8 CONSTANT_String
            4, // 9 CONSTANT_FieldRef
            4, // 10 CONSTANT_MethodRef
            4, // 11 CONSTANT_InterfaceMethodRef
            4, // 12 CONSTANT_NameAndType
                                                 };

    boolean                          isAbstract;
    boolean                          isPublic;
    boolean                          hasRuntimeAnnotations;
    boolean                          hasClassAnnotations;

    String                           className;
    Object                           pool[];
    int                              intPool[];
    Map<String, Map<String, String>> imports     = new HashMap<String, Map<String, String>>();
    String                           path;
    int                              minor       = 0;
    int                              major       = 0;

    String                           sourceFile;
    Set<String>                      xref;
    Set<Integer>                     classes;
    Set<Integer>                     descriptors;
    Set<String>                      annotations;
    int                              forName     = 0;
    int                              class$      = 0;
    String[]                         interfaces;
    String                           zuper;
    ClassDataCollector               cd          = null;
    Resource                         resource;

    public Clazz(String path, Resource resource) {
        this.path = path;
        this.resource = resource;
    }

    public Set<String> parseClassFile() throws IOException {
        return parseClassFileWithCollector(null);
    }

    public Set<String> parseClassFile(InputStream in) throws IOException {
        return parseClassFile(in, null);
    }

    public Set<String> parseClassFileWithCollector(ClassDataCollector cd)
            throws IOException {
        InputStream in = resource.openInputStream();
        try {
            return parseClassFile(in, cd);
        } finally {
            in.close();
        }
    }

    public Set<String> parseClassFile(InputStream in, ClassDataCollector cd)
            throws IOException {
        DataInputStream din = new DataInputStream(in);
        try {
            this.cd = cd;
            return parseClassFile(din);
        } finally {
            cd = null;
            din.close();
        }
    }

    Set<String> parseClassFile(DataInputStream in) throws IOException {

        xref = new HashSet<String>();
        classes = new HashSet<Integer>();
        descriptors = new HashSet<Integer>();

        boolean crawl = false; // Crawl the byte code
        int magic = in.readInt();
        if (magic != 0xCAFEBABE)
            throw new IOException("Not a valid class file (no CAFEBABE header)");

        minor = in.readUnsignedShort(); // minor version
        major = in.readUnsignedShort(); // major version
        int count = in.readUnsignedShort();
        pool = new Object[count];
        intPool = new int[count];

        process: for (int poolIndex = 1; poolIndex < count; poolIndex++) {
            byte tag = in.readByte();
            switch (tag) {
            case 0:
                break process;
            case 1:
                constantUtf8(in, poolIndex);
                break;

            case 3:
                constantInteger(in, poolIndex);
                break;

            case 4:
                constantFloat(in, poolIndex);
                break;

            // For some insane optimization reason are
            // the long and the double two entries in the
            // constant pool. See 4.4.5
            case 5:
                constantLong(in, poolIndex);
                poolIndex++;
                break;

            case 6:
                constantDouble(in, poolIndex);
                poolIndex++;
                break;

            case 7:
                constantClass(in, poolIndex);
                break;

            case 8:
                constantString(in, poolIndex);
                break;

            case 10: // Method ref
                methodRef(in, poolIndex);
                break;

            // Name and Type
            case 12:
                nameAndType(in, poolIndex, tag);
                break;

            // We get the skip count for each record type
            // from the SkipTable. This will also automatically
            // abort when
            default:
                if (tag == 2)
                    throw new IOException("Invalid tag " + tag);
                in.skipBytes(SkipTable[tag]);
                break;
            }
        }

        pool(pool, intPool);
        /*
         * Parse after the constant pool, code thanks to Hans Christian
         * Falkenberg
         */

        int access_flags = in.readUnsignedShort(); // access
        isAbstract = Modifier.isAbstract(access_flags);
        isPublic = Modifier.isPublic(access_flags);
        int this_class = in.readUnsignedShort();
        className = (String) pool[intPool[this_class]];
        if (cd != null)
            cd.classBegin(access_flags, className);

        try {

            int super_class = in.readUnsignedShort();
            zuper = (String) pool[intPool[super_class]];
            if (zuper != null) {
                String pack = getPackage(zuper);
                    packageReference(pack);
                if (cd != null)
                    cd.extendsClass(zuper);
            }

            int interfacesCount = in.readUnsignedShort();
            if (interfacesCount > 0) {
                interfaces = new String[interfacesCount];
                for (int i = 0; i < interfacesCount; i++)
                    interfaces[i] = (String) pool[intPool[in
                            .readUnsignedShort()]];
                if (cd != null)
                    cd.implementsInterfaces(interfaces);
            }

            int fieldsCount = in.readUnsignedShort();
            for (int i = 0; i < fieldsCount; i++) {
                access_flags = in.readUnsignedShort(); // skip access flags
                int name_index = in.readUnsignedShort();
                int descriptor_index = in.readUnsignedShort();

                // Java prior to 1.5 used a weird
                // static variable to hold the com.X.class
                // result construct. If it did not find it
                // it would create a variable class$com$X
                // that would be used to hold the class
                // object gotten with Class.forName ...
                // Stupidly, they did not actively use the
                // class name for the field type, so bnd
                // would not see a reference. We detect
                // this case and add an artificial descriptor
                String name = pool[name_index].toString(); // name_index
                if (name.startsWith("class$")) {
                    crawl = true;
                }
                if (cd != null)
                    cd.field(access_flags, pool[descriptor_index].toString());
                descriptors.add(new Integer(descriptor_index));
                doAttributes(in, ElementType.FIELD, false);
            }

            //
            // Check if we have to crawl the code to find
            // the ldc(_w) <string constant> invokestatic Class.forName
            // if so, calculate the method ref index so we
            // can do this efficiently
            //
            if (crawl) {
                forName = findMethod("java/lang/Class", "forName",
                        "(Ljava/lang/String;)Ljava/lang/Class;");
                class$ = findMethod(className, "class$",
                        "(Ljava/lang/String;)Ljava/lang/Class;");
            }

            //
            // Handle the methods
            //
            int methodCount = in.readUnsignedShort();
            for (int i = 0; i < methodCount; i++) {
                access_flags = in.readUnsignedShort();
                int name_index = in.readUnsignedShort();
                int descriptor_index = in.readUnsignedShort();
                // String s = (String) pool[name_index];
                descriptors.add(new Integer(descriptor_index));
                String name = pool[name_index].toString();
                String descriptor = pool[descriptor_index].toString();
                if ("<init>".equals(name)) {
                    if (cd != null)
                        cd.constructor(access_flags, descriptor);
                    doAttributes(in, ElementType.CONSTRUCTOR, crawl);
                } else {
                    if (cd != null)
                        cd.method(access_flags, name, descriptor);
                    doAttributes(in, ElementType.METHOD, crawl);
                }
            }

            doAttributes(in, ElementType.TYPE, false);

            //
            // Now iterate over all classes we found and
            // parse those as well. We skip duplicates
            //

            for (int n : classes) {
                String clazz = (String) pool[n];
                if (clazz.endsWith(";") || clazz.startsWith("["))
                    parseReference(clazz, 0);
                else {

                    String pack = getPackage(clazz);
                    packageReference(pack);
                }
            }

            //
            // Parse all the descriptors we found
            //

            for (Iterator<Integer> e = descriptors.iterator(); e.hasNext();) {
                Integer index = e.next();
                String prototype = (String) pool[index.intValue()];
                if (prototype != null)
                    parseDescriptor(prototype);
                else
                    System.err.println("Unrecognized descriptor: " + index);
            }
            Set<String> xref = this.xref;
            reset();
            return xref;
        } finally {
            if (cd != null)
                cd.classEnd();
        }
    }

    private void constantFloat(DataInputStream in, int poolIndex)
            throws IOException {
        if (cd != null)
            pool[poolIndex] = in.readInt();
        else
            in.skipBytes(4);
    }

    private void constantInteger(DataInputStream in, int poolIndex)
            throws IOException {
        if (cd != null)
            intPool[poolIndex] = in.readInt();
        else
            in.skipBytes(4);
    }

    protected void pool(Object[] pool, int[] intPool) {
    }

    /**
     * @param in
     * @param poolIndex
     * @param tag
     * @throws IOException
     */
    protected void nameAndType(DataInputStream in, int poolIndex, byte tag)
            throws IOException {
        int name_index = in.readUnsignedShort();
        int descriptor_index = in.readUnsignedShort();
        descriptors.add(new Integer(descriptor_index));
        pool[poolIndex] = new Assoc(tag, name_index, descriptor_index);
    }

    /**
     * @param in
     * @param poolIndex
     * @param tag
     * @throws IOException
     */
    private void methodRef(DataInputStream in, int poolIndex)
            throws IOException {
        int class_index = in.readUnsignedShort();
        int name_and_type_index = in.readUnsignedShort();
        pool[poolIndex] = new Assoc((byte) 10, class_index, name_and_type_index);
    }

    /**
     * @param in
     * @param poolIndex
     * @throws IOException
     */
    private void constantString(DataInputStream in, int poolIndex)
            throws IOException {
        int string_index = in.readUnsignedShort();
        intPool[poolIndex] = string_index;
    }

    /**
     * @param in
     * @param poolIndex
     * @throws IOException
     */
    protected void constantClass(DataInputStream in, int poolIndex)
            throws IOException {
        int class_index = in.readUnsignedShort();
        classes.add(new Integer(class_index));
        intPool[poolIndex] = class_index;
    }

    /**
     * @param in
     * @throws IOException
     */
    protected void constantDouble(DataInputStream in, int poolIndex)
            throws IOException {
        if (cd != null)
            pool[poolIndex] = in.readDouble();
        else
            in.skipBytes(8);
    }

    /**
     * @param in
     * @throws IOException
     */
    protected void constantLong(DataInputStream in, int poolIndex)
            throws IOException {
        if (cd != null) {
            pool[poolIndex] = in.readLong();
        } else
            in.skipBytes(8);
    }

    /**
     * @param in
     * @param poolIndex
     * @throws IOException
     */
    protected void constantUtf8(DataInputStream in, int poolIndex)
            throws IOException {
        // CONSTANT_Utf8

        String name = in.readUTF();
        xref.add(name);
        pool[poolIndex] = name;
    }

    /**
     * Find a method reference in the pool that points to the given class,
     * methodname and descriptor.
     * 
     * @param clazz
     * @param methodname
     * @param descriptor
     * @return index in constant pool
     */
    private int findMethod(String clazz, String methodname, String descriptor) {
        for (int i = 1; i < pool.length; i++) {
            if (pool[i] instanceof Assoc) {
                Assoc methodref = (Assoc) pool[i];
                if (methodref.tag == 10) {
                    // Method ref
                    int class_index = methodref.a;
                    int class_name_index = intPool[class_index];
                    if (clazz.equals(pool[class_name_index])) {
                        int name_and_type_index = methodref.b;
                        Assoc name_and_type = (Assoc) pool[name_and_type_index];
                        if (name_and_type.tag == 12) {
                            // Name and Type
                            int name_index = name_and_type.a;
                            int type_index = name_and_type.b;
                            if (methodname.equals(pool[name_index])) {
                                if (descriptor.equals(pool[type_index])) {
                                    return i;
                                }
                            }
                        }
                    }
                }
            }
        }
        return -1;
    }

    /**
     * Called for each attribute in the class, field, or method.
     * 
     * @param in
     *            The stream
     * @throws IOException
     */
    private void doAttributes(DataInputStream in, ElementType member,
            boolean crawl) throws IOException {
        int attributesCount = in.readUnsignedShort();
        for (int j = 0; j < attributesCount; j++) {
            // skip name CONSTANT_Utf8 pointer
            doAttribute(in, member, crawl);
        }
    }

    /**
     * Process a single attribute, if not recognized, skip it.
     * 
     * @param in
     *            the data stream
     * @throws IOException
     */
    private void doAttribute(DataInputStream in, ElementType member,
            boolean crawl) throws IOException {
        int attribute_name_index = in.readUnsignedShort();
        String attributeName = (String) pool[attribute_name_index];
        long attribute_length = in.readInt();
        attribute_length &= 0xFFFFFFFF;
        if ("RuntimeVisibleAnnotations".equals(attributeName))
            doAnnotations(in, member, RetentionPolicy.RUNTIME);
        else if ("RuntimeVisibleParameterAnnotations".equals(attributeName))
            doParameterAnnotations(in, member, RetentionPolicy.RUNTIME);
        else if ("RuntimeInvisibleAnnotations".equals(attributeName))
            doAnnotations(in, member, RetentionPolicy.CLASS);
        else if ("RuntimeInvisibleParameterAnnotations".equals(attributeName))
            doParameterAnnotations(in, member, RetentionPolicy.CLASS);
        else if ("SourceFile".equals(attributeName))
            doSourceFile(in);
        else if ("Code".equals(attributeName) && crawl)
            doCode(in);
        else if ("Signature".equals(attributeName))
            doSignature(in, member);
        else {
            if (attribute_length > 0x7FFFFFFF) {
                throw new IllegalArgumentException("Attribute > 2Gb");
            }
            in.skipBytes((int) attribute_length);
        }
    }

    /**
     * Handle a signature
     * 
     * <pre>
     * Signature_attribute { 
     *     u2 attribute_name_index; 
     *     u4 attribute_length; 
     *     u2 signature_index; 
     *     } 
     * </pre>
     * 
     * @param member
     */

    void doSignature(DataInputStream in, ElementType member) throws IOException {
        int signature_index = in.readUnsignedShort();
        String signature = (String) pool[signature_index];

        // The type signature is kind of weird,
        // lets skip it for now. Seems to be some kind of
        // type variable name index but it does not seem to
        // conform to the language specification.
        if (member != ElementType.TYPE)
            parseDescriptor(signature);
    }

    /**
     * <pre>
     * Code_attribute {
     * 		u2 attribute_name_index;
     * 		u4 attribute_length;
     * 		u2 max_stack;
     * 		u2 max_locals;
     * 		u4 code_length;
     * 		u1 code[code_length];
     * 		u2 exception_table_length;
     * 		{    	u2 start_pc;
     * 		      	u2 end_pc;
     * 		      	u2  handler_pc;
     * 		      	u2  catch_type;
     * 		}	exception_table[exception_table_length];
     * 		u2 attributes_count;
     * 		attribute_info attributes[attributes_count];
     * 	}
     * </pre>
     * 
     * @param in
     * @param pool
     * @throws IOException
     */
    private void doCode(DataInputStream in) throws IOException {
        /* int max_stack = */in.readUnsignedShort();
        /* int max_locals = */in.readUnsignedShort();
        int code_length = in.readInt();
        byte code[] = new byte[code_length];
        in.readFully(code);
        crawl(code);
        int exception_table_length = in.readUnsignedShort();
        in.skipBytes(exception_table_length * 8);
        doAttributes(in, ElementType.METHOD, false);
    }

    /**
     * We must find Class.forName references ...
     * 
     * @param code
     */
    protected void crawl(byte[] code) {
        ByteBuffer bb = ByteBuffer.wrap(code);
        bb.order(ByteOrder.BIG_ENDIAN);
        int lastReference = -1;

        while (bb.remaining() > 0) {
            int instruction = 0xFF & bb.get();
            switch (instruction) {
            case OpCodes.ldc:
                lastReference = 0xFF & bb.get();
                break;

            case OpCodes.ldc_w:
                lastReference = 0xFFFF & bb.getShort();
                break;

            case OpCodes.invokestatic:
                int methodref = 0xFFFF & bb.getShort();
                if ((methodref == forName || methodref == class$)
                        && lastReference != -1
                        && pool[intPool[lastReference]] instanceof String) {
                    String clazz = (String) pool[intPool[lastReference]];
                    if (clazz.startsWith("[") || clazz.endsWith(";"))
                        parseReference(clazz, 0);
                    else {
                        int n = clazz.lastIndexOf('.');
                        if (n > 0 )
                            packageReference(clazz.substring(0, n));
                    }
                }
                break;

            case OpCodes.tableswitch:
                // Skip to place divisible by 4
                while ((bb.position() & 0x3) != 0)
                    bb.get();
                /* int deflt = */
                bb.getInt();
                int low = bb.getInt();
                int high = bb.getInt();
                bb.position(bb.position() + (high - low + 1) * 4);
                lastReference = -1;
                break;

            case OpCodes.lookupswitch:
                // Skip to place divisible by 4
                while ((bb.position() & 0x3) != 0)
                    bb.get();
                /* deflt = */
                bb.getInt();
                int npairs = bb.getInt();
                bb.position(bb.position() + npairs * 8);
                lastReference = -1;
                break;

            default:
                lastReference = -1;
                bb.position(bb.position() + OpCodes.OFFSETS[instruction]);
            }
        }
    }

    private void doSourceFile(DataInputStream in) throws IOException {
        int sourcefile_index = in.readUnsignedShort();
        this.sourceFile = pool[sourcefile_index].toString();
    }

    private void doParameterAnnotations(DataInputStream in, ElementType member,
            RetentionPolicy policy) throws IOException {
        int num_parameters = in.readUnsignedByte();
        for (int p = 0; p < num_parameters; p++) {
            if (cd != null)
                cd.parameter(p);
            doAnnotations(in, member, policy);
        }
    }

    private void doAnnotations(DataInputStream in, ElementType member,
            RetentionPolicy policy) throws IOException {
        int num_annotations = in.readUnsignedShort(); // # of annotations
        for (int a = 0; a < num_annotations; a++) {
            if (cd == null)
                doAnnotation(in, member, policy, false);
            else {
                Annotation annotion = doAnnotation(in, member, policy, true);
                cd.annotation(annotion);
            }
        }
    }

    private Annotation doAnnotation(DataInputStream in, ElementType member,
            RetentionPolicy policy, boolean collect) throws IOException {
        int type_index = in.readUnsignedShort();
        if (annotations == null)
            annotations = new HashSet<String>();

        annotations.add(pool[type_index].toString());

        if (policy == RetentionPolicy.RUNTIME) {
            descriptors.add(new Integer(type_index));
            hasRuntimeAnnotations = true;
        } else {
            hasClassAnnotations = true;
        }
        String name = (String) pool[type_index];
        int num_element_value_pairs = in.readUnsignedShort();
        Map<String, Object> elements = null;
        for (int v = 0; v < num_element_value_pairs; v++) {
            int element_name_index = in.readUnsignedShort();
            String element = (String) pool[element_name_index];
            Object value = doElementValue(in, member, policy, collect);
            if (collect) {
                if (elements == null)
                    elements = new LinkedHashMap<String, Object>();
                elements.put(element, value);
            }
        }
        if (collect)
            return new Annotation(name, elements, member, policy);
        else
            return null;
    }

    private Object doElementValue(DataInputStream in, ElementType member,
            RetentionPolicy policy, boolean collect) throws IOException {
        char tag = (char) in.readUnsignedByte();
        switch (tag) {
        case 'B': // Byte
        case 'C': // Character
        case 'I': // Integer
        case 'S': // Short
            int const_value_index = in.readUnsignedShort();
            return intPool[const_value_index];

        case 'D': // Double
        case 'F': // Float
        case 's': // String
        case 'J': // Long
            const_value_index = in.readUnsignedShort();
            return pool[const_value_index];

        case 'Z': // Boolean
            const_value_index = in.readUnsignedShort();
            return intPool[const_value_index] == 0 ? false : true;

        case 'e': // enum constant
            int type_name_index = in.readUnsignedShort();
            if (policy == RetentionPolicy.RUNTIME)
                descriptors.add(new Integer(type_name_index));
            int const_name_index = in.readUnsignedShort();
            return pool[const_name_index];

        case 'c': // Class
            int class_info_index = in.readUnsignedShort();
            if (policy == RetentionPolicy.RUNTIME)
                descriptors.add(new Integer(class_info_index));
            return pool[class_info_index];

        case '@': // Annotation type
            return doAnnotation(in, member, policy, collect);

        case '[': // Array
            int num_values = in.readUnsignedShort();
            Object[] result = new Object[num_values];
            for (int i = 0; i < num_values; i++) {
                result[i] = doElementValue(in, member, policy, collect);
            }
            return result;

        default:
            throw new IllegalArgumentException(
                    "Invalid value for Annotation ElementValue tag " + tag);
        }
    }

    /**
     * Add a new package reference.
     * 
     * @param pack
     *            A '.' delimited package name
     */
    void packageReference(String pack) {
        if (!imports.containsKey(pack))
            imports.put(pack, new LinkedHashMap<String, String>());
    }

    /**
     * This method parses a descriptor and adds the package of the descriptor to
     * the referenced packages.
     * 
     * The syntax of the descriptor is:
     * 
     * <pre>
     *   descriptor ::= ( '(' references ')' )? references
     *   references ::= reference *
     *   reference  ::= 'L' classname ( '&lt;' references '&gt;' )? ';' | 'B' | 'Z' | ... | '+' | '-' | '[' 
     * </pre>
     * 
     * This methods uses heavy recursion to parse the descriptor and a roving
     * pointer to limit the creation of string objects.
     * 
     * @param descriptor
     *            The to be parsed descriptor
     * @param rover
     *            The pointer to start at
     */
    public void parseDescriptor(String descriptor) {
        // Some descriptors are weird, they start with a generic
        // declaration that contains ':', not sure what they mean ...
        if (descriptor.charAt(0) == '<')
            return;

        int rover = 0;
        if (descriptor.charAt(rover) == '(') {
            rover = parseReferences(descriptor, rover + 1, ')');
            rover++;
        }
        parseReferences(descriptor, rover, (char) 0);
    }

    /**
     * Parse a sequence of references. A sequence ends with a given character or
     * when the string ends.
     * 
     * @param descriptor
     *            The whole descriptor.
     * @param rover
     *            The index in the descriptor
     * @param delimiter
     *            The end character or 0
     * @return the last index processed, one character after the delimeter
     */
    int parseReferences(String descriptor, int rover, char delimiter) {
        while (rover < descriptor.length()
                && descriptor.charAt(rover) != delimiter) {
            rover = parseReference(descriptor, rover);
        }
        return rover;
    }

    /**
     * Parse a single reference. This can be a single character or an object
     * reference when it starts with 'L'.
     * 
     * @param descriptor
     *            The descriptor
     * @param rover
     *            The place to start
     * @return The return index after the reference
     */
    int parseReference(String descriptor, int rover) {

        char c = descriptor.charAt(rover);
        while (c == '[')
            c = descriptor.charAt(++rover);

        if (c == '<') {
            rover = parseReferences(descriptor, rover + 1, '>');
        } else if (c == 'T') {
            // Type variable name
            rover++;
            while (descriptor.charAt(rover) != ';')
                rover++;
        } else if (c == 'L') {
            StringBuilder sb = new StringBuilder();
            rover++;
            int lastSlash = -1;
            while ((c = descriptor.charAt(rover)) != ';') {
                if (c == '<') {
                    rover = parseReferences(descriptor, rover + 1, '>');
                } else if (c == '/') {
                    lastSlash = sb.length();
                    sb.append('.');
                } else
                    sb.append(c);
                rover++;
            }
            if (cd != null)
                cd.addReference(sb.toString());

            if (lastSlash > 0)
                packageReference(sb.substring(0, lastSlash));
        } else {
            if ("+-*BCDFIJSZV".indexOf(c) < 0)
                System.out.println("Should not skip: " + c);
        }

        // this skips a lot of characters
        // [, *, +, -, B, etc.

        return rover + 1;
    }

    public static String getPackage(String clazz) {
        int n = clazz.lastIndexOf('/');
        if (n < 0)
            return ".";
        return clazz.substring(0, n).replace('/', '.');
    }

    public Map<String, Map<String, String>> getReferred() {
        return imports;
    }

    String getClassName() {
        return className;
    }

    public String getPath() {
        return path;
    }

    public String getSourceFile() {
        return sourceFile;
    }

    /**
     * .class construct for different compilers
     * 
     * sun 1.1 Detect static variable class$com$acme$MyClass 1.2 " 1.3 " 1.4 "
     * 1.5 ldc_w (class) 1.6 "
     * 
     * eclipse 1.1 class$0, ldc (string), invokestatic Class.forName 1.2 " 1.3 "
     * 1.5 ldc (class) 1.6 "
     * 
     * 1.5 and later is not an issue, sun pre 1.5 is easy to detect the static
     * variable that decodes the class name. For eclipse, the class$0 gives away
     * we have a reference encoded in a string.
     * compilerversions/compilerversions.jar contains test versions of all
     * versions/compilers.
     */

    public void reset() {
        pool = null;
        intPool = null;
        xref = null;
        classes = null;
        descriptors = null;
    }

    public boolean is(QUERY query, Instruction instr,
            Map<String, Clazz> classspace) {
        switch (query) {
        case ANY:
            return true;

        case NAMED:
            if (instr.matches(getClassName()))
                return !instr.isNegated();
            return false;

        case VERSION:
            String v = major + "/" + minor;
            if (instr.matches(v))
                return !instr.isNegated();
            return false;

        case IMPLEMENTS:
            for (int i = 0; interfaces != null && i < interfaces.length; i++) {
                if (instr.matches(interfaces[i]))
                    return !instr.isNegated();
            }
            break;
        case EXTENDS:
            if (zuper == null)
                return false;

            if (instr.matches(zuper))
                return !instr.isNegated();
            break;

        case PUBLIC:
            return !isPublic;

        case CONCRETE:
            return !isAbstract;

        case ANNOTATION:
            if (annotations == null)
                return false;

            for (String annotation : annotations) {
                if (instr.matches(annotation))
                    return !instr.isNegated();
            }

            return false;

        case RUNTIMEANNOTATIONS:
            return hasClassAnnotations;
        case CLASSANNOTATIONS:
            return hasClassAnnotations;

        case ABSTRACT:
            return isAbstract;

        case IMPORTS:
            for (String imp : imports.keySet()) {
                if (instr.matches(imp.replace('.', '/')))
                    return !instr.isNegated();
            }
        }

        if (zuper == null || classspace == null)
            return false;

        Clazz clazz = classspace.get(zuper + ".class");
        if (clazz == null)
            return false;

        return clazz.is(query, instr, classspace);
    }

    public String toString() {
        return getFQN();
    }

    public String getFQN() {
        return getClassName().replace('/', '.');
    }
}
