/*
* Copyright 2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
*
* Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file
* except in compliance with the License. A copy of the License is located at
*
* http://aws.amazon.com/apache2.0/
*
* or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for
* the specific language governing permissions and limitations under the License.
*/


package com.amazon.ask.model;

import java.util.Objects;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonSubTypes;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import java.time.OffsetDateTime;
import com.fasterxml.jackson.annotation.JsonProperty;

/**
 * A request object that provides the details of the user’s request. The request body contains the parameters necessary for the service to perform its logic and generate a response.
 */

@JsonTypeInfo(use = JsonTypeInfo.Id.NAME, include = JsonTypeInfo.As.PROPERTY, property = "type", visible = true )
@JsonSubTypes({
  @JsonSubTypes.Type(value = com.amazon.ask.model.interfaces.audioplayer.PlaybackStoppedRequest.class, name = "AudioPlayer.PlaybackStopped"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.interfaces.audioplayer.PlaybackFinishedRequest.class, name = "AudioPlayer.PlaybackFinished"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.events.skillevents.SkillEnabledRequest.class, name = "AlexaSkillEvent.SkillEnabled"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.services.listManagement.ListUpdatedEventRequest.class, name = "AlexaHouseholdListEvent.ListUpdated"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.interfaces.playbackcontroller.PreviousCommandIssuedRequest.class, name = "PlaybackController.PreviousCommandIssued"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.events.skillevents.SkillDisabledRequest.class, name = "AlexaSkillEvent.SkillDisabled"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.interfaces.display.ElementSelectedRequest.class, name = "Display.ElementSelected"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.services.listManagement.ListItemsUpdatedEventRequest.class, name = "AlexaHouseholdListEvent.ItemsUpdated"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.events.skillevents.PermissionChangedRequest.class, name = "AlexaSkillEvent.SkillPermissionChanged"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.services.listManagement.ListItemsCreatedEventRequest.class, name = "AlexaHouseholdListEvent.ItemsCreated"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.events.skillevents.AccountLinkedRequest.class, name = "AlexaSkillEvent.SkillAccountLinked"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.SessionEndedRequest.class, name = "SessionEndedRequest"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.services.listManagement.ListCreatedEventRequest.class, name = "AlexaHouseholdListEvent.ListCreated"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.interfaces.audioplayer.PlaybackStartedRequest.class, name = "AudioPlayer.PlaybackStarted"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.IntentRequest.class, name = "IntentRequest"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.interfaces.audioplayer.PlaybackNearlyFinishedRequest.class, name = "AudioPlayer.PlaybackNearlyFinished"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.services.listManagement.ListItemsDeletedEventRequest.class, name = "AlexaHouseholdListEvent.ItemsDeleted"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.interfaces.messaging.MessageReceivedRequest.class, name = "Messaging.MessageReceived"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.interfaces.audioplayer.PlaybackFailedRequest.class, name = "AudioPlayer.PlaybackFailed"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.interfaces.system.ExceptionEncounteredRequest.class, name = "System.ExceptionEncountered"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.events.skillevents.PermissionAcceptedRequest.class, name = "AlexaSkillEvent.SkillPermissionAccepted"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.services.listManagement.ListDeletedEventRequest.class, name = "AlexaHouseholdListEvent.ListDeleted"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.interfaces.playbackcontroller.NextCommandIssuedRequest.class, name = "PlaybackController.NextCommandIssued"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.interfaces.playbackcontroller.PauseCommandIssuedRequest.class, name = "PlaybackController.PauseCommandIssued"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.interfaces.playbackcontroller.PlayCommandIssuedRequest.class, name = "PlaybackController.PlayCommandIssued"),
  @JsonSubTypes.Type(value = com.amazon.ask.model.LaunchRequest.class, name = "LaunchRequest"),
})

public abstract class Request{

  protected String type = null;

  @JsonProperty("requestId")
  protected String requestId = null;

  @JsonProperty("timestamp")
  protected OffsetDateTime timestamp = null;

  @JsonProperty("locale")
  protected String locale = null;

  protected Request() {
  }

  /**
    * Describes the type of the request.
  * @return type
  **/
  @JsonIgnore
  public String getType() {
    return type;
  }

  /**
    * Represents the unique identifier for the specific request.
  * @return requestId
  **/
  public String getRequestId() {
    return requestId;
  }

  /**
    * Provides the date and time when Alexa sent the request as an ISO 8601 formatted string. Used to verify the request when hosting your skill as a web service.
  * @return timestamp
  **/
  public OffsetDateTime getTimestamp() {
    return timestamp;
  }

  /**
    * A string indicating the user’s locale. For example: en-US.
  * @return locale
  **/
  public String getLocale() {
    return locale;
  }

  @Override
  public boolean equals(java.lang.Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    Request request = (Request) o;
    return Objects.equals(this.type, request.type) &&
        Objects.equals(this.requestId, request.requestId) &&
        Objects.equals(this.timestamp, request.timestamp) &&
        Objects.equals(this.locale, request.locale);
  }

  @Override
  public int hashCode() {
    return Objects.hash(type, requestId, timestamp, locale);
  }

  @Override
  public String toString() {
    StringBuilder sb = new StringBuilder();
    sb.append("class Request {\n");
    
    sb.append("    type: ").append(toIndentedString(type)).append("\n");
    sb.append("    requestId: ").append(toIndentedString(requestId)).append("\n");
    sb.append("    timestamp: ").append(toIndentedString(timestamp)).append("\n");
    sb.append("    locale: ").append(toIndentedString(locale)).append("\n");
    sb.append("}");
    return sb.toString();
  }

  /**
   * Convert the given object to string with each line indented by 4 spaces
   * (except the first line).
   */
  private String toIndentedString(java.lang.Object o) {
    if (o == null) {
      return "null";
    }
    return o.toString().replace("\n", "\n    ");
  }

}

