/*
 * Copyright (C) 2017 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.builder.merge;

import com.android.annotations.NonNull;
import com.android.annotations.Nullable;

import com.google.common.base.Preconditions;
import com.google.common.io.Closer;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.ListIterator;
import java.util.stream.Collectors;

/**
 * An input stream that combines a list of input streams. The result when reading the entire
 * combined stream is the same as the result when reading the list of sub-streams in order.
 *
 * <p>This class is not thread-safe.
 */
final class CombinedInputStream extends InputStream {

    @NonNull private final ListIterator<InputStream> modifiableStreamList;
    @Nullable private InputStream currentStream;

    private final boolean newLinePadding;
    private int lastByteRead;

    /**
     * Creates a {@code CombinedInputStream} instance from the given list of input streams.
     *
     * <p>If {@code newLinePadding} is set to {@code true}, this class inserts '\n' characters
     * in-between sub-streams into the resulting combined stream . If a sub-stream already ends with
     * a '\n' character, then the inserting is skipped for that position.
     *
     * @param inputStreams the list of input streams to combine
     * @param newLinePadding whether '\n' characters should be inserted in-between input streams
     */
    public CombinedInputStream(@NonNull List<MergeInput> inputStreams, boolean newLinePadding) {
        Preconditions.checkArgument(!inputStreams.isEmpty());
        inputStreams.forEach(Preconditions::checkNotNull);

        this.modifiableStreamList =
                inputStreams.stream()
                        .map(MergeInput::getStream)
                        .collect(Collectors.toList())
                        .listIterator();
        this.currentStream = modifiableStreamList.next();

        this.newLinePadding = newLinePadding;
        this.lastByteRead = -1;
    }

    @Override
    public int read() throws IOException {
        //noinspection WhileLoopSpinsOnField
        while (currentStream != null) {
            int byteRead = currentStream.read();
            if (byteRead != -1) {
                return lastByteRead = byteRead;
            }
            advanceStream();
        }
        return lastByteRead = -1;
    }

    @Override
    public int read(@NonNull byte buffer[], int offset, int length) throws IOException {
        Preconditions.checkNotNull(buffer);
        Preconditions.checkArgument(offset >= 0);
        Preconditions.checkArgument(length >= 0);
        Preconditions.checkArgument(offset + length <= buffer.length);

        //noinspection WhileLoopSpinsOnField
        while (currentStream != null) {
            int byteCount = currentStream.read(buffer, offset, length);
            if (byteCount > 0) {
                lastByteRead = buffer[offset + byteCount - 1];
            }
            if (byteCount >= 0) {
                return byteCount;
            }
            advanceStream();
        }

        lastByteRead = -1;
        return -1;
    }

    private void advanceStream() throws IOException {
        Preconditions.checkNotNull(currentStream);
        currentStream.close();

        // When new line padding is enabled, if a sub-stream (except the last one) does not end with
        // a '\n' character, we will insert that character into the resulting combined stream
        if (newLinePadding) {
            if (modifiableStreamList.hasNext() && lastByteRead != '\n') {
                modifiableStreamList.add(new ByteArrayInputStream(new byte[] {'\n'}));
                // Move back the cursor to before the added stream
                modifiableStreamList.previous();
            }
        }

        currentStream = modifiableStreamList.hasNext() ? modifiableStreamList.next() : null;
    }

    @Override
    public int available() throws IOException {
        return currentStream != null ? currentStream.available() : 0;
    }

    @Override
    public void close() throws IOException {
        if (currentStream == null) {
            return;
        }

        try (Closer closer = Closer.create()) {
            closer.register(currentStream);
            modifiableStreamList.forEachRemaining(closer::register);
            currentStream = null;
        }
    }
}
