/*
 * Copyright (C) 2018 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.build.gradle.tasks

import com.android.build.gradle.internal.scope.BuildArtifactsHolder
import com.android.build.gradle.internal.scope.InternalArtifactType.AP_GENERATED_SOURCES
import com.android.build.gradle.internal.scope.InternalArtifactType.ANNOTATION_PROCESSOR_LIST
import com.android.build.gradle.internal.scope.VariantScope
import com.android.build.gradle.internal.tasks.VariantAwareTask
import com.android.build.gradle.internal.tasks.factory.VariantTaskCreationAction
import org.gradle.api.tasks.CacheableTask
import com.android.build.gradle.options.BooleanOption
import org.gradle.api.file.DirectoryProperty
import org.gradle.api.file.FileTree
import org.gradle.api.file.RegularFileProperty
import org.gradle.api.model.ObjectFactory
import org.gradle.api.tasks.InputFile
import org.gradle.api.tasks.InputFiles
import org.gradle.api.tasks.Internal
import org.gradle.api.tasks.OutputDirectory
import org.gradle.api.tasks.PathSensitive
import org.gradle.api.tasks.PathSensitivity
import org.gradle.api.tasks.SkipWhenEmpty
import org.gradle.api.tasks.TaskProvider
import org.gradle.api.tasks.compile.JavaCompile
import org.gradle.api.tasks.incremental.IncrementalTaskInputs
import javax.inject.Inject

/**
 * Task to perform annotation processing only, without compiling.
 *
 * This task may or may not be created depending on whether it is needed. See the documentation of
 * [AndroidJavaCompile] for more details.
 */
@CacheableTask
abstract class ProcessAnnotationsTask @Inject constructor(objects: ObjectFactory) : JavaCompile(),
    VariantAwareTask {

    @get:Internal
    override lateinit var variantName: String

    @get:InputFile
    @get:PathSensitive(PathSensitivity.NONE)
    abstract val processorListFile: RegularFileProperty

    @get:Internal
    lateinit var sourceFileTrees: () -> List<FileTree>
        private set

    @InputFiles
    @PathSensitive(PathSensitivity.RELATIVE)
    @SkipWhenEmpty
    fun getSources(): FileTree {
        return this.project.files(this.sourceFileTrees()).asFileTree
    }

    @get:OutputDirectory
    val annotationProcessorOutputDirectory: DirectoryProperty = objects.directoryProperty()

    override fun compile(inputs: IncrementalTaskInputs) {
        // Perform annotation processing only (but only if the user did not request -proc:none)
        if (options.compilerArgs.contains(PROC_NONE)) {
            return
        } else {
            options.compilerArgs.add(PROC_ONLY)
        }

        // Disable incremental mode as Gradle's JavaCompile currently does not work correctly in
        // incremental mode when -proc:only is used. We will revisit this issue later and
        // investigate what it means for an annotation-processing-only task to be incremental.
        options.isIncremental = false

        // For incremental compile to work, Gradle requires individual sources to be added one by
        // one, rather than adding one single composite FileTree aggregated from the individual
        // sources (method getSources() above).
        for (source in sourceFileTrees()) {
            this.source(source)
        }

        // If no annotation processors are present, the Java compiler will proceed to compile the
        // source files even when the -proc:only option is specified (see
        // https://bugs.openjdk.java.net/browse/JDK-6378728). That would mess up the setup of this
        // task which is meant to do annotation processing only. Therefore, we need to return here
        // in that case. (This decision can't be made at the task's configuration as we don't want
        // to resolve annotation processors at configuration time.)
        val annotationProcessors =
            readAnnotationProcessorsFromJsonFile(processorListFile.get().asFile)
        if (annotationProcessors.isEmpty()) {
            return
        }

        super.compile(inputs)
    }

    class CreationAction(variantScope: VariantScope) :
        VariantTaskCreationAction<ProcessAnnotationsTask>(variantScope) {

        override val name: String
            get() = variantScope.getTaskName("process", "AnnotationsWithJavac")

        override val type: Class<ProcessAnnotationsTask>
            get() = ProcessAnnotationsTask::class.java

        override fun handleProvider(taskProvider: TaskProvider<out ProcessAnnotationsTask>) {
            super.handleProvider(taskProvider)

            variantScope.artifacts.producesDir(
                AP_GENERATED_SOURCES,
                BuildArtifactsHolder.OperationType.APPEND,
                taskProvider,
                ProcessAnnotationsTask::annotationProcessorOutputDirectory
            )
        }

        override fun configure(task: ProcessAnnotationsTask) {
            super.configure(task)

            // Configure properties that are shared between AndroidJavaCompile and
            // ProcessAnnotationTask
            task.configureProperties(variantScope)

            // Configure properties that are specific to ProcessAnnotationTask
            variantScope.artifacts.setTaskInputToFinalProduct(
                ANNOTATION_PROCESSOR_LIST,
                task.processorListFile
            )

            // Configure properties for annotation processing
            task.configurePropertiesForAnnotationProcessing(
                variantScope,
                task.annotationProcessorOutputDirectory
            )

            // Collect the list of source files to process
            task.sourceFileTrees = { variantScope.variantData.javaSources }

            // Since this task does not output compiled classes, destinationDir will not be used.
            // However, Gradle requires this property to be set, so let's just set it to the
            // annotation processor output directory for convenience.
            task.setDestinationDir(task.annotationProcessorOutputDirectory.asFile)
        }
    }

    companion object {

        /**
         * Determine whether [ProcessAnnotationsTask] should be created.
         *
         * As documented at [AndroidJavaCompile], [ProcessAnnotationsTask] is needed if all of the
         * following conditions are met:
         *   1. Incremental compilation is requested (either by the user through the DSL or by
         *      default)
         *   2. Kapt is not used
         *   3. The [BooleanOption.ENABLE_SEPARATE_ANNOTATION_PROCESSING] flag is enabled
         */
        @JvmStatic
        fun taskShouldBeCreated(variantScope: VariantScope): Boolean {
            val globalScope = variantScope.globalScope
            val project = globalScope.project
            val compileOptions = globalScope.extension.compileOptions
            val separateAnnotationProcessingFlag = globalScope
                .projectOptions
                .get(BooleanOption.ENABLE_SEPARATE_ANNOTATION_PROCESSING)

            return compileOptions.incremental ?: DEFAULT_INCREMENTAL_COMPILATION
                    && !project.pluginManager.hasPlugin(KOTLIN_KAPT_PLUGIN_ID)
                    && separateAnnotationProcessingFlag
        }
    }
}