/*
 * Copyright (C) 2019 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.build.gradle.internal.tasks

import com.android.SdkConstants
import com.android.build.gradle.internal.scope.ExistingBuildElements
import com.android.build.gradle.internal.scope.InternalArtifactType
import com.android.build.gradle.internal.scope.VariantScope
import com.android.build.gradle.internal.tasks.factory.VariantTaskCreationAction
import com.android.ide.common.symbols.generateMinifyKeepRules
import com.android.ide.common.symbols.parseManifest
import com.google.common.collect.Iterables
import org.gradle.api.file.DirectoryProperty
import org.gradle.api.file.RegularFileProperty
import org.gradle.api.tasks.CacheableTask
import org.gradle.api.tasks.InputFiles
import org.gradle.api.tasks.OutputFile
import org.gradle.api.tasks.PathSensitive
import org.gradle.api.tasks.PathSensitivity
import org.gradle.api.tasks.TaskProvider
import java.io.File
import java.io.Serializable
import java.nio.file.Files
import javax.inject.Inject

/**
 * Generates the Proguard keep rules file `aapt_rules.txt`. In the past this was generated by AAPT,
 * but since we don't call AAPT(2) for packaging library resources we are generating the file
 * ourselves.
 *
 * This task takes the merged manifest and the merged resources directory (local only, small merge),
 * collects information from them and generates the keep rules file.
 */
@CacheableTask
abstract class GenerateLibraryProguardRulesTask : NonIncrementalTask() {

    @get:OutputFile
    abstract val proguardOutputFile: RegularFileProperty

    @get:InputFiles
    @get:PathSensitive(PathSensitivity.RELATIVE)
    abstract val manifestFiles: DirectoryProperty

    @get:InputFiles
    @get:PathSensitive(PathSensitivity.RELATIVE)
    abstract val inputResourcesDir: DirectoryProperty

    override fun doTaskAction() {
        val manifest = Iterables.getOnlyElement(
            ExistingBuildElements.from(InternalArtifactType.MERGED_MANIFESTS, manifestFiles))
            .outputFile

        getWorkerFacadeWithWorkers().use {
            it.submit(
                GenerateProguardRulesRunnable::class.java,
                GenerateProguardRulesParams(
                    manifestFile = manifest,
                    proguardOutputFile = proguardOutputFile.get().asFile,
                    inputResourcesDir = inputResourcesDir.get().asFile
                ))
        }

    }

    data class GenerateProguardRulesParams(
        val manifestFile: File,
        val proguardOutputFile: File,
        val inputResourcesDir: File
    ): Serializable

    class GenerateProguardRulesRunnable @Inject constructor(
        private val params: GenerateProguardRulesParams
    ): Runnable {
        override fun run() {
            // Generate `aapt_rules.txt` containing keep rules for Proguard.
            Files.write(
                params.proguardOutputFile.toPath(),
                generateMinifyKeepRules(
                    parseManifest(params.manifestFile), params.inputResourcesDir)
            )
        }
    }

    class CreationAction(
        variantScope: VariantScope
    ): VariantTaskCreationAction<GenerateLibraryProguardRulesTask>(variantScope) {

        override val name: String
            get() = variantScope.getTaskName("generate", "LibraryProguardRules")
        override val type: Class<GenerateLibraryProguardRulesTask>
            get() = GenerateLibraryProguardRulesTask::class.java

        override fun handleProvider(taskProvider: TaskProvider<out GenerateLibraryProguardRulesTask>) {
            super.handleProvider(taskProvider)

            variantScope.artifacts.producesFile(
                InternalArtifactType.AAPT_PROGUARD_FILE,
                taskProvider,
                GenerateLibraryProguardRulesTask::proguardOutputFile,
                SdkConstants.FN_AAPT_RULES
            )
        }

        override fun configure(task:GenerateLibraryProguardRulesTask) {
            super.configure(task)

            variantScope.artifacts.setTaskInputToFinalProduct(
                InternalArtifactType.PACKAGED_RES,
                task.inputResourcesDir
            )

             variantScope.artifacts.setTaskInputToFinalProduct(
                 InternalArtifactType.MERGED_MANIFESTS, task.manifestFiles)
        }
    }
}