/*
 * Copyright (C) 2022 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.tools.profgen

import java.io.File

fun dumpProfile(
    os: Appendable,
    profile: ArtProfile,
    apk: Apk,
    obf: ObfuscationMap,
    strict: Boolean = true,
) {
    for ((dexFile, dexFileData) in profile.profileData) {
        val file = apk.dexes.find { it.name == extractName(dexFile.name) }
          ?: if (strict) {
              throw IllegalStateException("Cannot find Dex File ${dexFile.name}")
          } else {
              continue
          }

        for ((key, method) in dexFileData.methods) {
            // Method data is not guaranteed to exist given they might be stored as
            // extra descriptors.

            // Context:
            // java/com/google/android/art/profiles/ArtProfileLoaderForS.java;l=469;rcl=382185618
            val dexMethod = file.methodPool.getOrNull(key)
            if (method.flags != 0 && dexMethod != null) {
                val deobfuscated = obf.deobfuscate(dexMethod)
                method.print(os)
                deobfuscated.print(os)
                os.append('\n')
            }
        }

        for (key in dexFileData.typeIndexes) {
            val dexClass = file.typePool.getOrNull(key)
            if (dexClass != null) {
                val deobfuscated = obf.deobfuscate(dexClass)
                for (type in deobfuscated) {
                    os.append(type)
                    os.append('\n')
                }
            }
        }
    }
}

fun dumpProfile(
        file: File,
        profile: ArtProfile,
        apk: Apk,
        obf: ObfuscationMap,
        strict: Boolean = true
) {
    dumpProfile(file.outputStream().bufferedWriter(), profile, apk, obf, strict = strict)
}

/**
 * Extracts the dex name from the incoming profile key.
 *
 * `base.apk!classes.dex` is a typical profile key.
 *
 * On Android O or lower, the delimiter used is a `:`.
 */
private fun extractName(profileKey: String): String {
    var index = profileKey.indexOf("!")
    if (index < 0) {
        index = profileKey.indexOf(":")
    }
    return profileKey.substring(index + 1)
}
