/*
 * Decompiled with CFR 0.152.
 */
package com.android.zipflinger;

import com.android.zipflinger.CentralDirectory;
import com.android.zipflinger.EndOfCentralDirectory;
import com.android.zipflinger.Entry;
import com.android.zipflinger.Ints;
import com.android.zipflinger.Location;
import com.android.zipflinger.Zip64;
import com.android.zipflinger.Zip64Eocd;
import com.android.zipflinger.Zip64Locator;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.nio.file.StandardOpenOption;
import java.util.HashMap;
import java.util.Map;

public class ZipMap {
    private final Map<String, Entry> entries = new HashMap<String, Entry>();
    private CentralDirectory cd = null;
    private final boolean accountDataDescriptors;
    private File file;
    private long fileSize;
    private Location payloadLocation;
    private Location cdLocation;
    private Location eocdLocation;

    private ZipMap(File file, boolean accountDataDescriptors) {
        this.file = file;
        this.accountDataDescriptors = accountDataDescriptors;
    }

    public static ZipMap from(File zipFile, boolean accountDataDescriptors) throws IOException {
        return ZipMap.from(zipFile, accountDataDescriptors, Zip64.Policy.ALLOW);
    }

    public static ZipMap from(File zipFile, boolean accountDataDescriptors, Zip64.Policy policy) throws IOException {
        ZipMap map = new ZipMap(zipFile, accountDataDescriptors);
        map.parse(policy);
        return map;
    }

    public Location getPayloadLocation() {
        return this.payloadLocation;
    }

    public Location getCdLoc() {
        return this.cdLocation;
    }

    public Location getEocdLoc() {
        return this.eocdLocation;
    }

    private void parse(Zip64.Policy policy) throws IOException {
        try (FileChannel channel = FileChannel.open(this.file.toPath(), StandardOpenOption.READ);){
            this.fileSize = channel.size();
            EndOfCentralDirectory eocd = EndOfCentralDirectory.find(channel);
            if (eocd.getLocation() == Location.INVALID) {
                throw new IllegalStateException(String.format("Could not find EOCD in '%s'", this.file));
            }
            this.eocdLocation = eocd.getLocation();
            this.cdLocation = eocd.getCdLocation();
            Zip64Locator locator = Zip64Locator.find(channel, eocd);
            if (locator.getLocation() != Location.INVALID) {
                if (policy == Zip64.Policy.FORBID) {
                    String message = String.format("Cannot parse forbidden zip64 archive %s", this.file);
                    throw new IllegalStateException(message);
                }
                Zip64Eocd zip64EOCD = Zip64Eocd.parse(channel, locator.getOffsetToEOCD64());
                this.cdLocation = zip64EOCD.getCdLocation();
                if (this.cdLocation == Location.INVALID) {
                    String message = String.format("Zip64Locator led to bad EOCD64 in %s", this.file);
                    throw new IllegalStateException(message);
                }
            }
            if (this.cdLocation == Location.INVALID) {
                throw new IllegalStateException(String.format("Could not find CD in '%s'", this.file));
            }
            this.parseCentralDirectory(channel, this.cdLocation, policy);
            this.payloadLocation = new Location(0L, this.cdLocation.first);
        }
    }

    private void parseCentralDirectory(FileChannel channel, Location location, Zip64.Policy policy) throws IOException {
        if (location.size() > Integer.MAX_VALUE) {
            throw new IllegalStateException("CD larger than 2GiB not supported");
        }
        int size = Math.toIntExact(location.size());
        ByteBuffer buf = ByteBuffer.allocate(size).order(ByteOrder.LITTLE_ENDIAN);
        channel.read(buf, location.first);
        buf.rewind();
        while (buf.remaining() >= 4 && buf.getInt() == 33639248) {
            Entry entry = new Entry();
            this.parseCentralDirectoryRecord(buf, channel, entry);
            if (!entry.getName().isEmpty()) {
                this.entries.put(entry.getName(), entry);
            }
            ZipMap.checkPolicy(entry, policy);
        }
        this.cd = new CentralDirectory(buf, this.entries);
        this.sanityCheck(location);
    }

    private static void checkPolicy(Entry entry, Zip64.Policy policy) {
        if (policy == Zip64.Policy.ALLOW) {
            return;
        }
        if (entry.getUncompressedSize() > 0xFFFFFFFFL || entry.getCompressedSize() > 0xFFFFFFFFL || entry.getLocation().first > 0xFFFFFFFFL) {
            String message = String.format("Entry %s infringes forbidden zip64 policy (size=%d, csize=%d, loc=%s)", entry.getName(), entry.getUncompressedSize(), entry.getCompressedSize(), entry.getLocation());
            throw new IllegalStateException(message);
        }
    }

    private void sanityCheck(Location cdLocation) {
        for (Entry e : this.entries.values()) {
            Location loc = e.getLocation();
            if (loc.first < 0L) {
                throw new IllegalStateException("Invalid first loc '" + e.getName() + "' " + loc);
            }
            if (loc.last >= this.fileSize) {
                throw new IllegalStateException(this.fileSize + "Invalid last loc '" + e.getName() + "' " + loc);
            }
            Location cdLoc = e.getCdLocation();
            if (cdLoc.first < 0L) {
                throw new IllegalStateException("Invalid first cdloc '" + e.getName() + "' " + cdLoc);
            }
            long cdSize = cdLocation.size();
            if (cdLoc.last < cdSize) continue;
            throw new IllegalStateException(cdSize + "Invalid last loc '" + e.getName() + "' " + cdLoc);
        }
    }

    public Map<String, Entry> getEntries() {
        return this.entries;
    }

    CentralDirectory getCentralDirectory() {
        return this.cd;
    }

    public void parseCentralDirectoryRecord(ByteBuffer buf, FileChannel channel, Entry entry) throws IOException {
        boolean hasDataDescriptor;
        long cdEntryStart = buf.position() - 4;
        buf.position(buf.position() + 4);
        short flags = buf.getShort();
        short compressionFlag = buf.getShort();
        entry.setCompressionFlag(compressionFlag);
        buf.position(buf.position() + 4);
        int crc = buf.getInt();
        entry.setCrc(crc);
        entry.setCompressedSize(Ints.uintToLong(buf.getInt()));
        entry.setUncompressedSize(Ints.uintToLong(buf.getInt()));
        int pathLength = Ints.ushortToInt(buf.getShort());
        int extraLength = Ints.ushortToInt(buf.getShort());
        int commentLength = Ints.ushortToInt(buf.getShort());
        buf.position(buf.position() + 8);
        entry.setLocation(new Location(Ints.uintToLong(buf.getInt()), 0L));
        ZipMap.parseName(buf, pathLength, entry);
        if (extraLength > 0) {
            int position = buf.position();
            int limit = buf.limit();
            buf.limit(position + extraLength);
            ZipMap.parseExtra(buf.slice(), entry);
            buf.limit(limit);
            buf.position(position + extraLength);
        }
        buf.position(buf.position() + commentLength);
        ByteBuffer localFieldBuffer = this.readLocalFields(entry.getLocation().first + 26L, entry, channel);
        int localPathLength = Ints.ushortToInt(localFieldBuffer.getShort());
        int localExtraLength = Ints.ushortToInt(localFieldBuffer.getShort());
        if (pathLength != localPathLength) {
            String message = String.format("Entry '%s' name differ (%d vs %d)", entry.getName(), localPathLength, pathLength);
            throw new IllegalStateException(message);
        }
        boolean isCompressed = compressionFlag != 0;
        long payloadSize = isCompressed ? entry.getCompressedSize() : entry.getUncompressedSize();
        long start = entry.getLocation().first;
        long end = start + 30L + (long)pathLength + (long)localExtraLength + payloadSize;
        entry.setLocation(new Location(start, end - start));
        Location payloadLocation = new Location(start + 30L + (long)pathLength + (long)localExtraLength, payloadSize);
        entry.setPayloadLocation(payloadLocation);
        long cdEntrySize = 46 + pathLength + extraLength + commentLength;
        entry.setCdLocation(new Location(cdEntryStart, cdEntrySize));
        boolean bl = hasDataDescriptor = (flags & 8) == 8;
        if (hasDataDescriptor) {
            if (this.accountDataDescriptors) {
                channel.position(end);
                ZipMap.parseDataDescriptor(channel, entry);
            } else {
                entry.setLocation(Location.INVALID);
            }
        }
    }

    private static void parseExtra(ByteBuffer buf, Entry entry) {
        buf.order(ByteOrder.LITTLE_ENDIAN);
        while (buf.remaining() >= 4) {
            short id = buf.getShort();
            int size = Ints.ushortToInt(buf.getShort());
            if (id == 1) {
                ZipMap.parseZip64Extra(buf, entry);
            }
            if (buf.remaining() < size) continue;
            buf.position(buf.position() + size);
        }
    }

    private static void parseZip64Extra(ByteBuffer buf, Entry entry) {
        if (entry.getUncompressedSize() == 0xFFFFFFFFL) {
            if (buf.remaining() < 8) {
                throw new IllegalStateException("Bad zip64 extra for entry " + entry.getName());
            }
            entry.setUncompressedSize(Ints.ulongToLong(buf.getLong()));
        }
        if (entry.getCompressedSize() == 0xFFFFFFFFL) {
            if (buf.remaining() < 8) {
                throw new IllegalStateException("Bad zip64 extra for entry " + entry.getName());
            }
            entry.setCompressedSize(Ints.ulongToLong(buf.getLong()));
        }
        if (entry.getLocation().first == 0xFFFFFFFFL) {
            if (buf.remaining() < 8) {
                throw new IllegalStateException("Bad zip64 extra for entry " + entry.getName());
            }
            long offset = Ints.ulongToLong(buf.getLong());
            entry.setLocation(new Location(offset, 0L));
        }
    }

    private ByteBuffer readLocalFields(long offset, Entry entry, FileChannel channel) throws IOException {
        ByteBuffer localFieldsBuffer = ByteBuffer.allocate(4).order(ByteOrder.LITTLE_ENDIAN);
        if (offset < 0L || offset + 4L > this.fileSize) {
            throw new IllegalStateException("Entry :" + entry.getName() + " invalid offset (" + offset + ")");
        }
        channel.read(localFieldsBuffer, offset);
        localFieldsBuffer.rewind();
        return localFieldsBuffer;
    }

    private static void parseName(ByteBuffer buf, int length, Entry entry) {
        byte[] pathBytes = new byte[length];
        buf.get(pathBytes);
        entry.setNameBytes(pathBytes);
    }

    private static void parseDataDescriptor(FileChannel channel, Entry entry) throws IOException {
        ByteBuffer dataDescriptorBuffer = ByteBuffer.allocate(4).order(ByteOrder.LITTLE_ENDIAN);
        channel.read(dataDescriptorBuffer);
        dataDescriptorBuffer.rewind();
        int dataDescriptorLength = 12;
        if (dataDescriptorBuffer.getInt() == 134695760) {
            dataDescriptorLength += 4;
        }
        Location adjustedLocation = new Location(entry.getLocation().first, entry.getLocation().size() + (long)dataDescriptorLength);
        entry.setLocation(adjustedLocation);
    }

    public File getFile() {
        return this.file;
    }
}

