/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.jira.util.stats;

import com.atlassian.jira.config.properties.JiraSystemProperties;
import com.atlassian.jira.util.stats.GsonStatsSerializer;
import com.atlassian.jira.util.stats.JiraStatsListener;
import com.atlassian.jira.util.stats.ManagedStats;
import com.atlassian.jira.util.stats.MustNotPrintStats;
import com.atlassian.jira.util.stats.SystemTimeProvider;
import com.atlassian.jira.util.stats.TimeProvider;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.reflect.AbstractInvocationHandler;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.Closeable;
import java.io.IOException;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.commons.collections.list.SynchronizedList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class JiraStats<T extends ManagedStats>
extends AbstractInvocationHandler
implements InvocationHandler,
Closeable {
    private static final Logger log = LoggerFactory.getLogger(JiraStats.class);
    public static final String COMMON_PREFIX = "[JIRA-STATS] ";
    public static final String JIRA_STATS_LOGGING_INTERVAL = "com.atlassian.jira.stats.logging.interval";
    static final Interval DEFAULT = Interval.INTERVAL5MIN;
    public static final String STATS_TYPE_SNAPSHOT = "snapshot";
    public static final String STATS_TYPE_TOTAL = "total";
    private static final Method CLOSE_METHOD = JiraStats.getMethod(ManagedStats.class, "close", new Class[0]);
    private static final Method ADD_LISTENER_METHOD = JiraStats.getMethod(ManagedStats.class, "addJiraStatsListener", JiraStatsListener.class);
    private static final Method REMOVE_LISTENER_METHOD = JiraStats.getMethod(ManagedStats.class, "removeJiraStatsListener", JiraStatsListener.class);
    private final TimeProvider timeProvider;
    private final Optional<ScheduledExecutorService> executorService;
    private final GsonStatsSerializer<T> statsSerializer = new GsonStatsSerializer();
    private final AtomicReference<T> snapshot = new AtomicReference();
    private final T total;
    private final List<JiraStatsListener> onPrintListeners = SynchronizedList.decorate(new ArrayList());
    private final String statsName;
    private final long statsPeriodMillis;
    private final Map<Method, Boolean> allowedToPrintStatsInThread;
    private final AtomicLong totalStatsOverheadNanos = new AtomicLong(0L);
    private final AtomicLong snapshotStatsOverheadNanos = new AtomicLong(0L);
    private final AtomicLong totalStatsInvocations = new AtomicLong(0L);
    private final AtomicLong snapshotStatsInvocations = new AtomicLong(0L);
    private final long totalStartTime;
    private AtomicLong snapshotStartTime;
    private final Class<T> interfaceClass;
    private final Supplier<T> objectFactory;

    static Interval fromIntervalName(String intervalName) {
        if (intervalName == null) {
            return DEFAULT;
        }
        try {
            return Interval.valueOf(intervalName.toUpperCase(Locale.ENGLISH));
        }
        catch (IllegalArgumentException e) {
            String allowedValues = Arrays.stream(Interval.values()).map(interval -> interval.name().toLowerCase(Locale.ENGLISH)).collect(Collectors.joining(", "));
            log.error("Unknown stats period: {}. Following values are allowed: {}. Using default: {}.", new Object[]{intervalName, allowedValues, DEFAULT.name().toLowerCase(Locale.ENGLISH)});
            return DEFAULT;
        }
    }

    public static long statsLoggingInterval(TimeUnit timeUnit) {
        String intervalName = JiraSystemProperties.getInstance().getProperty(JIRA_STATS_LOGGING_INTERVAL, DEFAULT.name().toLowerCase(Locale.ENGLISH));
        return timeUnit.convert(JiraStats.fromIntervalName((String)intervalName).periodInMin, TimeUnit.MINUTES);
    }

    public static <M extends ManagedStats> M create(Class<M> interfaceClass, Supplier<M> objectFactory, boolean printOnSeparateThread) {
        return JiraStats.create(interfaceClass, objectFactory, printOnSeparateThread ? JiraStats.getNewSingleThreadScheduledExecutor() : null, new SystemTimeProvider());
    }

    @VisibleForTesting
    public static <M extends ManagedStats> M create(Class<M> interfaceClass, Supplier<M> objectFactory, @Nullable ScheduledExecutorService executorService, TimeProvider timeProvider) {
        Preconditions.checkArgument((interfaceClass != null ? 1 : 0) != 0, (Object)"Stats interface must be set");
        Preconditions.checkArgument((objectFactory != null ? 1 : 0) != 0, (Object)"Stats objects factory must be set");
        Preconditions.checkArgument((timeProvider != null ? 1 : 0) != 0, (Object)"TimeProvider must be set");
        JiraStats<M> statsManager = new JiraStats<M>(interfaceClass, objectFactory, executorService, timeProvider);
        return (M)((ManagedStats)Proxy.newProxyInstance(objectFactory.getClass().getClassLoader(), new Class[]{interfaceClass}, statsManager));
    }

    private JiraStats(Class<T> interfaceClass, Supplier<T> objectFactory, @Nullable ScheduledExecutorService executorService, @VisibleForTesting TimeProvider timeProvider) {
        this.interfaceClass = (Class)Preconditions.checkNotNull(interfaceClass);
        this.objectFactory = (Supplier)Preconditions.checkNotNull(objectFactory);
        this.timeProvider = (TimeProvider)Preconditions.checkNotNull((Object)timeProvider);
        this.allowedToPrintStatsInThread = this.buildAllowedToPrintStatsInThreadMap(interfaceClass);
        this.totalStartTime = this.currentTimeMillis();
        this.snapshotStartTime = new AtomicLong(this.currentTimeMillis());
        this.total = (ManagedStats)objectFactory.get();
        this.snapshot.set(this.objectFactory.get());
        this.statsName = this.total.getStatsName();
        long defaultStatsPeriodMillis = JiraStats.statsLoggingInterval(TimeUnit.MILLISECONDS);
        long minStatsPeriodMillis = TimeUnit.MINUTES.toMillis(this.total.getMinInterval().periodInMin);
        if (minStatsPeriodMillis > defaultStatsPeriodMillis) {
            log.info("{}[{}] Custom stats period for: {} set to: {} millis", new Object[]{COMMON_PREFIX, this.statsName, this.statsName, minStatsPeriodMillis});
            this.statsPeriodMillis = minStatsPeriodMillis;
        } else {
            this.statsPeriodMillis = defaultStatsPeriodMillis;
        }
        Preconditions.checkArgument((boolean)this.statsName.matches("[A-Za-z][A-Za-z0-9_-]*"), (String)"Stats name '%s' does not match the regex '%s'", (Object)this.statsName, (Object)"[A-Za-z][A-Za-z0-9_-]*");
        this.executorService = Optional.ofNullable(executorService);
        this.executorService.ifPresent(executor -> executor.scheduleAtFixedRate(this::printAndReset, TimeUnit.MINUTES.toMillis(5L), this.statsPeriodMillis, TimeUnit.MILLISECONDS));
        log.info("{}[{}] stats created: loggingIntervalInMin={}, printInThread={}", new Object[]{COMMON_PREFIX, this.statsName, TimeUnit.MILLISECONDS.toMinutes(this.statsPeriodMillis), !this.executorService.isPresent()});
    }

    private Map<Method, Boolean> buildAllowedToPrintStatsInThreadMap(Class<T> interfaceClass) {
        return (Map)Arrays.stream(interfaceClass.getMethods()).collect(ImmutableMap.toImmutableMap(Function.identity(), method -> method.getAnnotation(MustNotPrintStats.class) == null));
    }

    private static ScheduledExecutorService getNewSingleThreadScheduledExecutor() {
        return Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setDaemon(true).setNameFormat("jira-stats-%d").build());
    }

    @Override
    public void close() throws IOException {
        log.debug("{}[{}] Closing {}", new Object[]{COMMON_PREFIX, this.statsName, this});
        if (this.executorService.isPresent()) {
            this.executorService.get().shutdown();
            try {
                this.executorService.get().awaitTermination(5L, TimeUnit.SECONDS);
            }
            catch (InterruptedException e) {
                log.error("{}[{}] Interrupted exception while closing {}", new Object[]{COMMON_PREFIX, this.statsName, this});
                Thread.currentThread().interrupt();
                throw new IOException(e);
            }
        }
        this.printAndReset();
        this.onPrintListeners.clear();
        this.total.close();
        log.debug("{}[{}] Closed {}", new Object[]{COMMON_PREFIX, this.statsName, this});
    }

    protected Object handleInvocation(Object proxyThatHasNoInterestingMethods, Method method, Object[] args) throws Throwable {
        long startTime = this.currentTimeNanos();
        Preconditions.checkArgument((boolean)this.interfaceClass.isAssignableFrom(proxyThatHasNoInterestingMethods.getClass()));
        Preconditions.checkArgument((boolean)this.allowedToPrintStatsInThread.containsKey(method));
        if (log.isTraceEnabled()) {
            log.trace("{}[{}] handleInvocation: proxy={}, method={}, args={}", new Object[]{COMMON_PREFIX, this.statsName, proxyThatHasNoInterestingMethods, method, args});
        }
        if (CLOSE_METHOD.equals(method)) {
            this.close();
            return null;
        }
        if (ADD_LISTENER_METHOD.equals(method)) {
            Preconditions.checkNotNull((Object)args[0]);
            this.onPrintListeners.add((JiraStatsListener)args[0]);
            return null;
        }
        if (REMOVE_LISTENER_METHOD.equals(method)) {
            Preconditions.checkNotNull((Object)args[0]);
            this.onPrintListeners.remove(args[0]);
            return null;
        }
        try {
            method.invoke(this.snapshot.get(), args);
            method.invoke(this.total, args);
        }
        catch (Throwable t) {
            log.error("Exception caught while calling {}", (Object)method, (Object)t);
        }
        this.snapshotStatsInvocations.incrementAndGet();
        this.totalStatsInvocations.incrementAndGet();
        if (!this.executorService.isPresent() && this.isAllowedToPrintStatsInThread(method)) {
            this.printNotTooOften();
        }
        long elapsedNanos = this.currentTimeNanos() - startTime;
        this.snapshotStatsOverheadNanos.addAndGet(elapsedNanos);
        this.totalStatsOverheadNanos.addAndGet(elapsedNanos);
        return null;
    }

    public String toString() {
        return "JiraStats{statsName=" + this.statsName + ", interfaceClass=" + this.interfaceClass.getName() + ", objectClass=" + ((ManagedStats)this.objectFactory.get()).getClass().getName() + '}';
    }

    private boolean isAllowedToPrintStatsInThread(Method method) {
        return this.allowedToPrintStatsInThread.get(method);
    }

    private boolean canPrintStats() {
        long millisSinceSnapshotStartTime = this.currentTimeMillis() - this.snapshotStartTime.get();
        return millisSinceSnapshotStartTime >= this.statsPeriodMillis;
    }

    private long currentTimeMillis() {
        return this.timeProvider.currentTimeMillis();
    }

    private long currentTimeNanos() {
        return this.timeProvider.nanoTime();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void printNotTooOften() {
        if (this.canPrintStats()) {
            JiraStats jiraStats = this;
            synchronized (jiraStats) {
                if (this.canPrintStats()) {
                    this.printAndReset();
                }
            }
        }
    }

    synchronized void printAndReset() {
        try {
            long currentTimeMillis = this.currentTimeMillis();
            String snapshotSerialized = this.log((ManagedStats)this.snapshot.getAndSet(this.objectFactory.get()), currentTimeMillis, this.snapshotStatsOverheadNanos.getAndSet(0L), this.snapshotStatsInvocations.getAndSet(0L), this.durationSince(this.snapshotStartTime.getAndSet(currentTimeMillis)), STATS_TYPE_SNAPSHOT);
            String totalSerialized = this.log(this.total, currentTimeMillis, this.totalStatsOverheadNanos.get(), this.totalStatsInvocations.get(), this.durationSince(this.totalStartTime), STATS_TYPE_TOTAL);
            if (!this.onPrintListeners.isEmpty()) {
                Map snapshotMap = this.statsSerializer.deserializeToMap(snapshotSerialized);
                Map totalMap = this.statsSerializer.deserializeToMap(totalSerialized);
                this.onPrintListeners.forEach(l -> this.notifyListenerSafely((JiraStatsListener)l, snapshotMap, totalMap));
            }
        }
        catch (Throwable t) {
            log.error("{}[{}] Failed printing stats", new Object[]{COMMON_PREFIX, this.statsName, t});
        }
    }

    private void notifyListenerSafely(JiraStatsListener listener, Map snapshotMap, Map totalMap) {
        try {
            long startTime = this.currentTimeMillis();
            listener.onStats(snapshotMap, totalMap);
            long elapsedMillis = this.currentTimeMillis() - startTime;
            log.trace("{}[{}] Listener {} notified in {}ms", new Object[]{COMMON_PREFIX, this.statsName, listener.getClass(), elapsedMillis});
        }
        catch (Throwable t) {
            log.error("{}[{}] Exception thrown by stats listener {} caught. Ignoring...", new Object[]{COMMON_PREFIX, this.statsName, listener.getClass(), t});
        }
    }

    private String log(T stats, long currentTimeMillis, long statsOverheadNanos, long statsInvocations, Duration statsDuration, String statsType) {
        String statsOverheadPercentage = JiraStats.getStatsOverheadPercentage(statsOverheadNanos, stats.getTotalMeasuredOperationsForStatsOverheadInMillis());
        ImmutableMap metadata = ImmutableMap.builder().put((Object)"_statsName", (Object)this.statsName).put((Object)"_statsType", (Object)statsType).put((Object)"_time", (Object)Instant.ofEpochMilli(currentTimeMillis).toString()).put((Object)"_timestamp", (Object)currentTimeMillis).put((Object)"_duration", (Object)statsDuration.toString()).put((Object)"_invocations", (Object)statsInvocations).put((Object)"_statsOverhead", (Object)statsOverheadPercentage).build();
        String serialized = this.statsSerializer.serialize((Map<String, Object>)metadata, stats);
        if (log.isInfoEnabled()) {
            log.info("{}[{}] {} stats: duration={}, statsOverhead={}, data={}", new Object[]{COMMON_PREFIX, this.statsName, statsType, statsDuration, statsOverheadPercentage, serialized});
        }
        return serialized;
    }

    private Duration durationSince(long startTime) {
        return Duration.ofMillis(this.currentTimeMillis() - startTime);
    }

    private static String getStatsOverheadPercentage(long statsOverheadNanos, Optional<Long> totalMeasuredOperationsForStatsOverheadMillis) {
        return totalMeasuredOperationsForStatsOverheadMillis.filter(overheadMillis -> overheadMillis > 0L).map(overheadMillis -> String.format("%.4f%% of %dms", 100.0 * (double)statsOverheadNanos / (double)TimeUnit.MILLISECONDS.toNanos((long)overheadMillis), overheadMillis)).orElse("n/a");
    }

    private static Method getMethod(Class<?> clazz, String methodName, Class<?> ... argTypes) {
        try {
            return clazz.getMethod(methodName, argTypes);
        }
        catch (NoSuchMethodException e) {
            throw new IllegalArgumentException(e);
        }
    }

    public static enum Interval {
        INTERVAL5MIN(5),
        INTERVAL15MIN(15),
        INTERVAL30MIN(30),
        INTERVAL60MIN(60);

        final int periodInMin;

        private Interval(int periodInMin) {
            this.periodInMin = periodInMin;
        }
    }
}

