define("jira-dashboard-items/sprint-health", [
    'jquery',
    'underscore',
    "jira-dashboard-items/sprint-health-view"
], function($, _, SprintHealthView) {

    var AUTO_SPRINT = "auto";

    /**
     *
     * @param {InlineGadgetAPI} API
     * @param {Object} [options] for gadget
     * @constructor
     */
    var DashboardItem = function(API, options) {
        this.API = API;
        this.options = options || {};
        this.sprintHealthView = new SprintHealthView(this.API.getGadgetId());
        this.initialiseEventHandlers();

    };

    /**
     * Render the configured sprint-health gadget.
     * @param {node} element to apply gadget to
     *
     * @param {Object} preferences for gadget that have been configured.
     */
    DashboardItem.prototype.render = function(element, preferences) {

        function getSprintIdFromData(sprintIdData){
            var sprint = _.findWhere(sprintIdData.sprints, { state: "ACTIVE" });
            return sprint ? sprint.id : null;
        }

        preferences = preferences || {};

        var dashboardItem = this;

        dashboardItem.sprintHealthView.setElement(element);

        var rapidViewId = preferences.rapidViewId,
            sprintId = preferences.sprintId,
            isConfigured = preferences.isConfigured;

        //  If the gadget does not have any active sprints and is therefore not configured,
        //  show an error/warning message.
        if(!isConfigured || !rapidViewId){
            dashboardItem.sprintHealthView.showSprintNotActiveErrorMessage();
            return;
        }

        dashboardItem.API.showLoadingBar();

        // Init auto-refresh
        dashboardItem.API.initRefresh(preferences, dashboardItem.render.bind(dashboardItem, element, preferences));

        //  If the sprintId is set to 'auto' (in the initialmost phase), then request all sprints associated
        //  with the given rapidViewId, otherwise return the given sprintId in an appropriate format.
        requestSprintData(sprintId, rapidViewId).done(function(sprintIdData){

            sprintId = getSprintIdFromData(sprintIdData);
            if(!sprintId){
                dashboardItem.sprintHealthView.showSprintNotActiveErrorMessage();
                dashboardItem.API.hideLoadingBar();
                return;
            }

            //  Request for data pertaining to the current rapidView (board) selected, and the current
            //  sprintId selected.
            $.when(requestSprintHealthData(sprintId, rapidViewId), requestRapidViewData(rapidViewId)).done(function(sprintHealthData, rapidViewData){

                if(sprintHealthData.length > 0 && rapidViewData.length > 0){
                    sprintHealthData = sprintHealthData[0];
                    rapidViewData = rapidViewData[0].currentViewConfig;
                } else {
                    dashboardItem.sprintHealthView.showRapidViewCannotBeFoundErrorMessage();
                    return;
                }

                //  After both AJAX requests have successfully completed, begin rendering the gadget
                //  (capture failure/errors below the .done sequence).
                dashboardItem._renderDashboardItem(element, preferences, sprintHealthData, rapidViewData);
                dashboardItem.API.hideLoadingBar();

            }).fail(function(response) {
                if (response.status === 403) {
                    dashboardItem.sprintHealthView.showGadgetNotAllowedToRenderErrorMessage();
                }
                else {
                    dashboardItem.sprintHealthView.showRapidViewCannotBeFoundErrorMessage();
                }
                dashboardItem.API.hideLoadingBar();
            });

        }).fail(function(response){
            if (response.status === 403) {
                dashboardItem.sprintHealthView.showGadgetNotAllowedToRenderErrorMessage();
            }
            else {
                dashboardItem.sprintHealthView.showRapidViewCannotBeFoundErrorMessage();
            }
            dashboardItem.API.hideLoadingBar();

        });
    };

    /**
     * Use the data acquired from AJAX requests to render the page
     * @param {Node} element to render to
     * @param {Object} preferences for dashboard item
     * @param {Object} sprintHealthData
     * @param {Object} rapidViewData
     */
    DashboardItem.prototype._renderDashboardItem = function(element, preferences, sprintHealthData, rapidViewData){

        function getProgressColumnValue(columnNumber){
            return sprintHealthData.progress.columns[columnNumber].value;
        }

        function getPercentageValue(value, total){
            return (100 * value) / total;
        }

        function getSprintMetricValue(key){
            return _.findWhere(sprintHealthData.sprintMetrics, { key: key}).value;
        }

        function createStatusList(statusArray) {
            var quoteChar = "%22";
            var statuses = _.map(statusArray, function(status) {
                return quoteChar + status + quoteChar;
            });
            return statuses.join(",");
        }

        var dashboardItem = this;

        var issueMetrics = {};
        _.each(sprintHealthData.issueMetrics, function(item) { issueMetrics[item.key] = item.value; });

        var progress = {total: 0};
        _.each(sprintHealthData.progress.columns, function(item) { progress.total += item.value; });


        var value = getProgressColumnValue(0);
        progress.todoWidth = getPercentageValue(value, progress.total);
        progress.todo = value;
        progress.todoMetric = value; //store the original for inline-dialog template
        progress.todoPercentage = Math.round(progress.todoWidth);

        value = getProgressColumnValue(1);
        progress.inprogressWidth = getPercentageValue(value, progress.total);
        progress.inprogress = value;
        progress.inprogressMetric = value;
        progress.inprogressPercentage = Math.round(progress.inprogressWidth);

        value = getProgressColumnValue(2);
        progress.doneWidth = getPercentageValue(value, progress.total);
        progress.done = value;
        progress.doneMetric = value;
        progress.donePercentage = Math.round(progress.doneWidth);

        if(sprintHealthData.timeBasedEstimation){
            progress.todo = GH.TimeFormat.formatShortDurationForTimeTrackingConfiguration(progress.todo);
            progress.inprogress = GH.TimeFormat.formatShortDurationForTimeTrackingConfiguration(progress.inprogress);
            progress.done = GH.TimeFormat.formatShortDurationForTimeTrackingConfiguration(progress.done);
        }

        var durationComplete = getSprintMetricValue("duration_completion_percentage");
        var workComplete = getSprintMetricValue("work_completion_percentage");
        var scopeChange = getSprintMetricValue("scope_change_percentage");
        scopeChange = parseInt(scopeChange, 10);

        //  Render the gadget, passing in all params required by the Soy file.
        var renderParams = {
            showBoardName:      preferences.showRapidViewName,
            boardName:          rapidViewData.name,
            boardId:            rapidViewData.id,
            boardStatistic:     rapidViewData.estimationStatistic.name,
            showSprintName:     preferences.showSprintName,
            issueMetrics:       issueMetrics,
            sprintName:         sprintHealthData.name,
            sprintId:           sprintHealthData.id,
            highestPriority:    sprintHealthData.highestPriority,
            assignees:          sprintHealthData.assignees,
            showAssignees:      preferences.showAssignees,
            progress:           progress,
            timeremaining:      sprintHealthData.timeRemaining,
            durationComplete:   durationComplete,
            workComplete:       workComplete,
            scopeChange:        scopeChange,
            timeTrackingEnable: sprintHealthData.timeTrackingEnabled,
            noEstimateValues:   sprintHealthData.noEstimateValues,
            flaggedField:       sprintHealthData.flaggedField,
            todoStatuses:       createStatusList(sprintHealthData.statusMap.notStarted),
            inProgressStatuses: createStatusList(sprintHealthData.statusMap.inProgress),
            doneStatuses:       createStatusList(sprintHealthData.statusMap.done)
        };

        dashboardItem.sprintHealthView.renderDashboardItem(element, renderParams);

    };


    /**
     * Render the configuration screen for the sprint-health gadget
     * @param {Node} element to render into
     * @param {Object} preferences for object.
     */
    DashboardItem.prototype.renderEdit = function (element, preferences) {

        function getRapidViewData(data){
            data = _.where(data.views, {sprintSupportEnabled: true});
            return transformData(data);
        }

        function transformData(data){
            return _.map(data, function(obj) {
                return {
                    text: obj.name,
                    value: obj.id
                };
            });
        }

        preferences = preferences || {};

        var dashboardItem = this;

        dashboardItem.sprintHealthView.setElement(element);

        dashboardItem.API.showLoadingBar();

        //  Fire AJAX request for all current rapidViews (boards)
        requestAllBoards().done(function(data) {

            //  If no boards are currently available, show the appropriate error message
            if(data.views.length === 0){
                dashboardItem.sprintHealthView.showNoBoardsAvailableErrorMessage();
                dashboardItem.API.hideLoadingBar();
                return;
            }

            // Prefix to avoid collisions with other instances of this gadget.
            var prefix          = dashboardItem.API.getGadgetId() + "-";
            var boardTypes      = getRapidViewData(data);
            var activeSprints   = {};

            var configViewParams = {
                prefix:         prefix,
                boardTypes:     boardTypes,
                activeSprints:  activeSprints,
                checkboxes:     [
                    {
                        'id': 'showRapidViewName',
                        'labelText' : AJS.I18n.getText('gh.gadget.userpref.showrvname.name'),
                        'isChecked' : preferences.showRapidViewName
                    },
                    {
                        'id': 'showSprintName',
                        'labelText' : AJS.I18n.getText('gh.gadget.userpref.showspname.name'),
                        'isChecked' : preferences.showSprintName
                    },
                    {
                        'id': 'showAssignees',
                        'labelText' : AJS.I18n.getText('gh.gadget.userpref.showassignees'),
                        'isChecked' : preferences.showAssignees
                    }

                ],
                preferences:    preferences
            };

            //  Render the configuration view, passing in all required params
            dashboardItem.sprintHealthView.renderConfigView(element, configViewParams);

        }).fail(function() {
            dashboardItem.sprintHealthView.showGadgetNotAllowedToRenderErrorMessage();
        }).always(function() {
            dashboardItem.API.hideLoadingBar();
        });

    };

    DashboardItem.prototype.initialiseEventHandlers = function(){

        var sprintHealth = this;
        var sprintHealthView = this.sprintHealthView;

        AJS.$(sprintHealthView).on('boardChange', function(e, rapidViewId) {
            sprintHealth.updateSprintData(rapidViewId);
        });

        AJS.$(sprintHealthView).on('cancel', function() {
            sprintHealth.API.closeEdit();
        });

        AJS.$(sprintHealthView).on('submit', function(e, preferences) {
            sprintHealth.API.savePreferences(preferences);
        });

        AJS.$(sprintHealthView).on('refreshLayout', function() {
            sprintHealth.API.forceLayoutRefresh();
        });

        AJS.$(sprintHealthView).on('hideLoadingBar', function() {
            sprintHealth.API.hideLoadingBar();
        });

    };

    DashboardItem.prototype.updateSprintData = function(rapidViewId){
        var self = this;

        requestSprintData(false, rapidViewId).done(function(data){
            data = _.where(data.sprints, { state: "ACTIVE" });
            self.sprintHealthView.renderCurrentSprints(data);
        });
    };

    function requestSprintData(sprintId, rapidViewId) {
        if(sprintId === AUTO_SPRINT || sprintId === false){
            return $.ajax({
                method: "GET",
                url: AJS.contextPath() + "/rest/greenhopper/1.0/sprintquery/" + rapidViewId
            });
        } else {
            var deferred = $.Deferred();
            deferred.resolve({
                sprints : [
                    {
                        id : sprintId,
                        state: "ACTIVE"
                    }
                ]
            });
            return deferred;
        }
    }

    function requestSprintHealthData(sprintId, rapidViewId){
        return $.ajax({
            method: "GET",
            url: AJS.contextPath() + "/rest/greenhopper/1.0/gadgets/sprints/health",
            data: {
                rapidViewId : rapidViewId,
                sprintId : sprintId
            }
        });
    }

    function requestRapidViewData(rapidViewId){
        return $.ajax({
            method: "GET",
            url: AJS.contextPath() + "/rest/greenhopper/1.0/xboard/config?rapidViewId=" + rapidViewId
        });
    }

    function requestAllBoards(){
        return $.ajax({
            method: "GET",
            url: AJS.contextPath() + "/rest/greenhopper/1.0/rapidview"
        });
    }

    return DashboardItem;
});