/**
 * A single-select list for selecting Sprints.
 *
 * @constructor JIRA.SprintPicker
 * @extends AJS.SingleSelect
 */
JIRA.SprintPicker = AJS.SingleSelect.extend({
    init: function (options) {

        var StateKeys = {
            ACTIVE: "ACTIVE",
            FUTURE: "FUTURE",
            CLOSED: "CLOSED"
        };
        var MAX_RESULTS_TO_DISPLAY = 10;
        var element = options.element;

        // references to default AJS.SingleSelect renderers
        var oldSuggestionRenderer,
            oldErrorMessageRenderer;

        // the current state and id of the issue if in edit context
        var savedState = element.data('saved-state'),
            savedId = element.data('saved-id');

        // a reference to 'this' instance and the scope change warning
        var $scopeChangeWarning,
            self = this;

        // An override for the suggestion handler to add the "Start typing..." description down the bottom
        var SprintPickerSuggestionHandler = AJS.SelectSuggestHandler.extend({
            formatSuggestions: function (descriptors, query) {
                descriptors = this._super(descriptors, query);
                if (query.length === 0) {
                    descriptors[descriptors.length - 1].footerText(AJS.I18n.getText("gh.sprint.customfield.dropdown.description"));
                }
                return descriptors;
            }
        });

        var wrmContextPath = require('wrm/context-path');

        // Configure the options
        AJS.$.extend(options, {
            submitInputVal: false,
            showDropdownButton: true,
            removeOnUnSelect: false,
            maxInlineResultsDisplayed: MAX_RESULTS_TO_DISPLAY,
            ajaxOptions: {
                url: wrmContextPath() + "/rest/greenhopper/1.0/sprint/picker",
                query: true,
                formatResponse: formatResponse,
                error: handleAjaxError
            }
        });

        initSprintPicker();

        /**
         * Appends the board name to the drop down items but does not affect the search. It will not append anything
         * if there is no board name.
         *
         * @param descriptor the current suggestion to render
         * @returns {jQuery}
         */
        function suggestionRenderer(descriptor) {
            var elem = oldSuggestionRenderer.apply(this, arguments);

            elem.find('a').append(GH.tpl.customfields.sprintpicker.renderSuggestionMeta(descriptor.properties));

            return elem;
        }

        /**
         * Validates the current input/selection and renders any error or warning messages.
         *
         * @param value the current typed value
         */
        function errorMessageRenderer(value) {
            var selectionDescriptor = this.getSelectedDescriptor(),
                errorMessage;

            // if the value is empty and there is no selection, consider it a change event
            if (!value && !selectionDescriptor) {
                registerSprintPickerChange(null);
            }

            hideScopeChangeWarning();
            this.hideErrorMessage();

            if (!selectionDescriptor) {
                // validate free input
                errorMessage = validateFreeInput(value);
            } else {
                // validate a selection
                errorMessage = validateSelection(selectionDescriptor);
            }

            if (errorMessage.message) {
                this.options.errorMessage = errorMessage.message;
                oldErrorMessageRenderer.apply(this, arguments);
                this.$errorMessage.addClass('inline-edit-error');
            }
            if (errorMessage.needsScopeChangeWarning) {
                renderScopeChangeWarning();
            }
        }

        /**
         * Validates free input and describes the resulting behaviour to the user.
         *
         * @param value
         * @returns {String}
         */
        function validateFreeInput(value) {
            var errorMessage = null,
                needsScopeChangeWarning = false;

            if (value && value.length > 0) {
                if (savedState) {
                    errorMessage = AJS.I18n.getText("gh.sprint.customfield.dropdown.invalid.sprint.backlog", value);
                    if (savedState === StateKeys.ACTIVE) {
                        needsScopeChangeWarning = true;
                    }
                } else {
                    errorMessage = AJS.I18n.getText("gh.sprint.customfield.dropdown.invalid.sprint", value);
                }
            } else {
                if (savedState === StateKeys.ACTIVE) {
                    needsScopeChangeWarning = true;
                }
            }

            return {
                message: errorMessage,
                needsScopeChangeWarning: needsScopeChangeWarning
            };
        }

        /**
         * Validates a selection and retrieves any appropriate error messages.
         * It will also render a warning if scope will change.
         *
         * @param selectionDescriptor
         */
        function validateSelection(selectionDescriptor) {
            var needsScopeChangeWarning = false;
            if (savedState === StateKeys.ACTIVE) {
                if (useEmbeddedScopeChangeWarningStyle()) {
                    needsScopeChangeWarning = true;
                } else {
                    needsScopeChangeWarning = (savedId != selectionDescriptor.properties.value);
                }
            } else if (savedState === StateKeys.FUTURE) {
                if (selectionDescriptor.properties.stateKey === StateKeys.ACTIVE) {
                    needsScopeChangeWarning = true;
                }
            } else if (!savedState) {
                if (selectionDescriptor.properties.stateKey === StateKeys.ACTIVE) {
                    needsScopeChangeWarning = true;
                }
            }

            return {
                message: null,
                needsScopeChangeWarning: needsScopeChangeWarning
            };
        }

        /**
         * Render the yellow box indicating to the user that scope is going to change as a result of this
         * operation.
         */
        function renderScopeChangeWarning() {
            if (!$scopeChangeWarning) {
                if (!useEmbeddedScopeChangeWarningStyle()) {
                    $scopeChangeWarning = AJS.$(GH.tpl.rapid.notification.renderAuiMessage({
                        type: 'warning',
                        className: 'ghx-sprint-picker-scope-warning',
                        icon: true,
                        title: AJS.I18n.getText('gh.sprint.issue.move.dialog.warning.scopechange')
                    }));
                }

                var inlineEditContainer = self.$container.closest('.inline-edit-fields');
                // we need to append to different containers depending on our context within either view issue or the
                // quick edit/create forms.
                if (inlineEditContainer.size() > 0) {
                    // view issue page
                    if (useEmbeddedScopeChangeWarningStyle()) {
                        $scopeChangeWarning = AJS.$('<div class="ghx-estimate-scope-warning"></div>');
                        $scopeChangeWarning.text(AJS.I18n.getText('gh.sprint.issue.move.dialog.warning.scopechange'));
                        inlineEditContainer.find('.field-group').append($scopeChangeWarning);
                    } else {
                        inlineEditContainer.closest('.editable-field').append($scopeChangeWarning);
                    }
                } else {
                    // quick edit/create form
                    if (useEmbeddedScopeChangeWarningStyle()) {
                        $scopeChangeWarning = AJS.$(GH.tpl.rapid.notification.renderAuiMessage({
                            type: 'warning',
                            className: 'ghx-sprint-picker-scope-warning',
                            icon: true,
                            title: AJS.I18n.getText('gh.sprint.issue.move.dialog.warning.scopechange')
                        }));
                    }
                    self.$container.closest('.field-group').append($scopeChangeWarning);
                }
            }
            $scopeChangeWarning.show();
        }

        function hideScopeChangeWarning() {
            if ($scopeChangeWarning) {
                $scopeChangeWarning.remove();
                $scopeChangeWarning = null;
            }
        }

        /**
         * Adds any sprints from the AJAX response to the appropriate group descriptors and formats them
         *
         * @param response
         * @returns {Array}
         */
        function formatResponse(response) {
            if (!response.suggestions && !response.allMatches) {
                return [];
            }

            var suggestionGroupDescriptor = new AJS.GroupDescriptor({
                weight: 0,
                label: AJS.I18n.getText('gh.sprint.customfield.dropdown.suggestions')
            });

            var allMatchesGroupDescriptor = new AJS.GroupDescriptor({
                weight: 1,
                label: AJS.I18n.getText('gh.sprint.customfield.dropdown.all')
            });

            _.each(sortSprintsAlphabetically(response.suggestions), function(sprint) {
                suggestionGroupDescriptor.addItem(getItemDescriptorForSprint(sprint, true));
            });

            _.each(sortSprintsAlphabetically(response.allMatches), function(sprint) {
                allMatchesGroupDescriptor.addItem(getItemDescriptorForSprint(sprint, false));
            });

            return [ suggestionGroupDescriptor, allMatchesGroupDescriptor ];
        }

        /**
         * Creates a new item descriptor to represent a sprint.
         *
         * @returns {AJS.ItemDescriptor}
         */
        function getItemDescriptorForSprint(sprint, isSuggestion) {
            var escapedName = AJS.escapeHTML(String(sprint.name));

            return new AJS.ItemDescriptor({
                value: sprint.id.toString(), // we need to convert as AJS.SingleSelect expects a string
                label: sprint.name,
                html: escapedName,
                stateKey: sprint.stateKey,
                boardName: sprint.boardName,
                allowDuplicate: false,
                isSuggestion: isSuggestion
            });
        }

        /**
         * Sorts sprints by lowercase name
         *
         * @param sprints
         * @returns {Array} sorted sprints
         */
        function sortSprintsAlphabetically(sprints) {
            return _.sortBy(sprints, function(sprint) {
                return sprint.name.toLowerCase();
            });
        }

        function registerSprintPickerChange(selectedItem) {
            if (GH.SprintConfig) {
                AJS.$(GH).trigger("QuickEdit.fieldChange", {
                    fieldId: GH.SprintConfig.getSprintFieldId(),
                    fieldChangeData: {
                        original: savedId,
                        updated: selectedItem ? selectedItem.properties.value : null
                    }
                });
            }
        }

        function initSprintPicker() {
            self._super(options);

            // override the suggestion renderer to show multiple lines
            oldSuggestionRenderer = self.listController._renders.suggestion;
            oldErrorMessageRenderer = self.showErrorMessage;

            self.listController._renders.suggestion = suggestionRenderer;
            self.showErrorMessage = errorMessageRenderer;

            if (useEmbeddedScopeChangeWarningStyle()) {
                var $inlineEditContainer = self.$container.closest('.inline-edit-fields');
                var isInlineEdit = $inlineEditContainer.size() > 0;

                if (isInlineEdit) {
                    var $input = element.parent().find('input');

                    $input.on('focus', function () {
                        self.showErrorMessage();
                    });
                }
            }

            element.bind('selected', function(e, selectedItem, context) {
                if (!useEmbeddedScopeChangeWarningStyle() || !isInlineEdit) {
                    context.showErrorMessage(context.getQueryVal());
                }

                registerSprintPickerChange(selectedItem);

                self.showErrorMessage();
            });

            // override the suggestion handler
            self.suggestionsHandler = new SprintPickerSuggestionHandler(self.options, self.model);
        }

        /**
         * The function to handle ajax error when requesting for the sprints, this will override the default error
         * behaviour in Jira to stop an alert from appearing if the ajax fails.
         *
         * @param jqXHR the xml http request object
         * @param textStatus the text status
         * @param errorThrown an error message
         */
        function handleAjaxError(jqXHR, textStatus, errorThrown) {
            var $element = AJS.$(element);

            $element.siblings('.ghx-error').remove();
            $element.before(GH.tpl.rapid.notification.renderAuiMessage({
                message: (AJS.I18n.getText("gh.error.errortitle") + ': ' + errorThrown),
                type: "error",
                className: "ghx-error aui-ss"
            }));
        }

        function useEmbeddedScopeChangeWarningStyle() {
            // This sprint picker component is also used in full page view context. In such cases, the warning will be
            // displayed as a big warning banner. Otherwise (in inline-editable agile detail view), the warning will be
            // displayed in a small dropdown underneath the field
            if (GH && GH.DetailsObjectFactory) {
                return GH.DetailsObjectFactory.isInlineEditableEnabled();
            }
            return false;
        }
    }
});

JIRA.SprintPicker.READY_EVENT = "gh.sprint-picker.ready";