AJS.$.namespace("GH.gadget.utils");

/**
 * Get the view name as defined in the url
 */
GH.gadget.utils.getViewName = function () {
    var regex = new RegExp("[\\?&]view=([^&#]*)");
    var viewType = regex.exec(window.location.href);
    return viewType ? viewType[1].toLowerCase() : '';
};

/**
 * Checks the page url for a view param abd returns true if it matches "wallboard".
 *
 * Note: According to http://wiki.opensocial.org/index.php?title=Gadgets.views_%28v0.9%29
 *       "wallboard" isn't really a supported view. Should it be default.wallboard instead?
 */
GH.gadget.utils.isWallboardView = function () {
    var viewName = GH.gadget.utils.getViewName();
    return viewName == 'wallboard';
};

/**
 * Generates the title html taking a list of items to display into account
 * @param gadget the gadget to create the dom for
 * @param data the parent element of project/version info objects
 * @param $prependTo a jquery object to which the header should be prepended to
 */
GH.gadget.utils.insertHeaderElements = function (elements, $prependTo) {
    // return if we have no items to display
    if (!elements || !elements.length || elements.length < 1) {
        return;
    }

    var titleHtml = '<h2>';
    AJS.$(elements).each(function (index, element) {
        if (index > 0) {
            titleHtml += ' : ';
        }
        titleHtml += element;
    });
    titleHtml += '</h2>';

    AJS.$(titleHtml).prependTo($prependTo);
};

GH.gadget.utils.getProjectLink = function (gadget, projectKey) {
    return gadget.getBaseUrl() + '/browse/' + projectKey;
};

GH.gadget.utils.getVersionLink = function (gadget, projectKey, versionId) {
    return GH.gadget.utils.getProjectLink(gadget, projectKey) + '/fixforversion/' + versionId;
};

GH.gadget.utils.redirectParent = function (href) {
    //Gadgets are in iframes, so window.top is required
    window.top.location.href = href;
};

/** Creates the project link markup. */
GH.gadget.utils.getProjectLinkMarkup = function (gadget, project) {
    var projectLink = GH.gadget.utils.getProjectLink(gadget, project.key);
    return '<a href="' + projectLink + '"><strong>' + AJS.escapeHTML(String(project.name)) + '</strong> (' + project.key + ')</a>';
};

/** Creates the version link markup. */
GH.gadget.utils.getVersionLinkMarkup = function (gadget, project, version) {
    // generate version link if id available, project link otherwise
    if (version.id) {
        var link = GH.gadget.utils.getVersionLink(gadget, project.key, version.id);
    } else {
        var link = GH.gadget.utils.getProjectLink(gadget, project.key);
    }
    return '<a href="' + link + '" class="gg-days-version"><strong>' + version.name + '</strong></a>';
};

/**
 * Generates the project/version title display
 * @param gadget the gadget to create the dom for
 * @param data the parent element of project/version info objects
 * @param $prependTo a jquery object to which the header should be prepended to
 */
GH.gadget.utils.getProjectVersionHeaderElements = function (gadget, data) {
    // add the title if requested
    var titleElements = [];
    var project = data.project;
    var version = data.version;

    // project
    var showProjectName = gadget.getPref('showProjectName') == 'true';
    if (showProjectName) {
        titleElements.push(GH.gadget.utils.getProjectLinkMarkup(gadget, project));
    }

    // version
    var showVersionName = gadget.getPref('showVersionName') == 'true';
    if (showVersionName) {
        titleElements.push(GH.gadget.utils.getVersionLinkMarkup(gadget, project, version));
    }

    return titleElements;
};

/**
 * Generates the project/version title display
 * @param gadget the gadget to create the dom for
 * @param data the parent element of project/version info objects
 * @param $prependTo a jquery object to which the header should be prepended to
 */
GH.gadget.utils.addProjectVersionHeader = function (gadget, data, $prependTo) {
    // fetch the elements to display
    var titleElements = GH.gadget.utils.getProjectVersionHeaderElements(gadget, data);
    GH.gadget.utils.insertHeaderElements(titleElements, $prependTo);
};

/**
 * Show inline warning message
 */
GH.gadget.utils.showInlineWarning = function (gadget, message) {
    var $view = gadget.getView();
    $view.empty();
    AJS.messages.warning($view, {
        autoHide: false,
        shadowed: false,
        closeable: false,
        body: message
    });
    gadget.resize();
};

AJS.$.namespace("GH.gadget.fields");

/**
 * Creates a section title
 */
GH.gadget.fields.sectionTitle = function (gadget, id, label) {
    return {
        label: 'none',
        id: id,
        type: "callbackBuilder",
        callback: function callback(parentDiv) {
            // remove the automatically generated title
            parentDiv.parent().children('label').remove();
            // remove the automatically generated error field
            parentDiv.parent().children('#-error').remove();

            // now generate the html
            var html = '<h3>' + gadget.getMsg(label) + '</h3>';
            parentDiv.append(AJS.$(html));
        }
    };
};

/**
 * Creates a checkbox
 */
GH.gadget.fields.checkbox = function (gadget, checkboxPref, label, selected, description) {
    return {
        userpref: checkboxPref,
        label: gadget.getMsg(label),
        id: "checkbox_" + checkboxPref,
        type: "callbackBuilder",
        callback: function callback(parentDiv) {
            // add the checkbox class and wrap with fieldset
            parentDiv.addClass('checkbox').wrap('<fieldset class="group" />');
            // remove the label added by the the caller code, as we want the label to appear after the input element
            parentDiv.parent().siblings('label').remove();
            // we don't want field-group and fieldset
            parentDiv.closest('.field-group').removeClass('field-group');
            // we don't want this legacy class either
            parentDiv.removeClass('builder-container');

            var checkboxId = checkboxPref + '_checkbox';
            var checkedHtml = selected ? ' checked' : '';
            var html = '<input type="checkbox" class="checkbox" id="' + checkboxId + '" ' + checkedHtml + ' />';
            html += '<label for="' + checkboxId + '">' + gadget.getMsg(label) + '</label>';
            html += '<input type="hidden" id="' + checkboxPref + '" name="' + checkboxPref + '" value="' + selected + '" />';
            parentDiv.append(AJS.$(html));

            // make sure we update the values that get submitted
            AJS.$('#' + checkboxId, parentDiv).each(function () {
                var checkbox = this;
                var $checkbox = AJS.$(checkbox);
                $checkbox.change(function () {
                    var hiddenInputSelector = '#' + checkboxPref;
                    AJS.$(hiddenInputSelector).attr('value', checkbox.checked);
                });
            });

            // add description
            if (description) {
                parentDiv.append(AJS.$("<div/>").addClass("description").text(gadget.getMsg(description)));
            }
        }
    };
};

/**
 * Updates the versionSelect Option dropdown with new values given a project selection list.
 * @param showUnscheduled should unscheduled versions be included in the list
 * @param showAutoVersion Should the next due version aka auto version be added to the list
 */
GH.gadget.fields.updateVersionSelectOptions = function (gadget, selectedProjectId, versionUserPref, showUnscheduled, showAutoVersion) {
    var versionSelectList = AJS.$("#" + versionUserPref);

    versionSelectList.empty();
    var versions = AJS.$.ajax({
        key: "versions",
        url: "/rest/greenhopper/1.0/versionBoard-list/generate",
        data: {
            showUnscheduled: showUnscheduled,
            projectId: selectedProjectId
        },
        success: function success(response) {
            versions = response;
            var versionUserPrefValue = gadget.getPref(versionUserPref);

            // add auto version if requested
            if (showAutoVersion) {
                // create and add auto option
                var autoOptionValue = 'auto';
                var autoOption = AJS.$("<option />").addClass("next-release-option").val(autoOptionValue).text(gadget.getMsg("gh.gadget.days.remaining.autoOption"));
                if (versionUserPrefValue == autoOptionValue) {
                    autoOption.attr("selected", "selected");
                }
                versionSelectList.append(autoOption);
            }

            // add all available versions
            AJS.$(versions.versionOptions).each(function () {
                // if we show an auto version, we do not want to display the "no versions available selection"
                if (showAutoVersion && this.value == '') {
                    return;
                }
                var versionOption = AJS.$("<option/>").attr("value", this.value).text(this.label);
                versionSelectList.append(versionOption);

                if (versionUserPrefValue == this.value) {
                    versionOption.attr("selected", "selected");
                }
            });
        }
    });
};

/**
 * Updates the watched field selection options
 */
GH.gadget.fields.updateFieldSelectOptions = function (gadget, selectedProjectId, fieldUserPref, showSystem) {
    var fieldSelectList = AJS.$("#" + fieldUserPref);

    fieldSelectList.empty();
    var fields = AJS.$.ajax({
        key: "fields",
        url: "/rest/greenhopper/1.0/watched-fields/generate",
        data: {
            projectId: selectedProjectId,
            showSystem: showSystem
        },
        success: function success(response) {
            if (!response) {
                return;
            }

            var fields = response.fields;
            var fieldUserPrefValue = gadget.getPref(fieldUserPref);

            // add all available options
            AJS.$(fields).each(function () {
                var option = AJS.$("<option/>").attr("value", this.value).text(this.label);
                fieldSelectList.append(option);

                if (fieldUserPrefValue == this.value) {
                    option.attr("selected", "selected");
                }
            });
        }
    });
};

/**
 * Updates the context selection options.
 */
GH.gadget.fields.updateContextSelectionOptions = function (gadget, selectedProjectId, contextUserPref) {
    // fetch and empty the multiselection list
    var contextMultiSelectList = AJS.$("#" + contextUserPref);
    contextMultiSelectList.empty();

    // fetch the contexts for the selected project
    var contexts = AJS.$.ajax({

        key: "contexts",
        url: "/rest/greenhopper/1.0/context-list/generate",
        data: {
            projectId: selectedProjectId,
            returnIds: true
        },
        success: function success(response) {
            var selectedContexts = gadget.getPref(contextUserPref).split("|");
            contexts = response;
            AJS.$(contexts.contextOptions).each(function () {
                var contextOption = AJS.$("<option/>").attr("value", this.value).text(this.label);
                contextMultiSelectList.append(contextOption);
                for (var i = 0, l = selectedContexts.length; i < l; i++) {
                    if (selectedContexts[i] == this.value) {
                        contextOption.attr("selected", "selected");
                    }
                }
            });
        }
    });
};

/**
 * Updates the task board mapping selection options.
 */
GH.gadget.fields.updateTBMappingSelectionOptions = function (gadget, selectedProjectId, tbMappingUserPref) {
    // fetch and empty the multiselection list
    var tbMappingMultiSelectList = AJS.$("#" + tbMappingUserPref);
    tbMappingMultiSelectList.empty();

    // fetch the contexts for the selected project
    var contexts = AJS.$.ajax({

        key: "tbmapping",
        url: "/rest/greenhopper/1.0/tbmapping-list/generate",
        data: {
            projectId: selectedProjectId
        },
        success: function success(response) {
            var selectedMappings = gadget.getPref(tbMappingUserPref).split("|");
            var tbMapping = response;
            AJS.$(tbMapping.tbMappingOptions).each(function () {
                var option = AJS.$("<option/>").attr("value", this.value).text(gadget.getMsg(this.label));
                tbMappingMultiSelectList.append(option);
                for (var i = 0, l = selectedMappings.length; i < l; i++) {
                    if (selectedMappings[i] == this.value) {
                        option.attr("selected", "selected");
                    }
                }
            });
        }
    });
};

/**
 * GreenHopper project picker
 *
 * @param projectUserPref the id under which the project is stored
 * @param projectOptions the result of the ajax call that provides the available projects
 */
GH.gadget.fields.projectPicker = function (gadget, projectUserPref, projectOptions) {
    return {
        userpref: projectUserPref,
        label: gadget.getMsg("gadget.common.project.label"),
        id: "project_picker_" + projectUserPref,
        type: "callbackBuilder",
        callback: function callback(parentDiv) {
            // add a project options box and description. Note that the label has already been added
            parentDiv.append(AJS.$("<select/>").attr({
                id: projectUserPref,
                name: projectUserPref
            }).addClass('select'));

            // now fill the project list with values
            var $projectSelectList = AJS.$("#" + projectUserPref);
            $projectSelectList.empty();

            //append the options to the projectPicker, selecting the currently selected project
            var projectUserPrefValue = gadget.getPref(projectUserPref);
            AJS.$(projectOptions).each(function () {
                var projectOption = AJS.$("<option/>").attr("value", this.value).text(this.label);
                $projectSelectList.append(projectOption);
                if (this.value == projectUserPrefValue) {
                    projectOption.attr({ selected: "selected" });
                }
            });
        }
    };
};

/**
 * GreenHopper project picker
 *
 * @param projectUserPref the id under which the project is stored
 * @param versionUserPref the id under which the version is stored
 * @param showUnscheduled whether the unscheduled version should be available
 * @param showAutoVerion whether a "Next Due Version" option should be available
 */
GH.gadget.fields.versionPicker = function (gadget, projectUserPref, versionUserPref, showUnscheduled, showAutoVersion) {
    return {
        userpref: versionUserPref,
        label: gadget.getMsg("gh.gadget.common.version"),
        id: "version_picker_" + versionUserPref,
        type: "callbackBuilder",
        callback: function callback(parentDiv) {
            // version box label, select box and description
            parentDiv.append(AJS.$("<select>").attr({
                id: versionUserPref,
                name: versionUserPref
            }).addClass('select'));

            // initialize the dropdown
            var $projectSelectList = AJS.$("#" + projectUserPref);
            GH.gadget.fields.updateVersionSelectOptions(gadget, $projectSelectList.val(), versionUserPref, showUnscheduled, showAutoVersion);

            // get informed of changes
            $projectSelectList.change(function (event) {
                GH.gadget.fields.updateVersionSelectOptions(gadget, $projectSelectList.val(), versionUserPref, showUnscheduled, showAutoVersion);
            });
        }
    };
};

/**
 * GreenHopper project picker
 *
 * @param projectUserPref the id under which the project is stored
 * @param fieldUserPref the id under which the field is stored
 * @param showSytem boolean that determines if system fields are shown or not
 */
GH.gadget.fields.fieldPicker = function (gadget, projectUserPref, fieldUserPref, showSystem) {
    //Label is hardcoded for burndown and taskboard gadget. Should probably be passed in.
    var label = '';
    if (showSystem) {
        label = gadget.getMsg("gh.gadget.task.board.count");
    } else {
        label = gadget.getMsg("gh.gadget.custom.field.burndown.field.label");
    }
    return {
        userpref: fieldUserPref,
        label: label,
        id: "field_picker_" + fieldUserPref,
        type: "callbackBuilder",
        callback: function callback(parentDiv) {
            parentDiv.append(AJS.$("<select>").attr({
                id: fieldUserPref,
                name: fieldUserPref
            }).addClass('select'));

            // update the values
            var $projectSelectList = AJS.$("#" + projectUserPref);
            GH.gadget.fields.updateFieldSelectOptions(gadget, $projectSelectList.val(), fieldUserPref, showSystem);

            // get informed of changes
            $projectSelectList.change(function (event) {
                GH.gadget.fields.updateFieldSelectOptions(gadget, $projectSelectList.val(), fieldUserPref, showSystem);
            });
        }
    };
};

/**
 * GreenHopper project picker
 *
 * @param projectUserPref the id under which the project is stored
 * @param contextUserPref the id under which the context is stored
 * @param multipleContexts if true a multiselect field will be displayed, a dropdown if false
 */
GH.gadget.fields.contextPicker = function (gadget, projectUserPref, contextUserPref, multipleContexts) {
    return {
        userpref: contextUserPref,
        label: gadget.getMsg("gh.gadget.common.context"),
        id: "context_picker_" + contextUserPref,
        type: "callbackBuilder",
        callback: function callback(parentDiv) {
            var selectAttrs = {
                id: contextUserPref,
                name: contextUserPref
            };
            var className = "select";
            if (multipleContexts) {
                selectAttrs.multiple = "multiple";
                selectAttrs.size = "4";
                className = "multi-select";
            }
            parentDiv.append(AJS.$("<select>").attr(selectAttrs).addClass(className));

            // update the values
            var $projectSelectList = AJS.$("#" + projectUserPref);
            GH.gadget.fields.updateContextSelectionOptions(gadget, $projectSelectList.val(), contextUserPref);

            // get informed of changes
            $projectSelectList.change(function (event) {
                GH.gadget.fields.updateContextSelectionOptions(gadget, $projectSelectList.val(), contextUserPref);
            });
        }
    };
};

/**
 * GreenHopper task board mapping picker
 *
 * @param projectUserPref the id under which the project is stored
 * @param tbMappingUserPref the id under which the context is stored
 */
GH.gadget.fields.tbMappingPicker = function (gadget, projectUserPref, tbMappingUserPref) {
    return {
        userpref: tbMappingUserPref,
        label: gadget.getMsg("gh.gadget.common.tbcolumns"),
        id: "tbMapping_picker_" + tbMappingUserPref,
        type: "callbackBuilder",
        callback: function callback(parentDiv) {
            var selectAttrs = {
                id: tbMappingUserPref,
                name: tbMappingUserPref
            };
            var className = "select";
            selectAttrs.multiple = "multiple";
            selectAttrs.size = "4";
            className = "multi-select";
            parentDiv.append(AJS.$("<select>").attr(selectAttrs).addClass(className));

            // update the values
            var $projectSelectList = AJS.$("#" + projectUserPref);
            GH.gadget.fields.updateTBMappingSelectionOptions(gadget, $projectSelectList.val(), tbMappingUserPref);

            // get informed of changes
            $projectSelectList.change(function (event) {
                GH.gadget.fields.updateTBMappingSelectionOptions(gadget, $projectSelectList.val(), tbMappingUserPref);
            });
        }
    };
};