/* global c3: true, d3: true */
/**
 * Plot an array of points as a simple barchart
 */
c3.barChart = function () {

    return c3.component('barChart')
        .extend(c3.drawable())
        .extend(c3.plottable())
        .elementTag('rect')
        .xAccessor(function (d, i) {
            return i;
        })
        .yAccessor(function (d) {
            return d;
        })
        .extend({
            baselineAccessor: c3.prop(function () {
                return 0;
            }),
            barWidth: function () {
                return this.xScale().rangeBand();
            },
            barHeight: function () {
                var baselineAccessor = this.baselineAccessor();
                var yAccessor = this.yAccessor();
                var yScale = this.yScale();

                return function (d) {
                    return yScale(baselineAccessor(d)) - yScale(yAccessor(d));
                };
            },
            maxBarWidth: c3.prop(0),
            padding: c3.prop(0.25).set(function (newValue) {
                if (newValue >= 0 && newValue <= 1) {
                    return newValue;
                } else {
                    throw new Error('Expected padding to be between 0 and 1 inclusive, but received ' + newValue);
                }
            }),
            outerPadding: c3.prop(0.25).set(function (newValue) {
                if (newValue >= 0 && newValue <= 1) {
                    return newValue;
                } else {
                    throw new Error('Expected outerPadding to be between 0 and 1 inclusive, but received ' + newValue);
                }
            }),
            showLabels: c3.prop(false),
            rotateLabels: c3.prop(false),
            labelAccessor: function () {
                return this.xAccessor();
            },
            xRange: function () {
                var chartWidth = this.width();
                var maxWidth = chartWidth;
                if (this.maxBarWidth() > 0) {
                    maxWidth = Math.floor(this.maxBarWidth() / (1 - this.padding()) * (this.xDomain().length - this.padding() + 2 * this.outerPadding()));
                }
                return [0, Math.min(maxWidth, chartWidth)];
            },
            xScale: function () {
                return d3.scale.ordinal()
                    .domain(this.xDomain())
                    .rangeRoundBands(this.xRange(), this.padding(), this.outerPadding());
            },
            xDomain: c3.inherit('xDomain').onDefault(function () {
                if (c3.isEmpty(this.data())) return [0, 1];
                return d3.range(this.data().length);
            }),
            yDomain: c3.inherit('yDomain').onDefault(function () {
                if (c3.isEmpty(this.data())) return [0, 1];

                var min = c3.checkIsNumber(d3.min(this.data(), this.baselineAccessor()));
                var max = c3.checkIsNumber(d3.max(this.data(), this.yAccessor()));
                return [min, max];
            })
        })
        .enter(function(event) {
            if (this.elementTag() === 'g' && this.showLabels()) {
                event.selection.append('text')
                    .classed('bar-label', true);
            }
        })
        .update(function(event) {
            var x = this.x();
            var y = this.y();
            var barWidth = this.barWidth();
            var barHeight = this.barHeight();
            var label = event.selection.select('text.bar-label');

            event.selection
                .attr('transform', function (d, i) {
                    return 'translate(' + x(d, i) + ',' + y(d, i) + ')';
                })
                .attr('height', barHeight)
                .attr('width', barWidth);

            if (this.showLabels()) {
                label
                    .attr('x', barWidth / 2)
                    .attr('y', barHeight)
                    .text(this.labelAccessor());

                if (this.rotateLabels()) {
                    label
                        .attr('transform', function (d,i) {
                            return 'rotate(90 ' + barWidth / 2 + ',' + barHeight(d) + ')';
                        })
                        .attr('text-anchor', 'start')
                        .attr('dx', '.5em')
                        .attr('dy', '.25em');
                } else {
                    label
                        .attr('transform', null)
                        .attr('text-anchor', 'middle')
                        .attr('dx', null)
                        .attr('dy', '1.25em')
                        .call(c3.utils.wrapText, barWidth);
                }

            }
        });
};