define('jira-agile/rapid/analytics-tracker', ['require'], function (require) {
    var _ = require('underscore');

    /**
     * Construct a new tracker object to trigger analytics events. The only required parameter is the category.
     *
     * See https://developers.google.com/analytics/devguides/collection/gajs/eventTrackerGuide#Anatomy for more on how
     * analytics data is shaped. Note that Google Analytics is not used to capture these events.
     *
     * @param {String} category. The category identifier to use for all events.
     * @param {String} [action] The action identifier to use for all events.
     * @param {String} [label] The label identifier to use for all events.
     * @constructor
     */
    function AnalyticsTracker(category, action, label) {
        if (_.isUndefined(category)) {
            throw new Error("You must specify a category");
        }
        this.params = {};
        this.params.category = category;

        var argIndex = 0;
        if (!_.isUndefined(action)) {
            this.params.action = action;
            argIndex++;
            if (!_.isUndefined(label)) {
                this.params.label = label;
                argIndex++;
            }
        }

        // based on which arguments were specified at construction time, this will be the list of arguments (in order) that
        // we will expect when trigger() is called
        this.requiredArgumentNames = ['action', 'label', 'value'].slice(argIndex);

        // by default we will track events asynchronously
        this.useAsync = true;
    }

    /**
     * Set whether this tracker should send async or sync events.
     *
     * @param useAsync {Boolean}
     * @return {AnalyticsTracker}
     */
    AnalyticsTracker.prototype.setAsync = function (useAsync) {
        this.useAsync = useAsync;
        return this;
    };

    /**
     * Triggers an event on <code>AJS.$(AJS)</code> with the data sent through to this function. The arguments to this function
     * will be interpreted based on how this tracker was constructed. If it was constructed with a <code>category</code> and
     * <code>action</code>, then the first argument to this function will be the <code>label</code>. If no arguments are passed,
     * Only the parameters specified at construction time will be sent.
     */
    AnalyticsTracker.prototype.trigger = function () {
        // if we have more arguments than we need, log
        if (this.requiredArgumentNames.length < arguments.length) {
            AJS.log("Warning: too many arguments passed in. Needed at most " + this.requiredArgumentNames.length + " but got " + arguments.length);
        }

        var params = _.clone(this.params);
        for (var i = 0; i < arguments.length && !_.isUndefined(this.requiredArgumentNames[i]); i++) {
            params[this.requiredArgumentNames[i]] = arguments[i];
        }

        this._validateParams(params);

        // trigger a tracking event
        var eventKey = "gh.analytics.async";
        AJS.$(AJS).trigger(eventKey, params);
    };

    AnalyticsTracker.prototype._validateParams = function (params) {
        // ensure category, action, label are strings
        var stringKeys = ['category', 'action', 'label'];
        _.each(stringKeys, function (key) {
            if (!_.isUndefined(params[key]) && !_.isString(params[key])) {
                params[key] = params[key].toString();
            } else if (_.isUndefined(params[key])) {
                params[key] = "";
            }
        });

        // ensure category and action strings do not contain spaces
        _.each(['category', 'action'], function (key) {
            params[key] = params[key].replace(/\s+/g, '');
        });

        // validate 'value' as integer
        if (!_.isUndefined(params.value)) {
            params.value = parseInt(params.value, 10);
            if (isNaN(params.value)) {
                params.value = undefined;
            }
        }
    };

    return AnalyticsTracker;
});

AJS.namespace('GH.AnalyticsTracker', null, require('jira-agile/rapid/analytics-tracker'));