/* global module, test, equal */

AJS.test.require("com.pyxis.greenhopper.jira:gh-rapid-common", function () {
    var TimeFormat = require("jira-agile/rapid/time-format");
    var TimeFormatDurationUnits = require("jira-agile/rapid/time-format-duration-units");
    var TimeFormatDisplays = require("jira-agile/rapid/time-format-displays");

    module("Time Format tests", {
        setup: function setup() {
            this.testFormatShortDuration = function (weeks, days, hours, minutes, opts, expected) {
                var MILLIS_PER_MINUTE = 1000 * 60;
                var MILLIS_PER_HOUR = MILLIS_PER_MINUTE * 60;

                var duration = MILLIS_PER_HOUR * (opts.hoursPerDay * opts.daysPerWeek * weeks) + MILLIS_PER_HOUR * (opts.hoursPerDay * days) + MILLIS_PER_HOUR * hours + MILLIS_PER_MINUTE * minutes;

                equal(TimeFormat._formatShortDuration(duration, _.extend(opts, { durationUnit: TimeFormatDurationUnits.MILLISECONDS })), expected, "Time not equal");
                equal(TimeFormat._formatShortDuration(duration / 1000, _.extend(opts, { durationUnit: TimeFormatDurationUnits.SECONDS })), expected, "Time not equal");
            };
        }
    });

    test("TimeFormat#formatDuration", function () {
        // 2 d, 1h, 20m
        var duration = 1000 * 60 * 60 * 24 * 2 + 1000 * 60 * 60 * 1 + 1000 * 60 * 20;

        //2 days 1 hour 20 minutes
        equal(TimeFormat.formatDuration(duration), "2 core.dateutils.days 1 core.dateutils.hour 20 core.dateutils.minutes", "Time not equal");
    });

    test("TimeFormat#formatShortDuration with normal display options", function () {
        var opts = {
            hoursPerDay: 24,
            daysPerWeek: 7,
            timeFormat: TimeFormatDisplays.PRETTY
        };

        // 0
        this.testFormatShortDuration(0, 0, 0, 0, opts, "0");

        // 2d, 1h, 20m
        this.testFormatShortDuration(0, 2, 1, 20, opts, "2core.durationutils.unit.day 1core.durationutils.unit.hour 20core.durationutils.unit.minute");

        // 3w, 2d, 1h, 20m
        this.testFormatShortDuration(3, 2, 1, 20, opts, "3w 2core.durationutils.unit.day 1core.durationutils.unit.hour 20core.durationutils.unit.minute");

        // using DAYS format
        opts.timeFormat = TimeFormatDisplays.DAYS;

        // 3w, 2d, 1h using DAYS display format - 23d 1h
        this.testFormatShortDuration(3, 2, 1, 0, opts, "23core.durationutils.unit.day 1core.durationutils.unit.hour");

        // 3w, 2d, 1h, 20m using DAYS display format - 23d 1.33h
        this.testFormatShortDuration(3, 2, 1, 20, opts, "23core.durationutils.unit.day 1.33core.durationutils.unit.hour");

        // 3w, 2d, 1h, 30m using DAYS display format - 23d 1.5h
        this.testFormatShortDuration(3, 2, 1, 30, opts, "23core.durationutils.unit.day 1.5core.durationutils.unit.hour");

        // using HOURS format
        opts.timeFormat = TimeFormatDisplays.HOURS;

        // 1w, 3d, 5h using HOURS display format - 245h
        this.testFormatShortDuration(1, 3, 5, 0, opts, "245core.durationutils.unit.hour");

        // 1w, 3d, 5h, 45m using HOURS display format - 245.75h
        this.testFormatShortDuration(1, 3, 5, 45, opts, "245.75core.durationutils.unit.hour");

        // 1w, 3d, 5h, 30m using HOURS display format - 245.5h
        this.testFormatShortDuration(1, 3, 5, 30, opts, "245.5core.durationutils.unit.hour");
    });

    test("TimeFormat#formatShortDuration with custom options", function () {
        var opts = {
            hoursPerDay: 8,
            daysPerWeek: 5,
            timeFormat: TimeFormatDisplays.PRETTY
        };

        // using PRETTY format

        // 2d, 1h, 20m
        this.testFormatShortDuration(0, 2, 1, 20, opts, "2core.durationutils.unit.day 1core.durationutils.unit.hour 20core.durationutils.unit.minute");

        // 3w, 2d, 1h, 20m
        this.testFormatShortDuration(3, 2, 1, 20, opts, "3w 2core.durationutils.unit.day 1core.durationutils.unit.hour 20core.durationutils.unit.minute");

        // using DAYS format
        opts.timeFormat = TimeFormatDisplays.DAYS;

        // 3w, 2d, 1h using DAYS display format - 17d 1h
        this.testFormatShortDuration(3, 2, 1, 0, opts, "17core.durationutils.unit.day 1core.durationutils.unit.hour");

        // 3w, 2d, 1h, 20m using DAYS display format - 17d 1.33h
        this.testFormatShortDuration(3, 2, 1, 20, opts, "17core.durationutils.unit.day 1.33core.durationutils.unit.hour");

        // 3w, 2d, 1h, 30m using DAYS display format - 17d 1.5h
        this.testFormatShortDuration(3, 2, 1, 30, opts, "17core.durationutils.unit.day 1.5core.durationutils.unit.hour");

        // using HOURS format
        opts.timeFormat = TimeFormatDisplays.HOURS;

        // 1w, 3d, 5h using HOURS display format - 69h
        this.testFormatShortDuration(1, 3, 5, 0, opts, "69core.durationutils.unit.hour");

        // 1w, 3d, 5h, 45m using HOURS display format - 69.75h
        this.testFormatShortDuration(1, 3, 5, 45, opts, "69.75core.durationutils.unit.hour");

        // 1w, 3d, 5h, 30m using HOURS display format - 69.75h
        this.testFormatShortDuration(1, 3, 5, 30, opts, "69.5core.durationutils.unit.hour");
    });

    test("TimeFormat#formatShortDuration with small duration", function () {
        var opts = {
            hoursPerDay: 8,
            daysPerWeek: 5,
            timeFormat: TimeFormatDisplays.PRETTY
        };

        // NOTE: core.durationutils.unit.minute resolves to {0}m (which becomes < 1m), which is why the test doesn't show '< 1core.durationutils.unit.minute'

        // using PRETTY format - < 1m
        this.testFormatShortDuration(0, 0, 0, 0.5, opts, "core.durationutils.unit.minute");

        // using DAYS format - < 1h
        opts.timeFormat = TimeFormatDisplays.DAYS;
        this.testFormatShortDuration(0, 0, 0, 0.5, opts, "core.durationutils.unit.hour");

        // using HOURS format - < 1h
        opts.timeFormat = TimeFormatDisplays.HOURS;
        this.testFormatShortDuration(0, 0, 0, 0.5, opts, "core.durationutils.unit.hour");
    });

    test("TimeFormat#formatShortDuration round to most significant unit", function () {
        var opts = {
            hoursPerDay: 10,
            daysPerWeek: 5,
            timeFormat: TimeFormatDisplays.MOST_SIGNIFICANT
        };

        this.testFormatShortDuration(0, 0, 0, 0, opts, "0"); // 0
        this.testFormatShortDuration(0, 0, 0, 0.5, opts, "core.durationutils.unit.minute"); // < 1m
        this.testFormatShortDuration(0, 0, 0, 1, opts, "1core.durationutils.unit.minute"); // 1m
        this.testFormatShortDuration(0, 0, 0, 59, opts, "59core.durationutils.unit.minute"); // 59m
        this.testFormatShortDuration(0, 0, 1, 0, opts, "1core.durationutils.unit.hour"); // 1h
        this.testFormatShortDuration(0, 0, 1, 30, opts, "1.5core.durationutils.unit.hour"); // 1.5h
        this.testFormatShortDuration(0, 0, 1, 59, opts, "2core.durationutils.unit.hour"); // 2h
        this.testFormatShortDuration(0, 1, 1, 59, opts, "1.2core.durationutils.unit.day"); // 1.2d
        this.testFormatShortDuration(1, 1, 1, 59, opts, "1.2w"); // 1 day is 0.2 weeks - 1.2w
        this.testFormatShortDuration(0, 4, 9, 29, opts, "4.9core.durationutils.unit.day"); // 4.9d
        this.testFormatShortDuration(0, 4, 9, 30, opts, "5core.durationutils.unit.day"); // 5d
        this.testFormatShortDuration(10, 0, 2, 29, opts, "10w"); // 10w
        this.testFormatShortDuration(10, 0, 2, 30, opts, "10.1w"); // 0.5 days is 0.1 weeks. Half of 0.5 days should round up. - 10.1w
    });

    /**
     * Test the order of inheritance for time format options.
     *
     * Expected order is:
     * {}
     * TimeFormat.normalDisplayOpts - default
     * TimeFormat.timeTrackingConfigOpts - user configured overrides (from server)
     * options - specific overrides for this invocation
     */
    test("TimeFormat#formatShortDurationForTimeTrackingConfiguration inherits options", function () {

        // we'll use hoursPerDay to test to order of inheritance as it's easy to check.

        var oneDaySeconds = 60 * 60 * TimeFormat.normalDisplayOpts.hoursPerDay;
        // test with no server overrides and undefined custom options
        TimeFormat.timeTrackingConfigOpts = {}; // set no server overrides
        // when options is not defined, input units are interpreted as seconds not milliseconds.
        equal("1core.durationutils.unit.day", TimeFormat.formatShortDurationForTimeTrackingConfiguration(oneDaySeconds));

        var oneDayMillis = oneDaySeconds * 1000;

        // test with no server overrides and no custom options
        TimeFormat.timeTrackingConfigOpts = {}; // set no server overrides
        var options = {};
        equal("1core.durationutils.unit.day", TimeFormat.formatShortDurationForTimeTrackingConfiguration(oneDayMillis, options));

        // test with server overrides with no custom options
        TimeFormat.timeTrackingConfigOpts = {
            hoursPerDay: 8
        };
        options = {};
        equal("3core.durationutils.unit.day", TimeFormat.formatShortDurationForTimeTrackingConfiguration(oneDayMillis, options));

        // test with server overrides and custom options - custom options should take precedence
        TimeFormat.timeTrackingConfigOpts = {
            hoursPerDay: 1
        };
        options = {
            hoursPerDay: 12
        };
        equal("2core.durationutils.unit.day", TimeFormat.formatShortDurationForTimeTrackingConfiguration(oneDayMillis, options));

        // test with no server overrides, but custom options - custom options should still take precedence
        TimeFormat.timeTrackingConfigOpts = {};
        options = {
            hoursPerDay: 6
        };
        equal("4core.durationutils.unit.day", TimeFormat.formatShortDurationForTimeTrackingConfiguration(oneDayMillis, options));
    });
});