/**
 * Triggers 'focus' and 'blur' events when the window is focused/blurred.
 * Unlike the native focus and blur events, these events are debounced so that only a single
 * event is triggered if focus/blur occurs in quick succession.
 * Triggers an 'unload' event when the window is unloaded.
 *
 * @constructor
 */
define("jira-agile/rapid/window-activity-notifier", ["underscore", "jquery", "jira-agile/rapid/events"], function (_, $, Events) {
    "use strict";

    var WindowActivityNotifier = new Events();

    WindowActivityNotifier._bindFocusBlur = function () {
        var notifier = this;
        var cancelledByUnload = false;
        // Focus and blur are consolidated into the same function so they can share
        // the same debounce timer.
        var handleFocusBlur = _.debounce(function (focus) {
            // Don't want to interrupt unload with additional events
            if (cancelledByUnload) {
                return;
            }
            if (focus) {
                notifier.trigger('focus');
            } else {
                notifier.trigger('blur');
            }
        }, 200);
        var focus = _.partial(handleFocusBlur, true);
        var blur = _.partial(handleFocusBlur, false);

        $(window).on({
            focus: focus,
            blur: blur,
            unload: function unload() {
                cancelledByUnload = true;
                notifier.trigger('unload');
            }
        });
    };

    WindowActivityNotifier._bindFocusBlur();

    return WindowActivityNotifier;
});

AJS.namespace("GH.windowActivityNotifier", null, require("jira-agile/rapid/window-activity-notifier"));