AJS.test.require(["com.pyxis.greenhopper.jira:gh-test-common-base", "com.pyxis.greenhopper.jira:gh-rapid-config"], function () {
    var $ = require('jquery');
    var _ = require('underscore');

    module("ColumnConfigAnalytics tests", {
        setup: function () {
            this.context = AJS.test.mockableModuleContext();

            this.jiraAnalytics = {
                send: sinon.spy()
            };
            this.context.mock('jira/analytics', this.jiraAnalytics);
            this.analytics = this.context.require("jira-agile/rapid/configuration/column-config-analytics");
            setUpGlobals(true, true, true);
        },
        teardown: function () {
        },

        assertEventTriggered: function (name, properties) {
            ok("Spy should be called once", this.jiraAnalytics.send.calledOnce);

            var options = this.jiraAnalytics.send.getCall(0).args[0];
            equal(options.name, name, "Name of event should be valid");
            deepEqual(options.properties, properties, "Properties should be equal");
        },

        assertTriggerDnD: function (properties) {
            this.assertEventTriggered("jira-software.config.columns.status.dnd", properties);
        },

        assertTriggerOpened: function (properties) {
            this.assertEventTriggered("jira-software.config.columns.opened", properties);
        }
    });

    function setUpGlobals(boardKanban, kanplanEnabled, kanplanEnabledForBoard) {
        GH.RapidBoard.Config.Columns.kanPlanConfig = {};
        GH.RapidBoard.Config.Columns.kanPlanConfig.isKanbanBoard = boardKanban;
        GH.RapidBoard.Config.Columns.kanPlanConfig.isKanPlanLabEnabled = kanplanEnabled;
        GH.RapidBoard.Config.Columns.kanPlanConfig.isKanPlanEnabled = kanplanEnabledForBoard;
    }

    function mockBacklogColumn() {
        return $('<div class="ghx-kanplan-backlog"></div>');
    }

    function mockUnmappedColumn() {
        return $('<div class="ghx-unmapped"></div>');
    }

    function mockColumn() {
        return $('<div></div>');
    }

    function column(backlog, firstColumnWithStatuses, statuses) {
        if (typeof statuses === "undefined") {
            statuses = [];
        }

        return {
            id: _.uniqueId(),
            isKanPlanColumn: backlog,
            isFirstColumnWithStatuses: firstColumnWithStatuses,
            mappedStatuses: statuses
        }
    }

    function status(issueCount) {
        return {
            issueCount: issueCount,
            id: _.uniqueId()
        }
    }

    function model(id, columns, unmapped, showEpicAsPanel) {
        if (typeof unmapped === 'undefined') {
            unmapped = [];
        }

        if (typeof showEpicAsPanel === 'undefined') {
            showEpicAsPanel = true;
        }

        return {
            rapidViewId: id,
            mappedColumns: columns,
            unmappedStatuses: unmapped,
            showEpicAsPanel: showEpicAsPanel
        }
    }

    test('assert sample data', function () {
        var columnModel = model(3, [
            column(true, false),
            column(false, false),
            column(false, true, [status(1), status(2)]),
            column(false, false, [status(10)])
        ], [], false);

        this.analytics.triggerStatusDnDAnalytics(columnModel, mockColumn(), mockColumn());

        this.assertTriggerDnD({
            "sourceColumnType": "column",
            "destinationColumnType": "column",
            "boardId": 3,
            "kanbanBoard": true,
            "kanplanForInstanceEnabled": true,
            "columns": 4,
            "issuesInFirstColumnWithStatuses": 3,
            "issues": 13,
            "unmappedIssues": 0,
            "kanplanDisabledAfterOperation": false,
            "kanplanEnabledAfterOperation": false,
            "issuesInBacklog": 0,
            "kanplanEnabled": false,
            "epicsPanelEnabled": false
        });
    });

    test('should send number of unmapped statuses', function () {
        var columnModel = model(3, [
            column(false, false)
        ], [status(4)]);

        this.analytics.triggerStatusDnDAnalytics(columnModel, mockColumn(), mockColumn());

        this.assertTriggerDnD({
            "sourceColumnType": "column",
            "destinationColumnType": "column",
            "boardId": 3,
            "kanbanBoard": true,
            "kanplanForInstanceEnabled": true,
            "columns": 1,
            "issuesInFirstColumnWithStatuses": 0,
            "issues": 0,
            "unmappedIssues": 4,
            "kanplanDisabledAfterOperation": false,
            "kanplanEnabledAfterOperation": false,
            "issuesInBacklog": 0,
            "kanplanEnabled": false,
            "epicsPanelEnabled": true
        });
    });

    test('kanplanEnabledAfterOperation is true if dest column is kanban column and status is first', function () {
        var columnModel = model(3, [
            column(true, false, [status(1)])
        ]);

        this.analytics.triggerStatusDnDAnalytics(columnModel, mockColumn(), mockBacklogColumn());

        this.assertTriggerDnD({
            "sourceColumnType": "column",
            "destinationColumnType": "backlog",
            "boardId": 3,
            "kanbanBoard": true,
            "kanplanForInstanceEnabled": true,
            "columns": 1,
            "issuesInFirstColumnWithStatuses": 0,
            "issues": 1,
            "unmappedIssues": 0,
            "kanplanDisabledAfterOperation": false,
            "kanplanEnabledAfterOperation": true,
            "issuesInBacklog": 1,
            "kanplanEnabled": true,
            "epicsPanelEnabled": true
        });
    });

    test('kanplanEnabledAfterOperation is false if dest column is kanban board and there is more than one status in backlog column', function () {
        var columnModel = model(3, [
            column(true, false, [status(1), status(2)])
        ]);

        this.analytics.triggerStatusDnDAnalytics(columnModel, mockColumn(), mockBacklogColumn());

        this.assertTriggerDnD({
            "sourceColumnType": "column",
            "destinationColumnType": "backlog",
            "boardId": 3,
            "kanbanBoard": true,
            "kanplanForInstanceEnabled": true,
            "columns": 1,
            "issuesInFirstColumnWithStatuses": 0,
            "issues": 3,
            "unmappedIssues": 0,
            "kanplanDisabledAfterOperation": false,
            "kanplanEnabledAfterOperation": false,
            "issuesInBacklog": 3,
            "kanplanEnabled": true,
            "epicsPanelEnabled": true
        });
    });

    test('kanplanEnabledAfterOperation and kanplanDisabledAfterOperation are false if source and dest column are kanban backlog column', function () {
        var columnModel = model(3, [
            column(true, false, [status(1)])
        ]);

        this.analytics.triggerStatusDnDAnalytics(columnModel, mockBacklogColumn(), mockBacklogColumn());

        this.assertTriggerDnD({
            "sourceColumnType": "backlog",
            "destinationColumnType": "backlog",
            "boardId": 3,
            "kanbanBoard": true,
            "kanplanForInstanceEnabled": true,
            "columns": 1,
            "issuesInFirstColumnWithStatuses": 0,
            "issues": 1,
            "unmappedIssues": 0,
            "kanplanDisabledAfterOperation": false,
            "kanplanEnabledAfterOperation": false,
            "issuesInBacklog": 1,
            "kanplanEnabled": true,
            "epicsPanelEnabled": true
        });
    });


    test('kanplanDisabledAfterOperation is false if source column is kanban backlog and status is not the last one', function () {
        var columnModel = model(3, [
            column(true, false, [status(2)]),
            column(false, true, [status(100), status(40)])
        ], [status(1)]);

        this.analytics.triggerStatusDnDAnalytics(columnModel, mockBacklogColumn(), mockUnmappedColumn());

        this.assertTriggerDnD({
            "sourceColumnType": "backlog",
            "destinationColumnType": "unmapped",
            "boardId": 3,
            "kanbanBoard": true,
            "kanplanForInstanceEnabled": true,
            "columns": 2,
            "issuesInFirstColumnWithStatuses": 140,
            "issues": 142,
            "unmappedIssues": 1,
            "kanplanDisabledAfterOperation": false,
            "kanplanEnabledAfterOperation": false,
            "issuesInBacklog": 2,
            "kanplanEnabled": true,
            "epicsPanelEnabled": true
        });
    });


    test('kanplanDisabledAfterOperation is true if source column is kanban column and status list is empty after operation', function () {
        var columnModel = model(3, [
            column(true, false),
            column(false, true, [status(1)])
        ]);

        this.analytics.triggerStatusDnDAnalytics(columnModel, mockBacklogColumn(), mockColumn());

        this.assertTriggerDnD({
            "sourceColumnType": "backlog",
            "destinationColumnType": "column",
            "boardId": 3,
            "kanbanBoard": true,
            "kanplanForInstanceEnabled": true,
            "columns": 2,
            "issuesInFirstColumnWithStatuses": 1,
            "issues": 1,
            "unmappedIssues": 0,
            "kanplanDisabledAfterOperation": true,
            "kanplanEnabledAfterOperation": false,
            "issuesInBacklog": 0,
            "kanplanEnabled": false,
            "epicsPanelEnabled": true
        });
    });

    test('kanplanDisabledAfterOperation and kanplanEnabledAfterOperation are false if board is scrum board', function () {
        setUpGlobals(false, true);

        var columnModel = model(5, [
            column(false, false),
            column(false, true, [status(1)])
        ]);

        this.analytics.triggerStatusDnDAnalytics(columnModel, mockUnmappedColumn(), mockColumn());

        this.assertTriggerDnD({
            "sourceColumnType": "unmapped",
            "destinationColumnType": "column",
            "boardId": 5,
            "kanbanBoard": false,
            "kanplanForInstanceEnabled": true,
            "columns": 2,
            "issuesInFirstColumnWithStatuses": 1,
            "issues": 1,
            "unmappedIssues": 0,
            "kanplanDisabledAfterOperation": false,
            "kanplanEnabledAfterOperation": false,
            "issuesInBacklog": 0,
            "kanplanEnabled": false,
            "epicsPanelEnabled": true
        });
    });

    test('kanplanDisabledAfterOperation and kanplanEnabledAfterOperation are false if kanplan is disabled', function () {
        setUpGlobals(true, false);

        var columnModel = model(5, [
            column(false, false),
            column(false, true, [status(10)])
        ]);

        this.analytics.triggerStatusDnDAnalytics(columnModel, mockColumn(), mockColumn());

        this.assertTriggerDnD({
            "sourceColumnType": "column",
            "destinationColumnType": "column",
            "boardId": 5,
            "kanbanBoard": true,
            "kanplanForInstanceEnabled": false,
            "columns": 2,
            "issuesInFirstColumnWithStatuses": 10,
            "issues": 10,
            "unmappedIssues": 0,
            "kanplanDisabledAfterOperation": false,
            "kanplanEnabledAfterOperation": false,
            "issuesInBacklog": 0,
            "kanplanEnabled": false,
            "epicsPanelEnabled": true
        });
    });


    test('opened - assert sample data', function () {
        var columnModel = model(3, [
            column(true, false),
            column(false, false),
            column(false, true, [status(1), status(2)]),
            column(false, false, [status(10)])
        ]);

        this.analytics.triggerTabOpenedAnalytics(columnModel, false);

        this.assertTriggerOpened({
            "kanbanBoard": true,
            "kanplanForInstanceEnabled": true,
            "boardId": 3,
            "issues": 13,
            "issuesInFirstColumnWithStatuses": 3,
            "unmappedIssues": 0,
            "columns": 4,
            "issuesInBacklog": 0,
            "canEdit": false,
            "kanplanEnabled": true,
            "epicsPanelEnabled": true
        });
    });

    test('opened event should send number of unmapped statuses', function () {
        var columnModel = model(3, [
            column(false, false)
        ], [status(4)]);

        this.analytics.triggerTabOpenedAnalytics(columnModel, true);

        this.assertTriggerOpened({
            "kanbanBoard": true,
            "kanplanForInstanceEnabled": true,
            "boardId": 3,
            "issues": 0,
            "issuesInFirstColumnWithStatuses": 0,
            "unmappedIssues": 4,
            "columns": 1,
            "issuesInBacklog": 0,
            "canEdit": true,
            "kanplanEnabled": true,
            "epicsPanelEnabled": true
        });
    });

    test('opened - globals settings - isKanbanBoard and kanplanEnabledForBoard', function () {
        setUpGlobals(true, false, true);

        var columnModel = model(5, [
            column(false, false),
            column(false, true, [status(10)])
        ]);

        this.analytics.triggerTabOpenedAnalytics(columnModel, false);

        this.assertTriggerOpened({
            "kanbanBoard": true,
            "kanplanForInstanceEnabled": false,
            "boardId": 5,
            "issues": 10,
            "issuesInFirstColumnWithStatuses": 10,
            "unmappedIssues": 0,
            "columns": 2,
            "issuesInBacklog": 0,
            "canEdit": false,
            "kanplanEnabled": true,
            "epicsPanelEnabled": true
        });
    });

    test('opened - globals settings - kanplanEnabled', function () {
        setUpGlobals(false, true, false);

        var columnModel = model(5, [
            column(true, false, [status(5), status(100)]),
            column(false, true, [status(10)])
        ]);

        this.analytics.triggerTabOpenedAnalytics(columnModel, false);

        this.assertTriggerOpened({
            "kanbanBoard": false,
            "kanplanForInstanceEnabled": true,
            "boardId": 5,
            "issues": 115,
            "issuesInFirstColumnWithStatuses": 10,
            "unmappedIssues": 0,
            "columns": 2,
            "issuesInBacklog": 105,
            "canEdit": false,
            "kanplanEnabled": false,
            "epicsPanelEnabled": true
        });
    });
});
