define("jira-agile/rapid/configuration/manage-boards-model", [
        "jira/util/formatter",
        "jira-agile/rapid/events",
        "jquery",
        "underscore"
    ],
    function (formatter, Events, $, _) {
        "use strict";

        var recentBoardsCategory = {name: formatter.I18n.getText('gh.rapid.manage.boards.recent'), id: 'recent'};
        var allBoardsCategory = {name: formatter.I18n.getText('gh.rapid.manage.boards.all'), id: 'all'};


        function ManageBoardsModel(options) {
            this.textFilter = '';
            this.typeFilter = [];
            this.recentBoards = [];
            this.allBoards = [];
            this.populated = false;
            this.boards = new GH.Pagination.PageableCollection([], options);
            this.boards.on('page-change', function () {
                this.trigger('change:boards');
            }.bind(this));
            this.filterAttributes = ['textFilter', 'typeFilter'];
            this.categories = [allBoardsCategory];
            this.currentCategory = 'all';
        }

        ManageBoardsModel.prototype = new Events();

        _.extend(ManageBoardsModel.prototype, {

            /**
             * Sets the recent boards in the model
             *
             * @param recentBoards Array of boards recently visited by user
             */
            setRecentBoards: function (recentBoards) {
                this.recentBoards = recentBoards;
                _.each(this.recentBoards, function (board) {
                    var boardAdmins = board.boardAdmins;
                    board.renderableBoardAdmins = GH.RapidBoard.BoardAdminsPicker.getOrderedUnion(boardAdmins);
                });
                if (this.recentBoards.length) {
                    this.categories = [recentBoardsCategory, allBoardsCategory];
                }
                else {
                    this.categories = [allBoardsCategory];
                }
                this.currentCategory = 'all';
                this.trigger('change:boards');
            },

            /**
             * Returns the recent boards. Note: we don't show recent boards when there is a text filter
             *
             * @returns Array recentBoards
             */
            getRecentBoards: function () {
                return this.recentBoards;
            },

            /**
             * Populate the model with boards fetched from the server
             *
             * @param boards array of boards
             */
            populate: function populate(boards) {
                boards = boards || [];
                _.each(boards, function (board) {
                    var boardAdmins = board.boardAdmins;
                    board.renderableBoardAdmins = GH.RapidBoard.BoardAdminsPicker.getOrderedUnion(boardAdmins);
                });
                this.boards.setCollection(boards);
                this.allBoards = boards;
                this.populated = true;
                this.trigger('change:boards');
            },

            /**
             * Get boards that should be displayed
             *
             * @returns Array of boards
             */
            getAllBoards: function getAllBoards() {
                return this.boards.getPaginated();
            },

            getBoardsCategories: function getBoardsCategories() {
                return this.categories;
            },

            setCurrentCategory: function setCurrentCategory(categoryId) {
                this.currentCategory = categoryId;
                this.trigger('change:boards');
            },

            getCurrentCategory: function getCurrentCategory() {
                var currentCat = this.currentCategory;
                return _.find(this.categories, function (cat) {
                    return cat.id === currentCat;
                });
            },

            getCurrentBoards: function getCurrentBoards() {
                if (this.currentCategory === 'recent') {
                    return this.getRecentBoards();
                }
                return this.getAllBoards();
            },

            isFilterable: function isFilterable() {
                return this.currentCategory !== 'recent' && this.populated === true;
            },

            isPopulated: function isPopulated() {
                return this.populated;
            },

            setTextFilter: function setTextFilter(filter) {
                var previousFilter = this.textFilter;
                this.textFilter = filter.toLowerCase();
            },

            clearTextFilter: function clearTextFilter() {
                this.textFilter = '';
            },

            setTypeFilter: function setTypeFilter(filteredTypes) {
                this.typeFilter = filteredTypes;
            },

            /**
             * Perform board filtering, triggers event when board lsit have changed
             */
            filterBoards: function filterBoards() {
                var filtered = this.allBoards;
                var textFilter = this.textFilter || '';
                var typeFilter = this.typeFilter.length === 1 ? this.typeFilter[0] : '';
                var adminPredicate = function (admin) {
                    return (admin.key || '').toLowerCase().indexOf(textFilter) !== -1;
                };
                var filterPredicate = function (item) {
                    return item.name.toLowerCase().indexOf(textFilter) !== -1 ||
                        _.any(item.renderableBoardAdmins, adminPredicate);
                };

                if (textFilter) {
                    filtered = filtered.filter(filterPredicate);
                }

                if (typeFilter) {
                    var sprintSupportEnabled = typeFilter === 'scrum';
                    filtered = filtered.filter(function (board) {
                        return sprintSupportEnabled === board.sprintSupportEnabled;
                    });
                }

                if (filtered.length !== this.boards.state.collectionLength) {
                    this.boards.setCollection(filtered);
                    this.trigger('change:boards');
                }
            },

            getPaginationState: function getPaginationState() {
                if (this.currentCategory === 'recent') {
                    //no pagination on recent boards
                    return {
                        totalPages: 0,
                        firstPage: 0,
                        lastPage: 0
                    };
                }
                else {
                    var state = _.clone(this.boards.state);
                    //we show five pages before and after current page:
                    state.firstPage = Math.max(state.currentPage - 5, state.firstPage);
                    state.totalPages = state.lastPage;
                    state.lastPage = Math.min(state.currentPage + 5, state.lastPage);
                    return state;
                }
            },

            getFilterState: function getFilterState() {
                var state = {};
                _.each(this.filterAttributes, function (attr) {
                    state[attr] = this[attr];
                }.bind(this));
                return state;
            },

            isEmpty: function isEmpty() {
                return this.populated && this.allBoards.length === 0;
            },

            changePage: function changePage(page) {
                this.boards.setPage(page);
            },

            removeBoard: function removeBoard(boardId) {
                boardId = +boardId;
                var predicate = function (board) {
                    return board.id !== boardId;
                };
                this.recentBoards = this.recentBoards.filter(predicate);
                if (this.recentBoards.length === 0) {
                    this.categories = [allBoardsCategory];
                    this.currentCategory = 'all';
                    this.trigger('change:categories');
                }
                this.allBoards = this.allBoards.filter(predicate);
                this.filterBoards();
            }
        });

        return ManageBoardsModel;
    });

AJS.namespace('GH.ManageBoards.ManageBoardsModel', null, require("jira-agile/rapid/configuration/manage-boards-model"));