define('jira-agile/rapid/configuration/manage-boards-view', ["jira-agile/rapid/user-data", "jira-agile/rapid/ui/notification", "jira/util/formatter", "underscore", "jquery", "jira-agile/rapid/events", "jira-agile/gh-shortener"], function (UserData, Notification, formatter, _, $, Events, Shortener) {

    function ManageBoardsView() {}

    ManageBoardsView.prototype = new Events();

    _.extend(ManageBoardsView.prototype, {
        /**
         * Initializes the view.
         */
        init: function init(model) {
            this.model = model;
            this._initHeaderPanel();
            this._initContentPanel();
            this._assignEventHandlers();

            this.model.on('change:boards', this.render, this);
            this.model.on('change:categories', this.renderTabs, this);

            // touch detection
            if ($.support.touch) {
                $('#gh').removeClass('ghx-no-touch').addClass('ghx-touch');
            }
        },

        _initHeaderPanel: function _initHeaderPanel() {
            this.$headerPanel = $('#ghx-header');
            this.$headerPanel.html(GH.tpl.rapid.manageboards.renderHeaderPanelContent());
            this.$headerActions = this.$headerPanel.find('.aui-buttons');
        },

        _initContentPanel: function _initContentPanel() {
            var categories = this.model.getBoardsCategories() || [];
            this.$contentPanel = $('#manage-boards-page').find('.aui-page-panel-inner').html(GH.tpl.rapid.manageboards.renderContentPanel({
                categories: categories
            }));
            this.$boardsContainer = this.$contentPanel.find('#boards-table');
            this.$categoryName = this.$contentPanel.find('.ghx-boards-list h2');
            this.$filterForm = this.$contentPanel.find('.filter-form');
            this.$paginationContainer = this.$contentPanel.find('.pagination');
            this.$navigationContainer = this.$contentPanel.find('.aui-page-panel-nav');

            //hide nav sidebar whem there are less than two categories
            if (categories.length < 2) {
                this.$contentPanel.find('.aui-page-panel-nav').addClass('hidden');
            }

            this.$filterForm.html(GH.tpl.rapid.manageboards.filterForm(this.model.getFilterState()));

            this.$searchBoardsInput = this.$contentPanel.find('.js-search-boards-input');
            this.$searchBoardsInputIcon = this.$contentPanel.find('.js-search-page-icon');
            this.$typeFilterTrigger = this.$contentPanel.find('.aui-dropdown2-trigger');
            this.$typeFilter = this.$contentPanel.find('.aui-dropdown2 a');

            this.$navigationContainer.find('a[data-item-id="recent"]').on('click', function (event) {
                AJS.trigger('analyticsEvent', { name: 'jira-software.board.boardmenu.click.recently-visited-boards-tab' });
            });
            this.$navigationContainer.find('a[data-item-id="all"]').on('click', function (event) {
                AJS.trigger('analyticsEvent', { name: 'jira-software.board.boardmenu.click.all-boards-tab' });
            });
        },

        _assignEventHandlers: function _assignEventHandlers() {
            // Initialize the create board dialog
            GH.Dialog.CreateView.init();

            this.$searchBoardsInput.on('input', this.filterInput.bind(this));

            this.$searchBoardsInputIcon.click(function () {
                this.$searchBoardsInput.val('');
                this.$searchBoardsInputIcon.addClass('aui-iconfont-search-small').removeClass('aui-iconfont-remove');
                this.trigger('filter:text', '');
            }.bind(this));

            var view = this;
            this.$typeFilter.click(function (e) {
                var $checkbox = $(e.currentTarget).find('input');
                $checkbox.prop('checked', !$checkbox.prop('checked'));
                view.trigger('filter:type', view._getCurrentType());
                view._setTriggerContent();
            });

            // handle delete dialog success
            $(GH).bind('ghx.rapidview.deleted', function (event, data) {
                Notification.showSuccess(data.message);

                // remove boards form collection

                this.model.removeBoard(data.rapidViewId);
            }.bind(this));

            this.$paginationContainer.on('click', 'li a', function (event) {
                event.preventDefault();
                var page = $(event.currentTarget).data('page');
                this.trigger('change:page', page);
            }.bind(this));

            this.$navigationContainer.on('click', 'li a', function (event) {
                event.preventDefault();
                var $link = $(event.currentTarget);
                $link.blur();
                var categoryId = $link.data('itemId');
                if (categoryId !== this.model.getCurrentCategory().id) {
                    this.trigger('filter:category', categoryId);
                }
            }.bind(this));

            this.$contentPanel.find('.filter-form').on('submit', function (event) {
                //prevent form submit
                event.preventDefault();
            });
        },

        _assignBoardActionHandlers: function _assignBoardActionHandlers() {
            this.$boardsContainer.find('.js-copy-board-action').on('click', function (event) {
                event.preventDefault();
                var boardId = $(this).attr('data-board-id');
                var boardName = $(this).attr('data-board-name');
                GH.ViewActions.actionCopy(boardId, boardName);
                AJS.trigger('analyticsEvent', { name: 'jira-software.manage.operationsmenu.click.copy-board' });
            });

            this.$boardsContainer.find('.js-delete-board-action').on('click', function (event) {
                event.preventDefault();
                var rapidViewId = $(this).attr('data-board-id');
                var rapidViewName = $(this).attr('data-board-name');
                GH.RapidBoard.DeleteDialog.showDeleteDialog(rapidViewId, rapidViewName);
                AJS.trigger('analyticsEvent', { name: 'jira-software.manage.operationsmenu.click.delete-board' });
            });

            this.$boardsContainer.find('.js-edit-view').on('click', function (event) {
                AJS.trigger('analyticsEvent', { name: 'jira-software.manage.operationsmenu.click.configure-board' });
            });

            this.$boardsContainer.find('tr[data-board-id]').on('click', function (event) {
                AJS.trigger('analyticsEvent', {
                    name: 'jira-software.board.boardtable.click.board-name',
                    data: { boardId: $(event.delegateTarget).attr('data-board-id') }
                });
            });
        },

        _getCurrentType: function _getCurrentType() {
            var result = [];
            this.$typeFilter.find('input:checked').each(function () {
                result.push($(this).val());
            });
            return result;
        },

        _setTriggerContent: function _setTriggerContent() {
            var currentFilter = '';
            if (this.model.typeFilter.length !== 1) {
                currentFilter = formatter.I18n.getText('common.words.all');
            } else if (this.model.typeFilter[0] === 'kanban') {
                currentFilter = formatter.I18n.getText('gh.agile.kanban');
            } else {
                currentFilter = formatter.I18n.getText('gh.agile.scrum');
            }
            this.$typeFilterTrigger.text(formatter.I18n.getText('gh.rapid.view.list.filter.type', currentFilter));
        },

        filterInput: _.debounce(function filterInput() {
            var searchTerm = this.$searchBoardsInput.val();
            if (searchTerm != null && searchTerm.length > 0) {
                // change the icon of the input to remove icon
                this.$searchBoardsInputIcon.removeClass('aui-iconfont-search-small').addClass('aui-iconfont-remove');
            } else {
                // change the icon of the input to search icon
                this.$searchBoardsInputIcon.addClass('aui-iconfont-search-small').removeClass('aui-iconfont-remove');
            }
            this.trigger('filter:text', searchTerm);
        }, 300),

        showSpinner: function showSpinner() {
            this.$contentPanel.find('#ghx-page-loading-spinner').spin();
        },

        hideSpinner: function hideSpinner() {
            this.$contentPanel.find('#ghx-page-loading-spinner').spinStop();
        },

        renderTabs: function renderTabs() {
            var categories = this.model.getBoardsCategories() || [];
            this.$navigationContainer.html(GH.tpl.rapid.manageboards.tabs({
                items: categories
            }));
            //hide nav sidebar whem there are less than two categories
            if (categories.length < 2) {
                this.$navigationContainer.addClass('hidden');
            }
        },

        render: function render() {
            if (this.model.isEmpty()) {
                this.renderEmptyView();
            } else {
                this.renderBoards();
            }
        },

        renderEmptyView: function renderEmptyView() {
            $('#ghx-content-main').html(GH.tpl.board.x.renderNoViews());
        },

        renderBoards: function renderBoards() {
            var category = this.model.getCurrentCategory();
            var boards = this.model.getCurrentBoards();

            this.$categoryName.text(category.name);
            if (this.model.isFilterable()) {
                this.$filterForm.removeClass('hidden');
            } else {
                this.$filterForm.addClass('hidden');
            }

            if (boards.length === 0 && this.model.isPopulated() === false) {
                this.$boardsContainer.html(GH.tpl.rapid.manageboards.renderPageLoadingSpinner());
                this.showSpinner();
            } else {
                this.$boardsContainer.html(GH.tpl.rapid.manageboards.renderBoardsTable({
                    category: category.id,
                    canCopy: UserData.hasUser(),
                    boards: boards
                }));

                _.each(boards, function (board) {
                    board.shortener = new Shortener({
                        items: 'span.ghx-board-admin-list-item',
                        element: '.board-admins-list-' + board.id + '-' + category.id,
                        numRows: 1
                    });
                });

                this.$paginationContainer.html(GH.tpl.rapid.manageboards.pagination(this.model.getPaginationState()));

                this._assignBoardActionHandlers();
            }
            var $categoryEls = this.$navigationContainer.find('li');
            $categoryEls.removeClass('aui-nav-selected');
            $categoryEls.children('a[data-item-id=' + category.id + ']').parent().addClass('aui-nav-selected');
        },

        renderBoardsActions: function renderBoardsActions() {
            this.$headerActions.html(GH.tpl.rapid.manageboards.headerActions());

            this.$headerActions.find('.js-view-action-create').on('click', function (event) {
                AJS.trigger('analyticsEvent', { name: 'jira-software.manage.boardmenu.click.create-board' });
            });
        }
    });

    return ManageBoardsView;
});

AJS.namespace('GH.ManageBoards.ManageBoardsView', null, require('jira-agile/rapid/configuration/manage-boards-view'));