/**
 * Allows callbacks to be registered/executed as a result of specific events.
 *
 * This bus executes everything synchronously, and thus allows registering
 * callbacks without having to tightly couple and/or passing through the callback through several layers
 *
 */

GH.CallbackManager = {};

/**
 * Holds the current callbacks to execute
 */
GH.CallbackManager.callbacks = {};

GH.CallbackManager.executeCallbacks = function(namespace, data) {
    GH.Logger.log('Executing callbacks for ' + namespace, GH.Logger.Contexts.callback);
    var callbacks = GH.CallbackManager.callbacks[namespace];
    if (callbacks && !_.isEmpty(callbacks)) {
        _.each(callbacks, function(callback) {
            GH.Logger.log('Executing callback ' + callback.name, GH.Logger.Contexts.callback);
            callback.fn(data);
        });
        delete GH.CallbackManager.callbacks[namespace];
    }
    GH.Logger.log('Finished callbacks for ' + namespace, GH.Logger.Contexts.callback);
};

/**
 * Register a callback with a given name
 */
GH.CallbackManager.registerCallback = function(namespace, name, callbackFn) {
    GH.Logger.log('Registering callback ' + name + ' with namespace ' + namespace);
    var callbacks = GH.CallbackManager.callbacks[namespace];
    if (!callbacks) {
        callbacks = GH.CallbackManager.callbacks[namespace] = [];
    }
    callbacks.push({
        name: name,
        fn: callbackFn
    });
};

/**
 * Delete all callbacks for a given namespace PREFIX, e.g. rapid.work.detail would be deleted with rapid.work
 * @param namespace
 */
GH.CallbackManager.deleteCallbacks = function(namespace) {
    _.each(_.keys(GH.CallbackManager.callbacks), function(key) {
        if (key.substr(0, namespace.length) == namespace) {
            delete GH.CallbackManager.callbacks[key];
        }
    });
};

/**
 * Get all callbacks that match a given namespace
 */
GH.CallbackManager.getCallbacks = function(namespace) {
    return GH.CallbackManager.callbacks[namespace] || [];
};

/**
 * Clears all callbacks
 */
GH.CallbackManager.clear = function() {
    GH.CallbackManager.callbacks = {};
};
