/**
 * Manages the url state for the rapid board
 */
define("jira-agile/rapid/state/rapid-board-url-state", [
    "jira-agile/rapid/global-events",
    "underscore"
], function (GlobalEvents, _) {
    "use strict";

    let PlanController;

    GlobalEvents.on("pre-initialization", function () {
        PlanController = require("jira-agile/rapid/ui/plan/plan-controller");
    });

    const RapidBoardUrlState = {};

    RapidBoardUrlState.self = "RapidBoardUrlState";
    /**
     * Is this state applicable for the given normalized state
     * @param normalized
     */
    RapidBoardUrlState.isApplicable = function (normalized) {
        return true;
    };

    /**
     * Normalizes the url state for the plan specific values.
     * @param urlParams the parsed url parameters
     * @param normalized normalized parameters
     */
    RapidBoardUrlState.getNormalizedFromUrl = function (urlParams, normalized) {
        // rapid view id
        normalized['rapidView'] = GH.RapidBoard.UrlState.normalizeNumberParam(urlParams['rapidView']); // null by default

        delete normalized['projectKey'];
        var projectKey = GH.RapidBoard.UrlState.normalizeStringParam(urlParams['projectKey'], null);
        if (projectKey !== null) {
            normalized['projectKey'] = projectKey;
        }

        // if the URL contains a flag saying 'useStoredSettings' then we don't want to override the mode (as we do below), so
        // fail fast.
        // NOTE: this only allows us to specify the view, and does not help in the case where we want to specify some settings,
        // but want to inherit all other stored settings.
        // tracked in GHS-4732
        var tmpMode = GH.RapidBoard.UrlState.normalizeStringParam(urlParams['view'], ''); // empty string by default
        var useStoredSettings = GH.RapidBoard.UrlState.normalizeBooleanParam(urlParams['useStoredSettings'], false);
        if (useStoredSettings && tmpMode === '') {
            return;
        }

        // Perform KanPlan check.
        // If using is browsing KanPlan and the KanPlan is disable, redirect user to work mode
        if (GH.RapidBoard.State.isLoadingKanPlanWhileItDisabled(tmpMode)) {
            tmpMode = 'detail';
        }

        // view, or mode, whatever...
        switch (tmpMode) {
            case 'reporting':
                normalized['mode'] = 'report';
                break;
            case 'planning':
                normalized['mode'] = 'plan';
                normalized['detailOpened'] = true;
                break;
            case 'planning.nodetail':
                normalized['mode'] = 'plan';
                normalized['detailOpened'] = false;
                break;
            case 'detail':
                normalized['mode'] = 'work';
                normalized['detailOpened'] = true;
                break;
            case '':
            /* falls through */
            default:
                // fall back to work if we don't know
                normalized['mode'] = 'work';
                normalized['detailOpened'] = false;
        }
    };

    /**
     * Transforms the normalized url parameters into actual url parameters
     * @param normalized the normalized params
     * @param urlParams the url params
     */
    RapidBoardUrlState.toUrl = function (normalized, urlParams) {
        // first the view id
        if (!_.isUndefined(normalized['rapidView'])) {
            urlParams['rapidView'] = normalized['rapidView'];
        }
        delete urlParams['projectKey'];
        if (!_.isUndefined(normalized['projectKey'])) {
            urlParams['projectKey'] = normalized['projectKey'];
        }

        // then the mode
        switch (normalized['mode']) {
            case 'work':
                if (normalized['detailOpened']) {
                    urlParams['view'] = 'detail';
                }
                break;
            case 'report':
                urlParams['view'] = 'reporting';
                break;
            case 'plan':
                if (!normalized['detailOpened']) {
                    urlParams['view'] = 'planning.nodetail';
                } else {
                    urlParams['view'] = 'planning';
                }
                break;
        }
    };

    /**
     * Get the normalized state from the app
     * @param normalized the normalized state to put the values into
     */
    RapidBoardUrlState.getNormalizedFromInternal = function (normalized) {
        // rapid view id
        normalized['rapidView'] = GH.RapidBoard.State.getRapidViewId();

        // selected mode
        normalized['mode'] = GH.RapidBoard.State.getMode();

        delete normalized['projectKey'];
        var projectKey = GH.RapidBoard.State.getProjectKey();
        if (!_.isUndefined(projectKey)) {
            normalized['projectKey'] = projectKey;
        }

        // detail view
        if (normalized['mode'] === 'work') {
            normalized['detailOpened'] = GH.RapidBoard.State.isDetailsViewOpened();
        } else if (normalized['mode'] === 'plan') {
            normalized['detailOpened'] = GH.RapidBoard.State.isDetailsViewPlanOpened();
        }

    };

    /**
     * Update the app with the values stored in the normalized state
     * @param normalized the normalized state to update the values from
     */
    RapidBoardUrlState.updateInternalFromNormalized = function (normalized) {
        if (!_.isUndefined(normalized['rapidView'])) {
            GH.RapidBoard.State.setRapidViewId(normalized['rapidView']);
        }

        var mode = normalized['mode'];
        var detailOpened = normalized['detailOpened'];

        if (!_.isUndefined(mode)) {
            GH.RapidBoard.State.setMode(normalized['mode']);
        }
        GH.RapidBoard.State.setProjectKey(normalized['projectKey']);

        if (!_.isUndefined(detailOpened)) {
            if (mode === "plan") {
                PlanController.setDetailViewOpenedState(normalized['detailOpened']);
            } else {
                GH.WorkController.setDetailViewOpenedState(normalized['detailOpened']);
            }
        }

    };

    return RapidBoardUrlState;
});

AJS.namespace("GH.RapidBoardUrlState", null, require("jira-agile/rapid/state/rapid-board-url-state"));