/**
 * Burndown Timeline Producer
 * @module jira-agile/rapid/ui/chart/burndown-timeline-producer
 * @requires module:underscore
 */
define('jira-agile/rapid/ui/chart/burndown-timeline-producer', ['require'], function(require) {
    'use strict';

    var _ = require('underscore');

    /**
     * Produces a timeline out of the raw data of issue changes
     *
     * This involves taking the raw event-based input and producing a more complete structure of change items, while
     * preserving all information required to explain the origin of each change
     *
     * Note: sprint and column status are handled by the producer itself, but a statisticConsumer is used to handle
     * statistic specific timeline changes
     */
    var BurndownTimelineProducer = {};

    /**
     * Create a timeline out of the raw data. The output will contain for each date at most one entry per issue
     * describing the change as well as the complete issue state at given point in time.
     */
    BurndownTimelineProducer.calculateTimelineData = function (rawData, statisticConsumer) {
        var data = {
            timeline: []
        };

        // we can only proceed if we got a start and end date that differ
        if (rawData.startTime === rawData.endTime) {
            return data;
        }

        var completeTime = _.isUndefined(rawData.completeTime) ? false : rawData.completeTime;
        var calculatedData = BurndownTimelineProducer.calculateTimeline(rawData.startTime, completeTime, rawData.changes,
            rawData.openCloseChanges, statisticConsumer, rawData.estimatableIssueKeys);

        // set all the data
        data.timeline = calculatedData.timeline;
        data.now = rawData.now;
        data.startTime = rawData.startTime;
        data.endTime = rawData.endTime;
        data.completeTime = completeTime;
        data.maxValues = calculatedData.maxValues;
        data.statisticField = rawData.statisticField;
        data.finalIssueStates = calculatedData.issueState;

        return data;
    };

    /**
     * Calculates the sprint timeline for a given start and end time as well as a list of changes
     */
    BurndownTimelineProducer.calculateTimeline = function (sprintStartTime, sprintCompleteTime, changes, openCloseChanges,
                                                           statisticConsumer, estimatableIssueKeys) {
        // data carrying object
        var data = {
            issueState: {},
            timeline: [],
            currentValues: {},
            maxValues: {},
            pastSprintStart: false,
            statisticConsumer: statisticConsumer,
            estimatableIssueKeys: estimatableIssueKeys
        };

        // initialize the current values to 0 for all statistics
        BurndownTimelineProducer.addStatisticValueDeltas({}, data.currentValues, 1, statisticConsumer);

        // process all changes
        var changeDates = _.keys(changes);

        var stateChangeKeys = openCloseChanges ? _.keys(openCloseChanges) : [];
        var currentStateChangeIndex = 0;

        for (var i = 0; i < changeDates.length; i++) {
            // convert current date
            var key = changeDates[i];
            var currentMillis = parseInt(key, 10);

            // did we pass startTime? if so the currently recorded values will be the initial values
            if (!data.pastSprintStart && sprintStartTime <= currentMillis) {
                // create a start value
                BurndownTimelineProducer.addSprintStartTimelineEntry(data, sprintStartTime);
                data.pastSprintStart = true;
            }

            // stop if we are past the sprint complete time
            if (sprintCompleteTime && (sprintCompleteTime < currentMillis)) {
                break;
            }

            // process all changes
            var changeItems = changes[key];
            var changeData = BurndownTimelineProducer.processTimeChanges(data, changeItems, currentMillis);

            // add an entry if we are past sprint start
            if (data.pastSprintStart && changeData) {

                // add rows for any sprint state change events that happened before the above changes
                if (stateChangeKeys.length > 0) {
                    var currentStateChangeKey = stateChangeKeys[currentStateChangeIndex];

                    while (currentStateChangeKey < key) {
                        var stateChangeMillis = parseInt(currentStateChangeKey, 10);
                        var instances = openCloseChanges[currentStateChangeKey];

                        BurndownTimelineProducer.addSprintStateChangeTimelineEntry(data, stateChangeMillis, instances);

                        currentStateChangeIndex++;
                        currentStateChangeKey = stateChangeKeys[currentStateChangeIndex];
                    }
                }

                BurndownTimelineProducer.addChangeTimelineEntry(data, currentMillis, changeData.issues, changeData.deltas);
            }
        }

        // ensure we processed the start
        if (!data.pastSprintStart) {
            BurndownTimelineProducer.addSprintStartTimelineEntry(data, sprintStartTime);
            data.pastSprintStart = true;
        }

        // add complete point if sprint has completed
        if (sprintCompleteTime) {
            BurndownTimelineProducer.addSprintCompleteTimelineEntry(data, sprintCompleteTime);
        }

        return data;
    };

    /**
     * Processes all change items for a given changeDate.
     * Each change is applied to the corresponding issue
     */
    BurndownTimelineProducer.processTimeChanges = function (data, changeItems, changeDate) {
        var deltas = {};
        var validIssues = [];

        _.each(changeItems, function (change) {
            // keeps the deltas for this single change
            var localDeltas = {};

            // fetch issue buffer
            var issueData = data.issueState[change.key];
            if (!issueData) {
                issueData = {
                    key: change.key,
                    isInChart: false, // true if in column and sprint
                    burned: false, // true if the value has been burned (thus issue is in last column)
                    values: {}, // values stored against this issue
                    estimatable: _.contains(data.estimatableIssueKeys, change.key) // true if issue is estimatable
                };
                data.issueState[change.key] = issueData;
            }

            // remove old counted values
            BurndownTimelineProducer.removeStatisticValues(issueData, localDeltas, data.statisticConsumer);
            BurndownTimelineProducer.removeStatisticValues(issueData, deltas, data.statisticConsumer);

            // give the consumer a chance to do some preprocessing work
            data.statisticConsumer.beforeProcessChange(issueData, changeDate, change);

            // was this issue in the chart previously? Will be used to evaluate whether the change qualifies to be in the timeline
            var wasInChart = issueData.isInChart || false;

            // update inChart
            BurndownTimelineProducer.processInChartChanges(issueData, change);

            // handle the statistic value changes
            data.statisticConsumer.processChange(issueData, change);

            // add new values
            BurndownTimelineProducer.addStatisticValues(issueData, localDeltas, data.statisticConsumer);
            BurndownTimelineProducer.addStatisticValues(issueData, deltas, data.statisticConsumer);

            // if the change was or is in the chart we can create a data entry for this change
            if (wasInChart || issueData.isInChart) {
                var state = BurndownTimelineProducer.getCurrentState(data, issueData, localDeltas);
                state.change = change; // retain this change (though probably not required anymore
                validIssues.push(state);
            }

            // give the consumer a chance to do postprocessing work
            data.statisticConsumer.afterProcessChange(issueData, changeDate, change);
        });

        // return if we got no valid issues
        if (_.isEmpty(validIssues)) { return false; }

        // update the summed up values in data
        BurndownTimelineProducer.addStatisticValueDeltas(deltas, data.currentValues, 1, data.statisticConsumer);

        // return the issues that actually changes as well as the overall deltas
        return {
            issues: validIssues,
            deltas: deltas
        };
    };

    /**
     * Processes sprint/column changes
     */
    BurndownTimelineProducer.processInChartChanges = function (issueData, change) {
        // process the current state of the issueData. Is the issue currently in the sprint and a column?
        if (!_.isUndefined(change.added)) {
            issueData.inScope = change.added;
            issueData.scopeChange = true;
        } else {
            issueData.scopeChange = false;
        }

        // which column
        if (change.column) {
            // if there is a field set to false, set the opposite field to true
            // Note that we test for strict equality here, since undefined fields are also false.
            if (change.column.done === false) {
                change.column.notDone = true;
            }

            if (change.column.notDone === false) {
                change.column.done = true;
            }

            // set new column as well as calculate the old column
            // TODO: we should directly ship this from the server in that format!
            if (change.column.notDone) {
                issueData.column = 'notDone';
                if (_.isBoolean(change.column.done)) {
                    issueData.oldColumn = 'done';
                } else {
                    issueData.oldColumn = undefined;
                }
            } else if (change.column.done) {
                issueData.column = 'done';
                if (_.isBoolean(change.column.notDone)) {
                    issueData.oldColumn = 'notDone';
                } else {
                    issueData.oldColumn = undefined;
                }
            } else {
                issueData.column = undefined;
                if (_.isBoolean(change.column.notDone)) {
                    issueData.oldColumn = 'notDone';
                } else {
                    issueData.oldColumn = 'done';
                }
            }

            issueData.columnChange = true;
        } else {
            issueData.columnChange = false;
        }

        // is this issue now in the chart?
        issueData.isInChart = (issueData.column && issueData.inScope) ? true : false;

        // if we are in the done column the issue has been "burned"
        issueData.burned = (issueData.column === 'done');
    };


//
// Statistic value sum and delta calculations, uses statisticConsumer to do actual mapping
//

    /**
     * Adds the statistics value contained in issueData to the values.
     */
    BurndownTimelineProducer.addStatisticValues = function (issueData, values, statisticConsumer) {
        if (issueData.isInChart) {
            BurndownTimelineProducer.addStatisticValueDeltas(issueData.values, values, 1, statisticConsumer);
        }
    };

    /**
     * Removes the statistics value contained in issueData from the values.
     */
    BurndownTimelineProducer.removeStatisticValues = function (issueData, values, statisticConsumer) {
        if (issueData.isInChart) {
            BurndownTimelineProducer.addStatisticValueDeltas(issueData.values, values, -1, statisticConsumer);
        }
    };

    BurndownTimelineProducer.addStatisticValueDeltas = function (from, to, factor, statisticConsumer) {
        if (!factor) { factor = 1; }
        _.each(statisticConsumer.dataFields, function (key) {
            to[key] = (to[key] || 0) + (from[key] || 0) * factor;
        });
    };

    BurndownTimelineProducer.getMaxStatisticValues = function (a, b, statisticConsumer) {
        var res = {};
        _.each(statisticConsumer.dataFields, function (key) {
            res[key] = a[key] > b[key] ? a[key] : b[key];
        });
        return res;
    };


//
// Timeline entry generation
//

    /**
     * Add a change timeline entry
     */
    BurndownTimelineProducer.addChangeTimelineEntry = function (data, time, issues, deltas) {
        var timelineEntry = {
            time: time,
            issues: issues,
            values: _.clone(data.currentValues),
            deltas: deltas
        };
        data.timeline.push(timelineEntry);

        // update the max values according to the current values
        data.maxValues = BurndownTimelineProducer.getMaxStatisticValues(data.maxValues, data.currentValues, data.statisticConsumer);
    };

    /**
     * Add a change timeline entry
     */
    BurndownTimelineProducer.addSprintStateChangeTimelineEntry = function (data, time, instances) {
        var timelineEntry = {
            time: time,
            issues: {},
            values: {},
            deltas: {},
            openCloseInstances: instances
        };
        data.timeline.push(timelineEntry);
    };

    /**
     * Add a sprint start timeline entry
     */
    BurndownTimelineProducer.addSprintStartTimelineEntry = function (data, time) {
        // give the consumer a chance to reset data
        data.statisticConsumer.beforeSprintStart(data);

        // create an entry for the current state
        var entry = BurndownTimelineProducer.getTimelineEntryForCurrentState(data, time);
        entry.initial = true;
        data.timeline.push(entry);

        // also set the max values
        data.maxValues = _.clone(data.currentValues);
    };

    /**
     * Add a sprint complete timeline entry
     */
    BurndownTimelineProducer.addSprintCompleteTimelineEntry = function (data, time) {
        var entry = BurndownTimelineProducer.getTimelineEntryForCurrentState(data, time);
        entry.complete = true;
        data.timeline.push(entry);
    };

    /**
     * Creates a timeline entry for the current state of all issues. Used to create sprint start/complete entries
     */
    BurndownTimelineProducer.getTimelineEntryForCurrentState = function (data, time) {
        // put in the current status for accounted issues
        var issues = [];
        _.each(data.issueState, function (issue) {
            if (issue.isInChart) {
                issues.push(BurndownTimelineProducer.getCurrentState(data, issue));
            }
        });

        // setup a data point
        var timelineEntry = {
            time: time,
            issues: issues,
            values: _.clone(data.currentValues)
        };
        return timelineEntry;
    };

    /**
     * Get the current state of an issue. Copies over all from the internal buffer into a new object (as the internal
     * buffer changes as we go along in the timeline)
     */
    BurndownTimelineProducer.getCurrentState = function (data, issueData, deltas) {
        var currentState = _.clone(issueData);
        currentState.values = _.clone(issueData.values);
        if (deltas) {
            currentState.deltas = deltas;

            // statistic consumer might want to copy more stuff
            data.statisticConsumer.copyToCurrentState(currentState, issueData);
        }
        return currentState;
    };


    /**
     * Consumer for time tracking changes
     */
    BurndownTimelineProducer.TimeTrackingStatisticConsumer = {
        /** Data fields set by this consumer. Fields are automatically included in deltas and rollups */
        dataFields: ['estimate', 'timeSpent'],

        /** Processes the change for a given issue */
        processChange: function (issueData, change) {
            // values stored in timeC
            if (change.timeC) {

                // process time spent if we got a value
                if (!_.isUndefined(change.timeC.timeSpent)) {
                    // only accept timeSpent if we are currently in-sprint
                    if (issueData.isInChart) {
                        issueData.values.timeSpent = (issueData.values.timeSpent || 0) + (change.timeC.timeSpent || 0);
                        issueData.timeChange = true;
                    }
                } else {
                    issueData.timeChange = false;
                }

                // process remaining estimate change
                if (!_.isUndefined(change.timeC.newEstimate)) {
                    // update the estimate, taking the estimate difference between the old and new estimate
                    // We do this as the actual estimate value is wrong if this is a backdated estimate.
                    // Note that the chart calculation itself therefore doesn't actually need to keep track of backdated vs non-backdated
                    // estimate changes, but the data table explaining what is happening is.
                    issueData.values.estimate = (issueData.values.estimate || 0) + ((change.timeC.newEstimate || 0) - (change.timeC.oldEstimate || 0));
                    issueData.estimateChange = true;
                } else {
                    issueData.estimateChange = false;
                }
            }
        },

        /** Called before changes get processed */
        beforeProcessChange: function (issueData, changeDate, change) {
            // clean up outdated estimate changes
            BurndownTimelineProducer.TimeTrackingStatisticConsumer.removeExpiredEstimateChanges(issueData, changeDate);
        },

        /** Called after a change gets processed */
        afterProcessChange: function (issueData, changeDate, change) {
            // add the current change as estimate change (will be removed as soon as outdated)
            // we do this after copying the current state of the issue, as the change should only affect later changes
            BurndownTimelineProducer.TimeTrackingStatisticConsumer.addEstimateChange(issueData, changeDate, change);
        },

        /** Called before sprint start takes place. Allows the producer to reset values */
        beforeSprintStart: function (data) {
            // As this is the sprint start, we reset all timespent values to 0
            // We already take care elsewhere that issues that are no in the sprint won't accumulate time, but this won't
            // cater for the case where an issue is added to a sprint at time x, and then the start date of the sprint moved
            // to time y. Inbetween x and y the time spent will be accumulated as from an issue point of the the issue is in
            // sprint (sprint field value set + mapped column)
            _.each(data.issueState, function (issue) {
                if (!_.isUndefined(issue.values.timeSpent)) { issue.values.timeSpent = 0; }
            });
            data.maxValues.timeSpent = 0;
            data.currentValues.timeSpent = 0;
        },

        /** Copies current issue data into the currentState object. */
        copyToCurrentState: function (currentState, issueData) {
            currentState.estimateDeltas = _.clone(issueData.estimateDeltas);
        },

        /** Removes expired estimate changes */
        removeExpiredEstimateChanges: function (issueData, date) {
            if (!issueData.estimateDeltas) { issueData.estimateDeltas = []; }
            issueData.estimateDeltas = _.reject(issueData.estimateDeltas, function (elem) {
                // remove items that are in the past
                return elem.date <= date;
            });
        },

        /** Add an estimate change for a given issue */
        addEstimateChange: function (issueData, changeDate, change) {
            if (!issueData.estimateDeltas) { issueData.estimateDeltas = []; }
            if (!change.timeC || !change.timeC.changeDate) { return; }

            var date = change.timeC.changeDate;
            var estimateDelta = (change.timeC.newEstimate || 0) - (change.timeC.oldEstimate || 0);

            // add an estimate change delta
            issueData.estimateDeltas.push({
                worklogDate: changeDate,
                date: date,
                issueKey: change.key,
                delta: estimateDelta,
                change: change // use to compare to current change
            });
        },

        /** Get the estimate changes that affect a current change */
        getAffectingEstimateChanges: function (issueData, change) {
            // Return empty list if the change isn't estimate change related, as it won't be affected
            if ((!change.timeC) || (!change.timeC.changeDate)) { return []; }

            // Filter out backdated changes that have been recorded before the current change, as this means
            // both estimateDelta as well as change have been backdated in parallel. In this case the first change
            // does not affect the second as they happen in chronological order
            return _.filter(issueData.estimateDeltas, function (estimateDelta) {
                // don't consider estimateDelta if its changeDate is after the changeDate
                return change.timeC.changeDate < estimateDelta.date; // true = keep = estimateDelta date is after changeDate
            });
        }
    };


    /**
     * Consumer for statistic field value changes
     */
    BurndownTimelineProducer.EstimateStatisticConsumer = {
        /** Data fields set by this consumer. Fields are automatically included in deltas and rollups */
        dataFields: ['realValue', 'estimate'],

        /** Processes the change for a given issue */
        processChange: function (issueData, change) {
            // value change
            if (change.statC) {
                issueData.values.oldEstimateFieldValue = change.statC.oldValue;
                issueData.values.estimateFieldValue = change.statC.newValue;
                issueData.estimateChange = true;
            } else {
                issueData.estimateChange = false;
            }

            // decide what the estimate value for the current state should be
            if (issueData.isInChart && !issueData.burned) {
                // show real field value
                issueData.values.estimate = issueData.values.estimateFieldValue;
            } else {
                // value has been burned, show 0 (or - if not set)
                issueData.values.estimate = issueData.values.estimateFieldValue ? 0 : issueData.values.estimateFieldValue;
            }
        },

        /** Called before changes get processed */
        beforeProcessChange: function (issueData, changeDate, change) {
        },

        /** Called after a change gets processed */
        afterProcessChange: function (issueData, changeDate, change) {
        },

        /** Called before sprint start takes place. Allows the producer to reset values */
        beforeSprintStart: function (data) {
        },

        /** Copies current issue data into the currentState object. */
        copyToCurrentState: function (currentState, issueData) {
        }
    };

    /**
     * Consumer for statistic field value changes
     */
    BurndownTimelineProducer.ProgressConsumer = {
        /** Data fields set by this consumer. Fields are automatically included in deltas and rollups */
        dataFields: ['issueCount', 'unestimatedIssueCount', 'totalEstimate', 'completedEstimate'],

        /** Processes the change for a given issue */
        processChange: function (issueData, change) {
            // value change
            if (change.statC) {
                issueData.values.oldEstimateFieldValue = change.statC.oldValue;
                issueData.values.estimateFieldValue = change.statC.newValue;
                issueData.estimateChange = true;
            } else {
                issueData.estimateChange = false;
            }

            if (issueData.isInChart) {
                issueData.values.issueCount = 1;
                issueData.values.unestimatedIssueCount = (issueData.estimatable && _.isUndefined(issueData.values.estimateFieldValue) && !issueData.burned) ? 1 : 0;
                var estimateOrZero = issueData.values.estimateFieldValue || 0;
                issueData.values.totalEstimate = estimateOrZero;
                issueData.values.completedEstimate = issueData.burned ? estimateOrZero : 0;
            } else {
                issueData.values.issueCount = 0;
                issueData.values.unestimatedIssueCount = 0;
                issueData.values.totalEstimate = 0;
                issueData.values.completedEstimate = 0;
            }

            // decide what the estimate value for the current state should be
            if (issueData.isInChart && !issueData.burned) {
                // show real field value
                issueData.values.estimate = issueData.values.estimateFieldValue;
            } else {
                // value has been burned, show 0 (or - if not set)
                issueData.values.estimate = issueData.values.estimateFieldValue ? 0 : issueData.values.estimateFieldValue;
            }
        },

        /** Called before changes get processed */
        beforeProcessChange: function (issueData, changeDate, change) {
        },

        /** Called after a change gets processed */
        afterProcessChange: function (issueData, changeDate, change) {
        },

        /** Called before sprint start takes place. Allows the producer to reset values */
        beforeSprintStart: function (data) {
        },

        /** Copies current issue data into the currentState object. */
        copyToCurrentState: function (currentState, issueData) {
        }
    };

    return BurndownTimelineProducer;
});
