/**
 * Chart View
 * @module jira-agile/rapid/ui/chart/chart-view
 * @requires module:jquery
 */
define('jira-agile/rapid/ui/chart/chart-view', ['require'], function(require) {
    'use strict';

    var $ = require('jquery');

    /*
     * Grouping together standard functions for all charts that share the same HTML structure
     * for overall chart content and spinner
     */

    var ChartView = {};

    /*
     * Element locators
     */
    ChartView.getChartContentElem = function (doNotUseCached) {
        if (!ChartView.chartContentElem || doNotUseCached) {
            ChartView.chartContentElem = $('#ghx-chart-content');
        }
        return ChartView.chartContentElem;
    };

    ChartView.getChartGroup = function (doNotUseCached) {
        if (!ChartView.chartGroup || doNotUseCached) {
            ChartView.chartGroup = $('#ghx-chart-group');
        }
        return ChartView.chartGroup;
    };

    ChartView.getChartStatus = function (doNotUseCached) {
        if (!ChartView.chartStatus || doNotUseCached) {
            ChartView.chartStatus = $('#ghx-chart-status');
        }
        return ChartView.chartStatus;
    };

    ChartView.getChartView = function (doNotUseCached) {
        if (!ChartView.chartView || doNotUseCached) {
            ChartView.chartView = $('#ghx-chart-view');
        }
        return ChartView.chartView;
    };

    ChartView.getChartControls = function (doNotUseCached) {
        if (!ChartView.chartControls || doNotUseCached) {
            ChartView.chartControls = $('#ghx-chart-controls');
        }
        return ChartView.chartControls;
    };

    ChartView.getChartAllControlsElem = function (doNotUseCached) {
        if (!ChartView.chartAllControlsElem || doNotUseCached) {
            ChartView.chartAllControlsElem = $('#ghx-chart-options');
        }
        return ChartView.chartAllControlsElem;
    };

    ChartView.getChartTimeControlsElem = function (doNotUseCached) {
        if (!ChartView.chartTimeControlsElem || doNotUseCached) {
            ChartView.chartTimeControlsElem = $('#ghx-chart-options-time');
        }
        return ChartView.chartTimeControlsElem;
    };

    ChartView.getChartSelector = function (doNotUseCached) {
        if (!ChartView.chartSelector || doNotUseCached) {
            ChartView.chartSelector = $('#ghx-chart-selector');
        }
        return ChartView.chartSelector;
    };

    ChartView.getChartLegend = function (doNotUseCached) {
        if (!ChartView.chartLegend || doNotUseCached) {
            ChartView.chartLegend = $('#ghx-chart-legend');
        }
        return ChartView.chartLegend;
    };
    ChartView.getChartLegendD3 = function (doNotUseCached) {
        if (!ChartView.chartLegendD3 || doNotUseCached) {
            ChartView.chartLegendD3 = $('.ghx-chart-legend');
        }
        return ChartView.chartLegendD3;
    };
    ChartView.getChartSnapshot = function (doNotUseCached) {
        if (!ChartView.chartSnapshot || doNotUseCached) {
            ChartView.chartSnapshot = $('#ghx-chart-snapshot');
        }
        return ChartView.chartSnapshot;
    };

    ChartView.getReopenButton = function(doNotUseCached) {
        if (!ChartView.reopenButton || doNotUseCached) {
            ChartView.reopenButton = $('#ghx-chart-reopen-sprint');
        }
        return ChartView.reopenButton;
    }


    /*
     * Hide / show elements
     */
    ChartView.showChartGroup = function () {
        ChartView.getChartGroup().show();
    };

    ChartView.hideChartGroup = function () {
        ChartView.getChartGroup().hide();
    };

    ChartView.showChartStatus = function () {
        ChartView.getChartStatus().show();
    };

    ChartView.hideChartStatus = function () {
        ChartView.getChartStatus().hide();
    };

    ChartView.hideChartControls = function () {
        ChartView.getChartControls().hide();
    };

    ChartView.showChartControls = function () {
        ChartView.getChartControls().show();
    };

    /**
     * Shows the spinner
     */
    ChartView.showSpinner = function () {
        var chartGroup = ChartView.getChartGroup(true).spin('medium');
        if (chartGroup.find('.ghx-chart-blanket.js-blanket-spinner').length === 0) {
            $('<div></div>').addClass('ghx-chart-blanket js-blanket-spinner').appendTo(chartGroup);
        }
    };

    /**
     * Hides the spinner
     */
    ChartView.hideSpinner = function () {
        ChartView.getChartGroup(true).spinStop()
            .find('.ghx-chart-blanket.js-blanket-spinner').remove();
    };

    /**
     * Shows a blank chart message over the chart area
     * @param {String} message Rendered message template
     */
    ChartView.showBlankStateMessage = function (message) {
        var chartGroup = ChartView.getChartGroup(true);
        // Allow updating existing blank message area. User input in response to message may still resulted in blank chart
        var blank = chartGroup.find('.ghx-chart-blanket.ghx-blank-state').empty();
        if (blank.length === 0) {
            blank = $('<div></div>').addClass('ghx-chart-blanket ghx-blank-state').appendTo(chartGroup);
        }
        blank.append(message);
    };

    /**
     * Hides blank chart message
     */
    ChartView.hideBlankStateMessage = function () {
        ChartView.getChartGroup(true)
            .find('.ghx-chart-blanket.ghx-blank-state').remove();
    };

    /**
     * Shows an IE-unsupported message over the entire chart view
     * @param {String} message Rendered message template
     */
    ChartView.showIE8SupportMessage = function (message) {
        var area = ChartView.getChartContentElem();
        var blank = area.find('.ghx-chart-blanket.ghx-no-ie8').empty();
        if (blank.length === 0) {
            blank = $('<div></div>').addClass('ghx-chart-blanket ghx-blank-state').appendTo(area);
        }
        blank.append(message);
    };

    /**
     * Hides IE-unsupported message
     */
    ChartView.hideIE8SupportMessage = function () {
        ChartView.getChartContentElem(true)
            .find('.ghx-chart-blanket.ghx-no-ie8').remove();
    };

    /**
     * Show a message in the top-right corner of the chart for a short duration.
     */
    ChartView.showToast = (function () {
        var TOAST_DURATION = 1000;
        var FADE_IN_DURATION = 500;
        var FADE_OUT_DURATION = 1000;
        var MARGIN = 20;
        var timeout = null;

        /**
         * Calculates where to put the top of the toast, so that it is visible even if the top of
         * the chart is scrolled offscreen.
         *
         * @param {jQuery} chartGroup
         * @returns {number}
         */
        function top(chartGroup) {
            var scrollContainer = $('#ghx-report-scrollarea');
            var offsetDiff = scrollContainer.offset().top - chartGroup.offset().top;
            return Math.max(offsetDiff, 0) + MARGIN;
        }

        /**
         * @param {String} message
         */
        return function (message) {
            var chartGroup = ChartView.getChartGroup(true);
            var toast = chartGroup.find('.ghx-chart-toast');
            if (toast.length === 0) {
                toast = $('<div></div>').addClass('ghx-chart-toast').appendTo(chartGroup);
            }
            toast.stop().text(message)
                .css({
                    opacity: 0,
                    top: top(chartGroup),
                    right: MARGIN
                })
                .animate({
                    opacity: 0.9
                }, FADE_IN_DURATION);
            if (timeout !== null) {
                clearTimeout(timeout);
            }
            timeout = setTimeout(function () {
                // Check that the element is still in the document
                if (toast.closest('html').length) {
                    toast.fadeOut(FADE_OUT_DURATION, function () {
                        toast.remove();
                    });
                }
            }, TOAST_DURATION);
        };
    }());

    ChartView.showChartUpdatedToast = function () {
        ChartView.showToast('Chart updated');
    };

    return ChartView;
});
