/**
 * Non-Working Days Util
 * @module jira-agile/rapid/ui/chart/non-working-days-util
 * @requires module:underscore
 * @requires module:jira-agile/rapid/ui/chart/burndown-rate
 */
define('jira-agile/rapid/ui/chart/non-working-days-util', ['require'], function (require) {
    'use strict';

    // REQUIRES

    var _ = require('underscore');
    var BurndownRate = require('jira-agile/rapid/ui/chart/burndown-rate');

    var NonWorkingDaysUtils = {};

    /**
     * Transforms the given series so that the returned series will any non working days deduced from their x value (time)
     * @param series
     * @param startDate
     * @param an array of serie id's. Series with one of these id's will be transformed.
     * @return {*} The transformed series
     */
    NonWorkingDaysUtils.transformSeriesExcludingNonWorkingDays = function (series, startDate, serieIdsToTransform) {

        if (_.isUndefined(serieIdsToTransform)) {
            serieIdsToTransform = _.pluck(series, "id");
        }

        // define function used to calculate the amount by which each point must move
        var getTransformValue = function getTransformValue(point, block) {
            if (point.xValue >= block.start) {
                if (block.start <= startDate && block.end >= startDate) {
                    if (point.xValue >= block.end) {
                        return block.end - startDate;
                    }
                } else {
                    if (point.xValue < block.end) {
                        // if it occurs between the start and end, move it back to the start of the non-working block
                        return point.xValue - block.moveTo;
                    } else {
                        // otherwise move it back by the length of the non-working block
                        return block.end - block.moveTo;
                    }
                }
            }
            return 0;
        };

        // create a new array to avoid changing the existing series
        var newSeries = _.map(series, function (serie) {
            var newSerie = _.clone(serie);
            newSerie.data = _.map(serie.data, function (point) {
                return {
                    xValue: point[0],
                    yValue: point[1],
                    xTransform: 0
                };
            });
            return newSerie;
        });

        // Look through all non working day blocks and adjust move any data points inside the blocks to the edges
        var nonWorkingBlocks = BurndownRate.getAdjustedNonWorkingDays(startDate);
        _.each(nonWorkingBlocks, function (block) {
            _.each(newSeries, function (serie) {
                serie.data = _.map(serie.data, function (point) {
                    return {
                        xValue: point.xValue,
                        yValue: point.yValue,
                        xTransform: point.xTransform + getTransformValue(point, block)
                    };
                });
            });
        });

        return _.map(newSeries, function (serie) {
            var transformedSerie = _.clone(serie);
            if (_.indexOf(serieIdsToTransform, serie.id) > -1) {
                transformedSerie.data = _.map(serie.data, function (point) {
                    return [point.xValue - point.xTransform, point.yValue];
                });
            } else {
                transformedSerie.data = _.map(serie.data, function (point) {
                    return [point.xValue, point.yValue];
                });
            }

            return transformedSerie;
        });
    };

    /**
     * If the given date is in a non working day, a new date will be returned that is in a working day.
     * The position inside the non working day block of the given date determines if the new date is before or after the non working day block.
     * @param date
     * @return a date that is not inside a non working day block
     */
    NonWorkingDaysUtils.transformDateIfInNonWorkingDays = function (date, startDate) {
        if (date < startDate) {
            return new Date(date);
        }

        var nonWorkingBlocks = BurndownRate.getAdjustedNonWorkingDays(startDate);
        var dateLabel = date;

        _.each(nonWorkingBlocks, function (block) {
            if (dateLabel >= block.start) {
                dateLabel += block.end - block.moveTo;
            }
        });

        return new Date(dateLabel);
    };

    /**
     * Deducts the non working days from the given date, starting from the given startDate until the given date.
     * @param date
     * @param startDate
     * @return {*}
     */
    NonWorkingDaysUtils.deductNonWorkingDays = function (date, startDate) {
        var nonWorkingBlocks = BurndownRate.getAdjustedNonWorkingDays(startDate);

        var startBlock = _.filter(nonWorkingBlocks, function (block) {
            return startDate >= block.start && startDate < block.end;
        });
        var dateBlock = _.filter(nonWorkingBlocks, function (block) {
            return date >= block.start && date < block.end;
        });

        var ajustedDate = date;
        if (dateBlock.length > 0) {
            ajustedDate = dateBlock[0].start;
        }

        var ajustedStartDate = startDate;
        if (startBlock.length > 0) {
            ajustedStartDate = startBlock[0].end;
            ajustedDate -= ajustedStartDate - startDate;
        }

        var localBlocks = _.filter(nonWorkingBlocks, function (block) {
            return block.end > ajustedStartDate && block.start < ajustedDate;
        });

        _.each(localBlocks, function (block) {
            ajustedDate = ajustedDate - (block.end - block.start);
        });
        return ajustedDate;
    };

    return NonWorkingDaysUtils;
});