/* global AJS, GH */
AJS.test.require(['com.pyxis.greenhopper.jira:gh-test-common', 'com.pyxis.greenhopper.jira:gh-rapid'], function() {
    var BurndownRate = require('jira-agile/rapid/ui/chart/burndown-rate');
    var BurndownChartController = require('jira-agile/rapid/ui/chart/burndown-chart-controller');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var NonWorkingDaysUtils = require('jira-agile/rapid/ui/chart/non-working-days-util');
    GlobalEvents.trigger('pre-initialization');

    module('Non-working days', {
        teardown: function() {
            // ensure non-working days rates are reloaded between tests
            delete BurndownChartController.nonWorkingDays;
        }
    });

    test('module "jira-agile/rapid/ui/chart/non-working-days-util" exists', function() {
        ok(NonWorkingDaysUtils);
        ok(GH.NonWorkingDaysUtils);
    });

    var startDate = new Date(2012, 10, 28).valueOf(); // Wednesday
    var endDate = new Date(2012, 11, 12).valueOf(); // Wednesday 2 weeks later
    var startWeekend1 = new Date(2012, 11, 1).valueOf();
    var endWeekend1 = new Date(2012, 11, 3).valueOf();
    var startWeekend2 = new Date(2012, 11, 8).valueOf();
    var endWeekend2 = new Date(2012, 11, 10).valueOf();

    var workRateData = {
        timezone: 'Australia/Sydney',
        rates: [
            { start: startDate, end: startWeekend1, rate: 1.0 },
            { start: startWeekend1, end: endWeekend1, rate: 0.0 },
            { start: endWeekend1, end: startWeekend2, rate: 1.0 },
            { start: startWeekend2, end: endWeekend2, rate: 0.0 },
            { start: endWeekend2, end: endDate, rate: 1.0 }
        ]
    };

    var startWithNonWorkingDayData = {
        timezone: 'Australia/Sydney',
        rates: [
            { start: startWeekend1, end: endWeekend1, rate: 0.0 },
            { start: endWeekend1, end: endDate, rate: 1.0 }
        ]
    };

    test('re-calculating series with non-working days', function() {
        BurndownRate.setRateData(workRateData);

        var exampleSeries = {
            id: 5,
            color: 'black',
            label: 'example',
            data: [
                [ new Date(2012, 10, 27).valueOf(), 10 ], // Tuesday - should not change
                [ startWeekend1, 10 ], // start of weekend - should not change
                [ new Date(2012, 11, 1).valueOf(), 10 ], // Saturday - expect to move to the start of the weekend
                [ new Date(2012, 11, 2).valueOf(), 10 ], // Sunday - expect to move to the start of the weekend
                [ endWeekend1, 10 ], // End of weekend will be moved back to start
                [ new Date(2012, 11, 4).valueOf(), 10 ], // After weekend 1 - should move back by length of weekend
                [ startWeekend2, 10 ], // After weekend 1 - should move back by length of one weekend
                [ new Date(2012, 11, 9).valueOf(), 10 ] // After weekend 2 - should move back by length of both weekends
            ]
        };

        var recalculatedSeries = NonWorkingDaysUtils.transformSeriesExcludingNonWorkingDays([exampleSeries], startDate);

        // make sure we just got one back
        equal(recalculatedSeries.length, 1);
        equal(recalculatedSeries[0].id, 5);
        equal(recalculatedSeries[0].color, 'black');
        equal(recalculatedSeries[0].label, 'example');


        var points = recalculatedSeries[0].data;
        // make sure it still has five points
        equal(points.length, 8);

        // dates should have been adjusted correctly
        equal(points[0][0], new Date(2012, 10, 27).valueOf());
        equal(points[1][0], startWeekend1);
        equal(points[2][0], startWeekend1);
        equal(points[3][0], startWeekend1);
        equal(points[4][0], startWeekend1);
        equal(points[5][0], new Date(2012, 11, 2).valueOf());
        equal(points[6][0], new Date(2012, 11, 6).valueOf());
        equal(points[7][0], new Date(2012, 11, 6).valueOf());

    });

    test('moves first date in series to end of non-working block when chart starts with a non-working day', function() {

        // data starts with weekend
        BurndownRate.setRateData(startWithNonWorkingDayData);

        var exampleSeries = {
            data: [
                [ new Date(2012, 11, 2), 10 ], // inside first weekend
                [ new Date(2012, 11, 9).valueOf(), 5 ]
            ]
        };
        var recalculatedSeries = NonWorkingDaysUtils.transformSeriesExcludingNonWorkingDays([exampleSeries], startWeekend1);
        var points = recalculatedSeries[0].data;

        equal(points[0][0], new Date(2012, 11, 2).valueOf()); // move to end of weekend
        equal(points[1][0], new Date(2012, 11, 7).valueOf()); // ignore the first weekend since it will just be cut off the chart
    });


    var week1 = {
        start: new Date(2012, 10, 27).valueOf(),
        end: new Date(2012, 11, 1).valueOf(),
        rate: 1.0
    };
    var weekend1 = {
        start: new Date(2012, 11, 1).valueOf(),
        end: new Date(2012, 11, 3).valueOf(),
        rate: 0.0
    };
    var week2 = {
        start: new Date(2012, 11, 3).valueOf(),
        end: new Date(2012, 11, 8).valueOf(),
        rate: 1.0
    };
    var weekend2 = {
        start: new Date(2012, 11, 8).valueOf(),
        end: new Date(2012, 11, 10).valueOf(),
        rate: 0.0
    };
    var week3 = {
        start: new Date(2012, 11, 10).valueOf(),
        end: new Date(2012, 11, 15).valueOf(),
        rate: 1.0
    };
    var weekend3 = {
        start: new Date(2012, 11, 15).valueOf(),
        end: new Date(2012, 11, 17).valueOf(),
        rate: 0.0
    };
    var week4 = {
        start: new Date(2012, 11, 17).valueOf(),
        end: new Date(2012, 11, 22).valueOf(),
        rate: 1.0
    };
    var weekend4 = {
        start: new Date(2012, 11, 22).valueOf(),
        end: new Date(2012, 11, 25).valueOf(), // mon non working day
        rate: 0.0
    };
    var week5 = {
        start: new Date(2012, 11, 25).valueOf(),
        end: new Date(2012, 11, 29).valueOf(),
        rate: 1.0
    };
    var NON_WORKING_DATA = {
        timezone: 'Australia/Sydney',
        rates: [week1, weekend1, week2, weekend2, week3, weekend3, week4, weekend4, week5]
    };

    function randomDate(period) {
        var dist = period.end - period.start;
        return period.start + (Math.random() * dist);
    }
    function midDate(period) {
        return period.start + (period.end - period.start) / 2;
    }
    function eqDates(actual, expected, message) {
        equal(actual, expected, (message || '') + ' :: ' + new Date(actual) + ' !=! ' + new Date(expected) + '');
    }

    var SECOND = 1000;
    var MINUTE = 60 * SECOND;
    var HOUR = 60 * MINUTE;
    var DAY = 24 * HOUR;

    test('deductNonWorkingDays - single non-working between start and end', function() {
        BurndownRate.setRateData(NON_WORKING_DATA);

        var start = randomDate(week2);
        var end = randomDate(week3);

        var target = NonWorkingDaysUtils.deductNonWorkingDays(end, start);
        eqDates(target, end - 2 * DAY);
    });

    test('deductNonWorkingDays - multi non-working between start and end', function() {
        BurndownRate.setRateData(NON_WORKING_DATA);

        var start = randomDate(week2);
        var end = randomDate(week4);

        var target = NonWorkingDaysUtils.deductNonWorkingDays(end, start);
        eqDates(target, end - 4 * DAY);
    });

    test('deductNonWorkingDays - start date in non-working day', function() {
        BurndownRate.setRateData(NON_WORKING_DATA);

        var start = midDate(weekend2);
        var end = midDate(week3);

        var target = NonWorkingDaysUtils.deductNonWorkingDays(end, start);
        eqDates(target, end - 1 * DAY);
    });

    test('deductNonWorkingDays - start date in non-working day 2', function() {
        BurndownRate.setRateData(NON_WORKING_DATA);

        var start = midDate(weekend2);
        var end = midDate(week4);

        var target = NonWorkingDaysUtils.deductNonWorkingDays(end, start);
        eqDates(target, end - 3 * DAY);
    });

    test('deductNonWorkingDays - end date in non-working day', function() {
        BurndownRate.setRateData(NON_WORKING_DATA);

        var start = midDate(week2);
        var end = midDate(weekend2);

        var target = NonWorkingDaysUtils.deductNonWorkingDays(end, start);
        eqDates(target, end - 1 * DAY);
    });

    test('deductNonWorkingDays - end date in non-working day 2', function() {
        BurndownRate.setRateData(NON_WORKING_DATA);

        var start = midDate(week2);
        var end = midDate(weekend3);

        var target = NonWorkingDaysUtils.deductNonWorkingDays(end, start);
        eqDates(target, end - 3 * DAY);
    });

    test('deductNonWorkingDays - start and end date in non-working day', function() {
        BurndownRate.setRateData(NON_WORKING_DATA);

        var start = midDate(weekend1);
        var end = midDate(weekend3);

        var target = NonWorkingDaysUtils.deductNonWorkingDays(end, start);
        eqDates(target, end - 4 * DAY);
    });
});
