(function (_, moment, jq, d3, Ajax, RapidViewConfig, NumberFormat, TimeFormat, Util, UrlFormat) {
    'use strict';

    var DATE_FORMAT = 'LL';
    var NUM_SPRINTS_USED_FOR_FORECAST = 3;

    /**
     * Flatten the structure from a map of arrays of changes to just an array of changes.
     * Convert events with multiple changes to multiple events with one change each.
     *
     * @param {Object} data
     * @returns {Object[]}
     */
    function simplifyChanges(data) {
        var changesMap = data.changes;
        var simplifiedChanges = [];
        var time;
        var change;

        function addSimplifiedChange(type, value) {
            var simplified = {
                time: +time, // Time is a string when it's a key in the map. Needs to be converted to a number.
                key: change.key
            };
            simplified[type] = value;
            simplifiedChanges.push(simplified);
        }

        for (time in changesMap) {
            if (changesMap.hasOwnProperty(time)) {
                for (var i = 0; i < changesMap[time].length; i++) {
                    change = changesMap[time][i];
                    if (change.column) {
                        addSimplifiedChange('done', !change.column.notDone);
                    }
                    if (change.statC) {
                        addSimplifiedChange('estimate', change.statC.newValue);
                    }
                    if ('added' in change) {
                        addSimplifiedChange('added', change.added);
                    }
                }
            }
        }

        // Ensure changes are sorted by time. This is likely to be the case, but because
        // the data comes from a map there's no guarantee of order.
        return _.sortBy(simplifiedChanges, 'time');
    }

    /**
     * Not all change entries are represented in the chart. Extract only the applicable entries.
     *   - Estimate changes are ignored.
     *   - Changes to "done" state are removed if it is undone later.
     *   - Changes to "not done" state are ignored.
     *
     * @param {Object[]} changes
     * @returns {Object[]}
     */
    function getApplicableChangesAndIssueStates(changes) {
        var result = [];
        var issueStates = {};

        function eraseWorkHistory(key) {
            result = _.reject(result, function (change) {
                if (change.key !== key) {
                    return false;
                }
                if (change.wasDone) {
                    change.wasDone = false;
                }
                return 'done' in change;
            });
            // Also update state
            issueStates[key].done = false;
        }

        function getDoneTime(key) {
            var doneChanges = _.where(changes, {
                key: key,
                done: true
            });
            return _.last(doneChanges).time;
        }

        _.each(changes, function (change) {
            // Ignore estimate changes
            if ('estimate' in change) {
                return;
            }

            var key = change.key;
            if (!issueStates[key]) {
                issueStates[key] = {};
            }
            var state = issueStates[key];

            if ('added' in change) {
                // Ignore if this is a duplicate added entry
                if (state.added !== change.added) {
                    // Note whether or not issue is already done when added or removed
                    if (state.done) {
                        change = _.extend({ wasDone: true }, change);
                        //if we add 'done' issues, we use 'Done' time instead of 'Added' time
                        //and add 'Done' change if current state is removed and done
                        if (change.added) {
                            change.time = getDoneTime(change.key);
                            result = _.reject(result, function (change) {
                                if (change.key !== key) {
                                    return false;
                                }
                                return 'done' in change;
                            });
                            result.push({
                                key: change.key,
                                done: true,
                                time: change.time
                            });
                        }
                    }
                    result.push(change);

                    state.added = change.added;
                }
            } else if ('done' in change) {
                // Pretend no work has been completed if an issue is moved back to not done
                if (!change.done) {
                    eraseWorkHistory(key);
                }
                // Only include a done change if it is currently not done and not removed.
                else if (!state.done && state.added !== false) {
                        result.push(change);
                    }

                state.done = change.done;
            }
        });

        return {
            changes: result,
            issueStates: issueStates
        };
    }

    /**
     * Retrieve issue keys that have not been removed later
     *
     * @param {Object[]} changes
     * @returns {String[]}
     */
    function getIssueKeys(changes) {
        var issueKeys = {};

        _.each(changes, function (change) {
            if ('added' in change) {
                if (change.added) {
                    issueKeys[change.key] = true;
                } else {
                    delete issueKeys[change.key];
                }
            }
        });

        return _.keys(issueKeys);
    }

    /**
     * Get the final estimate for each issue.
     *
     * We only consider a subset of all the estimation values: the ones listed in the estimatable issue keys.
     * It is possible for issues to have a value set in their estimation field, but not be estimatable. An
     * example of this would be a Story that was given an estimate, but its issue type gets changed to another
     * issue type which does not have the appropriate estimation field associated with it.
     *
     * @param {Object[]} changes
     * @param {String[]} estimatableIssueKeys a list of keys whose estimates should be used.
     * @returns {Object}
     */
    function getEstimates(changes, estimatableIssueKeys) {
        var estimates = {};
        var estimatableMap = _.groupBy(estimatableIssueKeys); // TODO: Upgrade to underscore 1.5.2, then use _.indexBy
        _.each(changes, function (change) {
            if ('estimate' in change && _.has(estimatableMap, change.key)) {
                if (_.isNumber(change.estimate)) {
                    estimates[change.key] = change.estimate;
                } else {
                    delete estimates[change.key];
                }
            }
        });
        return estimates;
    }

    /**
     * In the list of changes, find the time of the first issue transition.
     *
     * @param {Object[]} changes
     * @param {number} [atOrAfter] look for first time >= this value
     * @returns {number} timestamp, or -1 if no transitions were found.
     */
    function getFirstTransitionTime(changes, atOrAfter) {
        var firstTransition = _.find(changes, function (change) {
            if (atOrAfter != null && change.time < atOrAfter) {
                return false;
            }
            return 'done' in change;
        });
        return firstTransition ? firstTransition.time : -1;
    }

    /**
     * Get the sprint id associated with the given time.
     *
     * @param {Object[]} sprints sorted by start time
     * @param {number} time
     * @returns {number|'original'} sprint id, or 'original'
     */
    function getSprintIdAtTime(sprints, time) {
        // Handle cases of:
        //  - no sprints
        //  - event occurs before first sprint (i.e. part of original estimate)
        if (!sprints.length || sprints[0].startTime > time) {
            return 'original';
        }
        var sprint = _.find(sprints, function (sprint, i) {
            if (i === sprints.length - 1) {
                return true;
            }
            return sprint.startTime <= time && sprints[i + 1].startTime > time;
        });
        return sprint.id;
    }

    /**
     * Assign changes to sprints based on time of occurence.
     *
     * @param {Object[]} changes
     * @param {Object[]} sprints
     * @returns {Object} changes mapped by sprint id
     */
    function groupChangesBySprint(changes, sprints) {
        var grouped = {};
        var firstTransitionTime = getFirstTransitionTime(changes);
        if (firstTransitionTime === -1) {
            grouped.original = changes;
        } else {
            var firstTransitionSprintId = getSprintIdAtTime(sprints, firstTransitionTime);
            _.each(changes, function (change) {
                var sprintId = getSprintIdAtTime(sprints, change.time);
                if (change.time < firstTransitionTime && sprintId !== firstTransitionSprintId) {
                    sprintId = 'original';
                }
                if (!grouped[sprintId]) {
                    grouped[sprintId] = [];
                }
                grouped[sprintId].push(change);
            });
        }
        return normalizeGroupedChanges(grouped);
    }

    /**
     * Normalize grouped changes. Removing all duplicated add/remove histories.
     *
     * @param {Object} groupedChange
     * @returns {Object} groupedChanges without duplicated add/remove histories
     */
    function normalizeGroupedChanges(groupedChange) {
        var result = {};
        _.each(groupedChange, function (changes, key) {
            result[key] = result[key] || [];

            _.each(changes, function (change) {
                if (!('added' in change)) {
                    result[key].push(change);
                }
            });

            var groupedAddAndRemoveChanges = {};
            _.each(changes, function (change) {
                if ('added' in change) {
                    groupedAddAndRemoveChanges[change.key] = groupedAddAndRemoveChanges[change.key] || [];
                    groupedAddAndRemoveChanges[change.key].push(change);
                }
            });

            function getLastAddOrRemoveChange(changes, isAdd) {
                return _.find(_.clone(changes).reverse(), function (change) {
                    return change.added === isAdd;
                });
            }

            function removeAddRemoveChanges(changes) {
                return _.reject(changes, function (change) {
                    return 'added' in change;
                });
            }

            _.each(groupedAddAndRemoveChanges, function (issueAddAndRemoveChanges, issueKey) {
                var addRemoveNet = _.reduce(issueAddAndRemoveChanges, function (value, change) {
                    return value + (change.added ? 1 : -1);
                }, 0);

                if (addRemoveNet > 0) {
                    var lastAddedChange = getLastAddOrRemoveChange(issueAddAndRemoveChanges, true);
                    if (lastAddedChange) {
                        groupedAddAndRemoveChanges[issueKey] = removeAddRemoveChanges();
                        groupedAddAndRemoveChanges[issueKey].push(lastAddedChange);
                    }
                } else if (addRemoveNet < 0) {
                    var lastRemovedChange = getLastAddOrRemoveChange(issueAddAndRemoveChanges, false);
                    if (lastRemovedChange) {
                        groupedAddAndRemoveChanges[issueKey] = removeAddRemoveChanges();
                        groupedAddAndRemoveChanges[issueKey].push(lastRemovedChange);
                    }
                } else {
                    groupedAddAndRemoveChanges[issueKey] = removeAddRemoveChanges();
                    // if we add and remove issues from version/epic at the same time frame and last action is remove,
                    // we remove all histories belong to that issues
                    if (_.last(issueAddAndRemoveChanges).added === false) {
                        result[key] = _.reject(result[key], function (change) {
                            return change.key === issueKey;
                        });
                    }
                }
            });

            _.each(_.flatten(groupedAddAndRemoveChanges), function (change) {
                result[key].push(change);
            });
        });
        return result;
    }

    /**
     * Calculates the net work added in a set of changes.
     * A negative value indicates that the amount was removed.
     *
     * @param {Object} estimates
     * @param {Object[]} changes
     * @returns {{ added: number, removed: number, net: number}}
     */
    function getWorkAddedAndRemoved(estimates, changes) {
        var result = {
            added: 0,
            removed: 0,
            net: 0
        };
        _.each(changes || [], function (change) {
            if ('added' in change) {
                var estimate = estimates[change.key] || 0;
                if (change.added) {
                    result.added += estimate;
                } else if (!change.wasDone) {
                    result.removed += estimate;
                }
            }
        });
        result.net = result.added - result.removed;
        return result;
    }

    /**
     * Calculates the work completed in a set of changes.
     *
     * @param {Object} estimates
     * @param {Object[]} changes
     * @returns {number}
     */
    function getWorkCompleted(estimates, changes) {
        return _.reduce(changes, function (memo, change) {
            if (change.done) {
                var estimate = estimates[change.key] || 0;
                return memo + estimate;
            }
            return memo;
        }, 0);
    }

    /**
     * Picks a formatter to render the estimate based on the estimation statistic.
     *
     * @param {String} renderer
     * @returns {Object}
     * @property {Function} formatFull
     * @property {Function} formatCompact
     */
    function getEstimationStatisticFormatter(renderer) {
        var formatter;

        switch (renderer) {
            case 'duration':
                formatter = {
                    formatFull: function formatFull(seconds) {
                        return GH.TimeFormat.formatShortDurationForTimeTrackingConfiguration(seconds, {
                            durationUnit: GH.TimeFormatDurationUnits.SECONDS,
                            dropMinutesIfLongerThanADay: true
                        });
                    },
                    formatCompact: function formatCompact(seconds) {
                        return GH.TimeFormat.formatShortDurationForTimeTrackingConfiguration(seconds, {
                            durationUnit: GH.TimeFormatDurationUnits.SECONDS,
                            timeFormat: GH.TimeFormatDisplays.MOST_SIGNIFICANT
                        });
                    }
                };
                break;
            default:
                formatter = {
                    formatFull: GH.NumberFormat.format,
                    formatCompact: GH.NumberFormat.format
                };
                break;
        }

        return formatter;
    }

    /**
     * Returns forecast stats:
     *  - forecastError: a message explaining why a forecast cannot be made
     *  - sprintsRemaining: number of sprints remaining,
     *  - velocity: average velocity used for forecast
     *  - workRemaining: work remaining, incl. actual work done in the active sprint if applicable
     *  - isLastSprintForecast: whether or not the last sprint should be considered a forecast
     *  - baseline: the baseline of the last sprint
     *
     * @param {Object[]} sprints
     * @returns {Object}
     */
    function getForecastStats(sprints) {
        function getVelocity(sprints) {
            return Math.round(d3.mean(sprints, function (sprint) {
                return sprint.workCompleted;
            }));
        }

        // Determine if a forecast can be made
        var lastSprint = _.last(sprints);
        var isLastSprintActive = lastSprint && lastSprint.isActive;
        var completedSprints = isLastSprintActive ? sprints.slice(0, -1) : sprints;

        var stats = {
            baseline: lastSprint ? lastSprint.baseline : 0
        };

        // Not enough sprints
        if (completedSprints.length < NUM_SPRINTS_USED_FOR_FORECAST) {
            return _.extend(stats, {
                forecastError: AJS.I18n.getText('gh.rapid.charts.scopeburndown.notenoughsprints', NUM_SPRINTS_USED_FOR_FORECAST)
            });
        }

        // No work remaining
        if (lastSprint.workRemaining === 0) {
            return _.extend(stats, {
                sprintsRemaining: 0
            });
        }

        // Get velocity and work remaining
        var velocity = getVelocity(_.last(completedSprints, NUM_SPRINTS_USED_FOR_FORECAST));
        var workRemaining = lastSprint.workRemaining;
        var isLastSprintForecast = false;

        // Special handling if the last sprint is active
        if (isLastSprintActive) {
            if (lastSprint.workCompleted > velocity) {
                // Use the active sprint in the velocity
                velocity = getVelocity(_.last(sprints, NUM_SPRINTS_USED_FOR_FORECAST));
            } else {
                // Treat the active sprint as a forecast sprint, and use the forecast for the work remaining
                var totalWork = lastSprint.workAtStart + lastSprint.workAdded - lastSprint.workRemoved;
                workRemaining = totalWork - Math.min(totalWork, velocity);
                isLastSprintForecast = true;
            }
        }

        // No velocity
        if (velocity === 0) {
            return _.extend(stats, {
                forecastError: AJS.I18n.getText('gh.rapid.charts.scopeburndown.novelocity', NUM_SPRINTS_USED_FOR_FORECAST)
            });
        }

        return _.extend(stats, {
            sprintsRemaining: Math.ceil(workRemaining / velocity),
            velocity: velocity,
            workRemaining: workRemaining,
            isLastSprintForecast: isLastSprintForecast
        });
    }

    /**
     * Returns n forecast sprints. If scope will be completed before n sprints,
     * this will still return n sprints, but padded with empty ones.
     *
     * @param {Object} forecastStats from getForecastStats
     * @param {number} n
     * @returns {Object[]} forecast sprints
     */
    function getForecastSprints(forecastStats, n) {
        if (forecastStats.forecastError || forecastStats.sprintsRemaining === 0) {
            return _.map(_.range(n), function (i) {
                return {
                    sprintId: 'forecast' + i,
                    sprintName: '',
                    baseline: forecastStats.baseline,
                    workAtStart: 0,
                    workAdded: 0,
                    workRemoved: 0,
                    workCompleted: 0,
                    workRemaining: 0,
                    isForecast: true,
                    isActive: false
                };
            });
        }

        // Generate forecast sprints
        var workRemaining = forecastStats.workRemaining;
        return _.map(_.range(n), function (i) {
            var workAtStart = workRemaining;
            var workCompleted = Math.min(workRemaining, forecastStats.velocity);
            workRemaining -= workCompleted;
            return {
                sprintId: 'forecast' + i,
                sprintName: '',
                baseline: forecastStats.baseline,
                workAtStart: workAtStart,
                workAdded: 0,
                workRemoved: 0,
                workCompleted: workCompleted,
                workRemaining: workRemaining,
                isForecast: true,
                isActive: false
            };
        });
    }

    /**
     * Transform sprint and scope data to a form that the chart can render.
     *
     * @param {Object} data
     * @param {boolean} [zeroBaseline]
     * @returns {Object} transformed
     */
    function getScopeBySprintData(data, zeroBaseline) {
        var transformed = {
            sprints: [],
            series: {
                baseline: [],
                workAdded: [],
                workRemaining: [],
                workCompleted: []
            },
            estimationStatistic: data.estimationStatistic,
            labels: data.labels || {},
            rapidViewId: data.rapidViewId
        };

        if (data.epic) {
            transformed.epic = data.epic;
        }

        if (data.version) {
            transformed.version = data.version;
        }

        var simplifiedChanges = simplifyChanges(data);
        var applicableChangesAndIssueStates = getApplicableChangesAndIssueStates(simplifiedChanges);
        var applicableChanges = applicableChangesAndIssueStates.changes;
        var estimates = getEstimates(simplifiedChanges, data.estimatableIssueKeys);
        var allSprints = _.sortBy(data.sprints, 'startTime');
        var sprintChanges = groupChangesBySprint(applicableChanges, allSprints);
        var originalEstimate = getWorkAddedAndRemoved(estimates, sprintChanges.original).net;
        // It is possible for work to be completed before the first sprint starts
        var originalWorkCompleted = getWorkCompleted(estimates, sprintChanges.original);

        // A mock sprint for the original estimate
        var originalSprint = {
            sprintId: 'original',
            sprintName: transformed.labels.originalEstimate,
            baseline: 0,
            workAtStart: originalEstimate,
            workAdded: 0,
            workRemoved: 0,
            workCompleted: originalWorkCompleted,
            workRemaining: originalEstimate - originalWorkCompleted,
            isForecast: false,
            isActive: false,
            issues: _.pluck(_.where(sprintChanges.original, { done: true }), 'key')
        };

        var previousSprint = originalSprint;
        transformed.sprints = _.map(allSprints, function (sprint) {
            var changes = sprintChanges[sprint.id];
            var workAddedAndRemoved = getWorkAddedAndRemoved(estimates, changes);
            var workCompleted = getWorkCompleted(estimates, changes);
            var result = {
                sprintId: sprint.id,
                sprintName: sprint.name,
                baseline: zeroBaseline ? 0 : previousSprint.baseline - workAddedAndRemoved.net,
                workAtStart: previousSprint.workRemaining,
                workAdded: workAddedAndRemoved.added,
                workRemoved: workAddedAndRemoved.removed,
                workCompleted: workCompleted,
                workRemaining: previousSprint.workRemaining - workCompleted + workAddedAndRemoved.net,
                startTime: sprint.startTime,
                endTime: sprint.endTime,
                isForecast: false,
                isActive: sprint.state === 'ACTIVE',
                issues: _.pluck(_.where(changes, { done: true }), 'key')
            };
            previousSprint = result;
            return result;
        });

        // Filter out unimportant sprints
        if (transformed.sprints.length) {
            var firstTransitionTime = getFirstTransitionTime(applicableChanges, transformed.sprints[0].startTime);
            var firstTransitionSprintId = getSprintIdAtTime(allSprints, firstTransitionTime);
            var firstSprintIndex = Util.indexOf(transformed.sprints, function (sprint) {
                return sprint.sprintId === firstTransitionSprintId;
            });
            var lastSprintIndex = Util.indexOf(transformed.sprints, function (sprint) {
                // Working backwards, first sprint in which something happens or there is work remaining
                return sprint.workAdded || sprint.workRemoved || sprint.workCompleted || sprint.workRemaining;
            }, true);
            transformed.sprints = firstSprintIndex >= 0 && lastSprintIndex >= 0 ? transformed.sprints.slice(firstSprintIndex, lastSprintIndex + 1) : [];
        }

        var lastSprint = _.last(transformed.sprints);

        // Calculate overall work remaining before forecasting
        var workRemaining = transformed.sprints.length ? lastSprint.workRemaining : originalSprint.workRemaining;
        // Calculate overall work completed before forecasting
        var workCompleted = originalSprint.workCompleted + _.reduce(transformed.sprints, function (sum, sprint) {
            return sum + sprint.workCompleted;
        }, 0);

        var forecast = transformed.forecast = getForecastStats(transformed.sprints);

        transformed.sprints.unshift(originalSprint);
        transformed.getForecast = _.partial(getForecastSprints, forecast);

        var issuesKeys = getIssueKeys(applicableChanges);

        var estimatedIssuesKeys = _.intersection(issuesKeys, _.keys(estimates));

        function getInScopeFilter(issueStates) {
            return function (issue) {
                return issueStates[issue].added;
            };
        }

        if (workRemaining === 0) {
            var completedIssueKeys = _.pluck(_.where(applicableChanges, { done: true }), 'key');
            var completedIssueKeysInScope = _.filter(completedIssueKeys, getInScopeFilter(applicableChangesAndIssueStates.issueStates));
            var estimatableIssueKeysInScope = _.filter(data.estimatableIssueKeys, getInScopeFilter(applicableChangesAndIssueStates.issueStates));

            var remainingEstimatableIssueKeys = _.difference(estimatableIssueKeysInScope, completedIssueKeysInScope);
            var remainingEstimatableIssueCount = remainingEstimatableIssueKeys.length;
            transformed.forecast.remainingEstimatableIssueCount = remainingEstimatableIssueCount;
        }

        transformed.snapshot = {
            workRemaining: workRemaining,
            workCompleted: workCompleted,
            estimatedIssueCount: estimatedIssuesKeys.length,
            estimatableIssueCount: _.intersection(issuesKeys, data.estimatableIssueKeys).length,
            issueCount: issuesKeys.length
        };

        return transformed;
    }

    /**
     * Transform issue and sprint data to a form that the table can render.
     *
     * @param {Object} issueData
     * @param {Object} sprintData
     * @returns {Object} transformed
     */
    function getIssuesBySprintData(issueData, sprintData) {
        var contents = issueData.contents;
        var issueMap = _.object(_.pluck(contents.completedIssues, 'key'), contents.completedIssues);

        // Add estimation statistic
        issueData.estimationStatistic = sprintData.estimationStatistic;

        contents.incompleteIssues = _.union(contents.incompleteEstimatedIssues, contents.incompleteUnestimatedIssues);
        contents.incompleteIssuesSum = {
            text: issueData.estimationStatistic.formatter.formatFull(contents.incompletedIssuesEstimateSum.value),
            value: contents.incompletedIssuesEstimateSum.value
        };

        // Add Issue Navigator URL for incomplete issues
        contents.incompleteIssuesUrl = UrlFormat.getUrlForIssues(contents.incompleteIssues);

        // Exclude original estimate sprint if it doesn't contain completed issues
        issueData.sprints = _.filter(sprintData.sprints, function (sprint) {
            return sprint.sprintId !== 'original' || sprint.issues.length;
        }).reverse();

        // Prepare data for each sprint
        issueData.sprints = _.map(issueData.sprints, function (sprint) {
            var formattedSprint = {
                sprintId: sprint.sprintId
            };

            // Swap out issue keys in each sprint with issue objects
            formattedSprint.issues = _.map(sprint.issues, function (issueKey) {
                return issueMap[issueKey];
            });

            // Descoped issues can creep in as undefineds
            formattedSprint.issues = _.compact(formattedSprint.issues);

            // Calculate Issue Nav and Sprint Report URLs
            formattedSprint.issueNavUrl = UrlFormat.getUrlForIssues(formattedSprint.issues);
            if (sprint.sprintId === 'original') {
                formattedSprint.sprintName = AJS.I18n.getText('gh.rapid.charts.scopeburndown.beforeinitialsprint');
            } else {
                formattedSprint.sprintName = sprint.sprintName;
                formattedSprint.sprintReportUrl = UrlFormat.prependBaseUrl('/secure/RapidBoard.jspa?rapidView=' + issueData.rapidViewId + '&view=reporting&chart=sprintRetrospective&sprint=' + sprint.sprintId);
            }

            if (sprint.startTime) {
                formattedSprint.formattedStartTime = moment(sprint.startTime).format(DATE_FORMAT);
            }

            if (sprint.endTime) {
                formattedSprint.formattedEndTime = moment(sprint.endTime).format(DATE_FORMAT);
            }

            // Format work completed value
            formattedSprint.formatedWorkCompleted = issueData.estimationStatistic.formatter.formatFull(sprint.workCompleted);

            return formattedSprint;
        });

        return issueData;
    }

    /**
     * This function is used for debugging only.
     * It groups all simplified change events by issue, so that the entire history of a single issue can be easily
     * traced.
     *
     * @param {Object} raw chart data
     * @returns {Object} issue history mapped by issue key
     */
    function getIssueHistory(data) {
        var simplifiedChanges = simplifyChanges(data);
        var issues = {};
        _.each(simplifiedChanges, function addChangeToIssueHistory(change) {
            var history = issues[change.key] = issues[change.key] || [];
            history.push(change);
        });
        return issues;
    }

    var ScopeBurndownBySprintTransformer = {

        getScopeBySprintData: getScopeBySprintData,
        getIssuesBySprintData: getIssuesBySprintData,
        getEstimationStatisticFormatter: getEstimationStatisticFormatter,

        // For debugging
        _getIssueHistory: getIssueHistory
    };

    GH.Reports = GH.Reports || {};
    GH.Reports.ScopeBurndownBySprintTransformer = ScopeBurndownBySprintTransformer;
})(_, moment, AJS.$, d3, GH.Ajax, GH.RapidViewConfig, GH.NumberFormat, GH.TimeFormat, GH.Util, GH.UrlFormat);