/* global c3: true */
/**
 * The scrubber component for the control chart.
 *
 * Adds:
 *  - a vertical line positioned next to the mouse
 *  - a dialog indicating the current time the mouse indicates or the brush range (if dragging)
 *  - a ControlChartBrush used to drill down into the chart.
 */
GH.Reports.BurnupChartScrubber = function () {

    var scrubber = c3.scrubber().elementClass('ghx-scrubber').shouldHideScrubber(function () {
        return GH.Reports.BurnupChartDialog.isVisible();
    });

    // use a brush to detect drag events and hide the dialog when the user clicks away.
    var brush = c3.brush().elementClass('ghx-chart-brush').onDrag(function () {
        // hide the dialog
        GH.Reports.BurnupChartDialog.hide();
    });

    /**
     * Disable the scrubber
     */

    function disableScrubber() {
        GH.Reports.BurnupChartScrubberDialog.disable();
    }

    /**
     * Enable the scrubber
     */
    function enableScrubber() {
        GH.Reports.BurnupChartScrubberDialog.enable();
    }

    /**
     * Show the scrubber dialog.
     */
    function showScrubberDialog() {
        GH.Reports.BurnupChartScrubberDialog.show();
    }

    /**
     * Hide the scrubber dialog.
     */
    function hideScrubberDialog() {
        GH.Reports.BurnupChartScrubberDialog.hide();
    }

    /**
     * Hide the scrubber line.
     */
    function hideScrubber() {
        scrubber.hideScrubber();
    }

    function cancelDrag() {
        brush.cancelDrag();
    }

    function getStatForTime(series, time) {
        //loop backwards looking for the first event before target time.
        // that event is the 'current' value for time.
        for (var i = series.length - 1; i >= 0; i--) {
            if (series[i].timestamp <= time) {
                return series[i].value;
            }
        }

        return series[0].value;
    }

    /**
     * Refresh the dialog:
     *
     *  - Update the position of the dialog to be beside the mouse
     *  - Update the content of the dialog to display either the range (if dragging) or the current time
     *    indicated by the scrubber.
     */
    function refreshScrubberDialog() {
        var mouse = this.mouseCoordinates();
        var invert = this.xScale().invert;

        var chartData = this.data();

        if (mouse) {
            var time = invert(mouse[0]);
            GH.Reports.BurnupChartScrubberDialog.refresh({
                time: chartData.formatDate(time),
                scope: chartData.formatStatistic(getStatForTime(this.data().scope, time)),
                completedWork: chartData.formatStatistic(getStatForTime(this.data().work, time))
            });
        }
    }

    /**
     * Initialise the dialog:
     *
     *  - Add the scrubber and brush components.
     *  - Initialise the scrubber dialog.
     *  - Bind to mouse events to update scrubber dialog.
     */
    function initialise() {

        var selection = this.selection();

        scrubber.parent(this);
        scrubber.interactiveLayer(this);
        scrubber(selection);

        brush.parent(this);
        brush.interactiveLayer(this);
        brush(selection);

        GH.Reports.BurnupChartScrubberDialog.init(scrubber.elements()[0]);

        var self = this;

        // Replace the dragging dialog with the hover dialog when done dragging
        //brush.onEndDrag(_.bind(refreshScrubberDialog, self));

        this.mouseover(function () {
            showScrubberDialog.call(self);
        });

        this.mousemove('ccscrubber', function () {
            refreshScrubberDialog.call(self);
        });

        this.mouseout(function () {
            hideScrubberDialog.call(self);
        });
    }

    return c3.component().extend(_.once(initialise)).extend(cancelDrag) // cancel drag if chart is updated
    .extend({
        hideScrubber: hideScrubber,
        disableScrubber: disableScrubber,
        enableScrubber: enableScrubber
    });
};