(function (_, moment, jq, d3, Ajax, RapidViewConfig, NumberFormat, TimeFormat, Util, UrlFormat, ChartUtils) {
    'use strict';

    /**
     * Transform sprint and scope data to a form that the chart can render.
     *
     * @param {Object} input
     * @returns {Object} transformed
     */
    function generateChartData(input) {

        var endTime = _calculateEndTime(input);
        var plots = _generatePlots(input, endTime);

        // put all the chart data in the input object
        input.chartData = {
            now: input.now,
            start: input.startTime,
            startKeys: plots.startKeys,

            // use actual completion date of the sprint if it's done, or else use the sprint's set end date.
            end: endTime,
            sprintCompleted: input.completeTime ? true : false, // if completeTime is defined then the sprint is complete.
            endKeys: plots.endKeys,
            nonWorkPeriods : _buildNonWorkPeriods(input.workRateData),
            scope : plots.scope,
            work : plots.done,
            statisticField: input.statisticField,
            rapidViewId: input.rapidViewId,
            // number of events that had estimates on them.  if this is 0 then we have no interesting data to display
            estimatedIssueEvents: plots.scope.filter(function (change) { return change.value > 0;}).length,
            issueToSummary: input.issueToSummary
        };

        input.chartData.formatDate = function (timestamp) {
            return moment.utc(timestamp).format('lll');
        };

        input.chartData.formatShortDate = function (timestamp) {
            // if the date range is over a year
            if (moment.duration((moment.utc(input.chartData.end).diff(input.chartData.start))).years() > 0) {
                // e.g. Jan 2015
                return moment.utc(timestamp).format('MMM YYYY');
            }
            //e.g. Jan 7
            return moment.utc(timestamp).format('MMM D');
        };

        input.chartData.formatStatistic = function(value) {
            return ChartUtils.renderStatisticText(value, false, input.chartData.statisticField.renderer);
        };

        input.chartData.formatEventType = function(type) {
            switch (type) {
                case 'estimate update':
                    return AJS.I18n.getText('gh.rapid.charts.burnup.event.estimate.updated');

                case 'issue done':
                    return AJS.I18n.getText('gh.rapid.charts.burnup.event.issue.done');

                case 'issue not done':
                    return AJS.I18n.getText('gh.rapid.charts.burnup.event.issue.reopened');

                case 'issue removed':
                    return AJS.I18n.getText('gh.rapid.charts.burnup.event.issue.removed');

                case 'issue added':
                    return AJS.I18n.getText('gh.rapid.charts.burnup.event.issue.added');

                case 'start event':
                    return AJS.I18n.getText('gh.rapid.charts.burnup.event.sprint.started');

                case 'end event':
                    return AJS.I18n.getText('gh.rapid.charts.burnup.event.sprint.finished');

                default:
                    return type;
            }
        };

        return input;
    }

    function _calculateEndTime(input) {
        if (input.completeTime) {
            // sprint is completed - use the time it was completed.
            return input.completeTime;
        } else if (input.now > input.endTime) {
            // sprint open and current time after the planned end time - use current time
            return input.now;
        } else {
            // sprint open and current time is before scheduled end time - use end time
            return input.endTime;
        }
    }

    function _processChanges(changes, sprintState) {

        // the lists of changes to scope and completed (work) that we want to show on the graph.  Not every change gets
        //   graphed (e.g. updates to issues after they were removed from the sprint).
        var scope = [];
        var done = [];

        // We'll apply each change to the sprint state in order to it to build up the scope and completed work values
        //   for each point in the sprint.
        //var sprintState = {};

        changes.forEach(function(change) {

            // ensure we have a place holder issue to track the change.

            // changes can arrive in any order (i.e. update estimate before add to sprint) so we need to do this
            //   every time
            var issue = sprintState[change.key];

            if (!issue) {
                issue = { key: change.key, prevEstimate: 0, estimate: 0, inSprint: false, done: false };
                sprintState[change.key] = issue;
            }

            // track if updates require the scope or done lines to be updated.
            var scopeDirty = false;
            var doneDirty = false;

            // apply the change to the model.
            switch (change.type) {
                case 'estimate update':
                    if (issue.estimate !== change.newEstimate) {
                        issue.prevEstimate = issue.estimate;
                        issue.estimate = change.newEstimate;

                        // if the issue is in the sprint estimate changes always make the scope dirty
                        scopeDirty = issue.inSprint && !issue.isExcluded;
                        // update done if the is currently marked as done and the issue is in the sprint
                        doneDirty = issue.inSprint && !issue.isExcluded && issue.done;
                    }
                    break;

                case 'issue done':
                    if (!issue.done) {
                        issue.done = true;

                        // update done if the issue is in the sprint
                        doneDirty = issue.inSprint && !issue.isExcluded;
                    }
                    break;

                case 'issue not done':
                    if (issue.done) {
                        issue.done = false;

                        // Affect sprint scope for excluded issue.
                        if (issue.isExcluded) {
                            change.isInclusion = true;
                            issue.isExcluded = false;
                            scopeDirty = issue.inSprint;

                        // Affect sprint work done for included issues.
                        } else {
                            doneDirty = issue.inSprint;
                        }
                    }
                    break;

                case 'issue removed':
                    if (issue.inSprint) {
                        issue.inSprint = false;

                        // update scope
                        scopeDirty = !issue.isExcluded;
                        // update done if the issue was marked as done
                        doneDirty = !issue.isExcluded && issue.done;
                    }
                    break;

                case 'issue added':
                    if (!issue.inSprint) {
                        issue.inSprint = true;

                        // always show an event for adding an item to the sprint, even if it's estimate is 0.
                        scopeDirty = true;
                        // update done marked as done
                        doneDirty = issue.done;
                    }
                    break;
            }

            // raise updates to the chart
            if (scopeDirty) {
                scope.push(_chartEvent('scope', change.timestamp, _sum(sprintState, false), change, issue));
            }

            if (doneDirty) {
                done.push(_chartEvent('done', change.timestamp, _sum(sprintState, true), change, issue));
            }
        });

        return {scope: scope, done: done};
    }

    function _startEvent(line, timestamp, value) {
        return _chartEvent(line, timestamp, value, {type: 'start event'}, {});
    }

    function _endEvent(line, timestamp, value) {
        return _chartEvent(line, timestamp, value, {type: 'end event'}, {});
    }

    function _chartEvent(line, timestamp, runningTotal, change, issue) {
        change = change || {};
        issue = issue || {};


        var to = 0;
        var from = 0;
        if (change.type === 'issue removed') {
            to = 0;
            from = issue.estimate;
        } else if (change.type === 'issue added') {
            to = issue.estimate;
            from = 0;
        } else if (change.type === 'issue not done') {
            //aka reopening a closed issue.

            // If this is an "inclusion" event, (ie. an excluded issue is reopened),
            // increase the scope.
            // Otherwise decrease work done.
            if (change.isInclusion) {
                to = issue.estimate;
                from = 0;
            } else {
                to = 0;
                from = issue.estimate;
            }
        } else {
            to = issue.estimate;
            from = issue.prevEstimate;
        }

        return {
            timestamp: timestamp,
            value: runningTotal,
            change: {
                type: change.type,
                line: line,
                key: issue.key,
                fromValue: from,
                toValue: to
            }
        };
    }

    function _sum(sprintState, doneOnly) {
        var total = 0;
        for (var issueKey in sprintState) {
            if (sprintState.hasOwnProperty(issueKey)) {

                var issue = sprintState[issueKey];
                if (issue.inSprint && !issue.isExcluded) {
                    if ( (doneOnly && issue.done) || !doneOnly) {
                        total += issue.estimate;
                    }
                }
            }
        }

        return total;
    }

    // converts a list of issue deltas into the total done and total scope at each timestamp.
    function _generatePlots(data, end) {

        // flatten the nested object of changes into a simplified list, and sort it by timestamp.
        var changes = _flattenChangeList(data.changes).sort(function (a, b) {
            return a.timestamp > b.timestamp ? 1 : (a.timestamp === b.timestamp) ? 0 : -1;
        });

        var firstStateTime = parseInt(data.startTime);

        var lastStateTime = data.now > end ? end : data.now;

        // process all the changes up to the start of the sprint
        var sprintState = {};
        _processChanges(changes.filter(function(c) {
            return c.timestamp <= firstStateTime;
        }), sprintState);

        // Exclude issues which are completed at the start of the sprint.
        for (var key in sprintState) {
            if (sprintState.hasOwnProperty(key)) {
                var issue = sprintState[key];
                if (issue.done) {
                    issue.isExcluded = true;
                }
            }
        }

        var startKeys = _sortedKeysInSprint(sprintState);

        // generate the events for starting scope and done lines
        var scope = [_startEvent('scope', firstStateTime, _sum(sprintState, false))];
        var done = [_startEvent('done', firstStateTime, 0, 'start event')]; // done is always 0 at the start.

        // process all the remaining events.
        var finalEvents = _processChanges(changes.filter(function(c) {
            return c.timestamp > firstStateTime && c.timestamp <= lastStateTime;
        }), sprintState);

        var endKeys = _sortedKeysInSprint(sprintState);

        // add them to our scope and done lines
        scope = scope.concat(finalEvents.scope);
        done = done.concat(finalEvents.done);

        // add a point at the end of the chart
        scope.push(_endEvent('scope', lastStateTime, scope[scope.length -1].value));
        done.push(_endEvent('done', lastStateTime, done[done.length -1].value));

        return {scope: scope, done: done, startKeys: startKeys, endKeys: endKeys};
    }

    function _sortedKeysInSprint(sprintState) {
        return Object.keys(sprintState)
            .filter(function(key) {
                var issue = sprintState[key];
                return issue.inSprint && !issue.isExcluded;
            }).sort(_smartIssueComparitor);
    }

    // Sort based first on the project key part, then on the numeric part.
    function _smartIssueComparitor(a, b) {
        if (a === b) {
            return 0;
        }

        // Parse the keys.  Put malformed keys first.
        var sortPattern = /(\w+)-(\d+)/;
        var aParts = sortPattern.exec(a);
        if (!aParts) {
            return -1;
        }
        var bParts = sortPattern.exec(b);
        if (!bParts) {
            return 1;
        }

        // Sort on project key part first.
        var aProject = aParts[1];
        var bProject = bParts[1];
        if (aProject !== bProject) {
            return aProject > bProject ? 1 : -1;
        }

        // Sort on numeric part second.
        var aNum = parseInt(aParts[2], 10);
        var bNum = parseInt(bParts[2], 10);
        return aNum === bNum ? 0 : (aNum > bNum ? 1 : -1);
    }


    function _flattenChangeList(changes) {

        var newChanges = [];

        var transform = function (change) {

            // include changes to the done value
            if ( change.column ) {

                if (change.column.notDone) {
                    // use column.notDone rather than column.done to keep consistency with the burnDown report
                    newChanges.push({
                        type: 'issue not done',
                        key: change.key,
                        timestamp: parseInt(timestamp)
                    });
                } else if (!change.column.notDone) {
                    newChanges.push({
                        type: 'issue done',
                        key: change.key,
                        timestamp: parseInt(timestamp)
                    });
                }
            }

            // include updates to estimate values
            if (change.statC && change.statC.newValue) {
                newChanges.push({
                    type: 'estimate update',
                    key: change.key,
                    timestamp: parseInt(timestamp),
                    newEstimate: change.statC.newValue > 0 ? change.statC.newValue : 0
                });
            }

            // include issues being removed
            if (change.added === false) {
                newChanges.push({
                    type: 'issue removed',
                    key: change.key,
                    timestamp: parseInt(timestamp)
                });
            }

            // include issues being added
            if (change.added === true) {
                newChanges.push({
                    type: 'issue added',
                    key: change.key,
                    timestamp: parseInt(timestamp)
                });
            }
        };

        for (var timestamp in changes) {
            if (changes.hasOwnProperty(timestamp)) {
                changes[timestamp].forEach(transform);
            }
        }

        return newChanges;
    }

    function _buildNonWorkPeriods(workRateData) {
        var nonWorkPeriods = [];

        workRateData.rates.forEach(function (workRateItem) {
            if (workRateItem.rate < 1) {
                nonWorkPeriods.push({
                    start: workRateItem.start,
                    end: workRateItem.end
                });
            }
        });

        return nonWorkPeriods;
    }

    var SprintBurnupTransformer = {
        getBurnupModelFromScopeBurndownModel: generateChartData
    };

    GH.Reports = GH.Reports || {};
    GH.Reports.SprintBurnupTransformer = SprintBurnupTransformer;
}(_, moment, AJS.$, d3, GH.Ajax, GH.RapidViewConfig, GH.NumberFormat, GH.TimeFormat, GH.Util, GH.UrlFormat, GH.ChartUtils));
