/**
 * Control Chart Controller.
 *
 * This controller has several functions that are required to work correctly within the 'framework' provided by GH.ReportController.
 *
 * These lifecycle functions are called by the GH.ReportController to show/hide/init the chart this controller manages
 *
 * - ControlChartController.init
 * - ControlChartController.show
 * - ControlChartController.hide
 *
 * These methods deal with serializing/deserializing the controller's state to and from a normalized state that can be represented in the URL :
 *
 *  - ControlChartController.isApplicable
 *  - ControlChartController.toUrl
 *  - ControlChartController.updateInternalFromNormalized
 *  - ControlChartController.getNormalizedFromInternal
 *  - ControlChartController.getNormalizedFromUrl
 *
 */
define('jira-agile/rapid/ui/chart/v2/controlchart/control-chart-controller', ['require'], function (require) {
    var ControlChartController = {};

    // REQUIRES
    var _ = require('underscore');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    var ControlChartView = require('jira-agile/rapid/ui/chart/v2/controlchart/control-chart-view');
    var ControlChartDataService = require('jira-agile/rapid/ui/chart/v2/controlchart/control-chart-data-service');
    var ControlChartRefinementsController = require('jira-agile/rapid/ui/chart/v2/controlchart/control-chart-refinements-controller');
    var ControlChartViewingOptionsController = require('jira-agile/rapid/ui/chart/v2/controlchart/control-chart-viewing-options-controller');

    var controlChartDataService = new ControlChartDataService();

    // GLOBALS... FIX ME
    var ChartTimeFrameModel;
    var ControlChartRefinementsModel;
    var ControlChartViewingOptionsModel;
    var ChartTimeFrames;
    var PromiseKeeper;
    var RapidBoardState;
    var RapidViewConfig;

    GlobalEvents.on('pre-initialization', function () {
        ChartTimeFrameModel = GH.Reports.ChartTimeFrameModel;
        ControlChartRefinementsModel = GH.Reports.ControlChartRefinementsModel;
        ControlChartViewingOptionsModel = GH.Reports.ControlChartViewingOptionsModel;
        ChartTimeFrames = GH.ChartTimeFrames;
        PromiseKeeper = GH.PromiseKeeper;
        RapidBoardState = GH.RapidBoard.State;
        RapidViewConfig = GH.RapidViewConfig;
    });

    ControlChartController.id = 'controlChart';
    ControlChartController.SETTINGS_KEY = 'chart.config.controlChart';
    ControlChartController.displayed = false;
    ControlChartController.rapidViewData = null;
    ControlChartController.initPhasePromise = null;
    ControlChartController.filterOptions = null;
    ControlChartController.firstTimeLoadingData = true;

    /**
     * @type module:jira-agile/rapid/analytics-tracker
     */
    ControlChartController.analytics = new AnalyticsTracker('gh.report.controlChart');

    ControlChartController.setRapidView = function (rapidViewData) {
        ControlChartController.rapidViewData = rapidViewData || {};
    };

    /**
     * Notify this controller that filterOptions have changed
     */
    ControlChartController.update = function () {
        ControlChartController.saveFilterOptions();
        RapidBoardState.pushState();
        ControlChartController.loadAndProcess();
    };

    /**
     * Shows the chart. Loads the chart's data and renders the chart.
     */
    ControlChartController.show = function () {
        var controller = ControlChartController;
        var filterOptions = controller.getFilterOptions();

        if (!controller.initPhasePromise) {
            controller.initPhasePromise = RapidViewConfig.fetchConfiguration(controller.rapidViewData.id).done(function (rapidViewConfigData) {
                // set the available refinements on the model
                filterOptions.refinements.setAvailable(_.pick(rapidViewConfigData, 'columns', 'swimlanes', 'quickFilters'));
            });
        }

        controller._dataPromiseKeeper = new PromiseKeeper();

        controller.initPhasePromise.done(function () {

            // load the chart data and render the view
            controller.displayed = true;

            var currentFilterOptions = ControlChartController.getFilterOptions();
            var isSwimlaneStrategyJQL = RapidViewConfig.currentData.data.swimlaneStrategy === 'custom';

            //select all the swimlanes if the strategy is not JQL so that all issues are displayed
            if (!isSwimlaneStrategyJQL) {
                filterOptions.refinements.selected.swimlaneIds = _.pluck(filterOptions.refinements.available.swimlanes, 'id');
            }

            RapidBoardState.pushState();

            // render the container
            ControlChartView.init();

            ControlChartRefinementsController.start(filterOptions.refinements);

            ControlChartViewingOptionsController.start(filterOptions.viewingOptions);

            ControlChartController.loadAndProcess();

            // Analytics
            var action = 'show.' + (ControlChartController.rapidViewData.sprintSupportEnabled ? 'scrum' : 'kanban');
            ControlChartController.analytics.trigger(action);
        });
    };

    /**
     * Hides the chart.
     */
    ControlChartController.hide = function () {
        ControlChartController.displayed = false;
        if (ControlChartController._dataPromiseKeeper) {
            ControlChartController._dataPromiseKeeper.reset();
        }
        ControlChartView.destroy();
        controlChartDataService.clearCache();
    };

    /**
     * Requests data from the ControlChartDataService with the current filterOptions applied,
     * then renders the controlChart with this data.
     */
    ControlChartController.loadAndProcess = function () {
        ControlChartView.showSpinner();

        var appliedFilterOptions = ControlChartController.getFilterOptions();

        // load the data according to the selected swimlanes and quickfilters
        var dataPromise = controlChartDataService.get(ControlChartController.rapidViewData.id, appliedFilterOptions);

        // Load and process chart data
        return ControlChartController._dataPromiseKeeper.add(dataPromise).done(function (data) {
            // process only of the chart is still visible
            if (ControlChartController.displayed) {
                // Toggle blank state messages
                if (data.issues.length > 0) {
                    ControlChartView.hideBlankState();
                } else {
                    ControlChartView.showBlankState(appliedFilterOptions);
                    ControlChartController.analytics.trigger('blank');
                }

                ControlChartController.processChartData(data);
            }
        }).always(function (data) {
            if (data !== 'abort') {
                ControlChartView.hideSpinner();
            }
        });
    };

    /**
     * Get the object that represents the current state of all the options the user can set for the control chart.
     * On the first call, the object is retrieved from localStorage.
     *
     * @returns {Object}
     */
    ControlChartController.getFilterOptions = function () {
        if (!ControlChartController.filterOptions) {
            var storedOptions = RapidBoardState.getBoardSetting(RapidBoardState.getRapidViewId(), ControlChartController.SETTINGS_KEY, {});
            ControlChartController.filterOptions = {
                timeFrame: new ChartTimeFrameModel().fromNormalized(storedOptions),
                refinements: new ControlChartRefinementsModel().fromNormalized(storedOptions),
                viewingOptions: new ControlChartViewingOptionsModel().fromNormalized(storedOptions)
            };
        }
        return ControlChartController.filterOptions;
    };

    ControlChartController.saveFilterOptions = function () {
        RapidBoardState.setBoardSetting(ControlChartController.SETTINGS_KEY, ControlChartController.getNormalizedFromInternal());
    };

    /**
     * Processes the returned chart data.
     */
    ControlChartController.processChartData = function (data) {
        // setup the time frame model
        var earliestDate = new Date(data.earliestTime);
        var currentDate = new Date(data.currentTime);

        var filterOptions = ControlChartController.getFilterOptions();
        filterOptions.timeFrame.setBounds(earliestDate, currentDate);

        // override the default timeframe to all time if the date range is less than 3 month
        if (ControlChartController.firstTimeLoadingData) {
            var dateRangeInDays = (currentDate - earliestDate) / ChartTimeFrames.day;
            ControlChartController.firstTimeLoadingData = false;
            if (filterOptions.timeFrame.isDefault() && dateRangeInDays < 90) {
                filterOptions.timeFrame.setAllTime();
                RapidBoardState.pushState();
            }
        }

        // render the chart
        ControlChartView.render(data, filterOptions.timeFrame);
    };

    /**
     * URL State - This is code so that our new chart can remain compatible with the old
     */

    /**
     * Is the chart available for this type of board
     * @param {Object} rapidViewConfig
     * @returns {boolean}
     */
    ControlChartController.isApplicable = function (rapidViewConfig) {
        return true;
    };

    /**
     * Return a normalized representation of the chart state.
     * Must contain the chart ID.
     *
     * @returns {Object}
     */
    ControlChartController.getNormalizedFromInternal = function () {
        var filterOptions = ControlChartController.getFilterOptions();
        return _.extend({
            chart: ControlChartController.id
        }, filterOptions.timeFrame.toNormalized(), filterOptions.refinements.toNormalized(), filterOptions.viewingOptions.toNormalized());
    };

    /**
     * Update the chart state from a normalized representation.
     * Saves the state to localStorage as part of the update.
     *
     * @param {Object} normalized
     */
    ControlChartController.updateInternalFromNormalized = function (normalized) {
        var filterOptions = ControlChartController.getFilterOptions();
        filterOptions.timeFrame.fromNormalized(normalized);
        filterOptions.refinements.fromNormalized(normalized);
        filterOptions.viewingOptions.fromNormalized(normalized);
        ControlChartController.saveFilterOptions();
    };

    /**
     * Convert a normalized representation of the chart state into url query params.
     *
     * @param {Object} normalized
     * @returns {Object}
     */
    ControlChartController.toUrl = function (normalized) {
        return {
            days: normalized.days,
            from: normalized.from,
            to: normalized.to,
            column: normalized.columnIds,
            swimlane: normalized.swimlaneIds,
            quickfilter: normalized.quickFilterIds,
            nwd: normalized.includeNonWorkingDays
        };
    };

    /**
     * Convert url query params into a normalized representation of the chart state.
     * Types must be cast correctly from string otherwise the URL manager will get confused when comparing equality.
     *
     * @param {Object} params
     * @returns {Object}
     */
    ControlChartController.getNormalizedFromUrl = function (params) {
        function castToArrayOfNumbers(array) {
            return _.isArray(array) ? _.map(array, Number) : [Number(array)];
        }

        function castToBoolean(string) {
            return string === true.toString();
        }

        var normalized = {};
        if ('days' in params) {
            normalized.days = isNaN(params.days) ? params.days : Number(params.days);
        }
        if ('to' in params) {
            normalized.to = params.to;
        }
        if ('from' in params) {
            normalized.from = params.from;
        }
        if ('column' in params) {
            normalized.columnIds = castToArrayOfNumbers(params.column);
        }
        if ('swimlane' in params) {
            normalized.swimlaneIds = castToArrayOfNumbers(params.swimlane);
        }
        if ('quickfilter' in params) {
            normalized.quickFilterIds = castToArrayOfNumbers(params.quickfilter);
        }
        if ('nwd' in params) {
            normalized.includeNonWorkingDays = castToBoolean(params.nwd);
        }
        return normalized;
    };

    return ControlChartController;
});