/**
 * Displays a dialog relating to an issue or cluster of issues.
 */
GH.Reports = GH.Reports || {};

GH.Reports.ControlChartDialog = function () {
    var DIALOG_ID = 'control-chart-issues';
    var SCROLL_CONTAINER = '#ghx-report-scrollarea > .aui-page-panel';
    var durationFormatter = GH.TimeFormat.formatShortDuration;
    var dialog;

    GH.InlineDialog.preventCloseOnClick(DIALOG_ID);

    function formatIssue(issue) {
        _.each(issue.columns, function (column) {
            column.formattedDuration = durationFormatter(column.duration);
        });
        issue.formattedDuration = durationFormatter(issue.duration);
        return issue;
    }

    function getClusterStats(issues) {
        var durationAccessor = function durationAccessor(d) {
            return d.duration;
        };
        return [{
            label: AJS.I18n.getText('gh.rapid.chart.controlchart.dialog.cycletime.max'),
            value: durationFormatter(d3.max(issues, durationAccessor))
        }, {
            label: AJS.I18n.getText('gh.rapid.chart.controlchart.dialog.cycletime.average'),
            value: durationFormatter(d3.mean(issues, durationAccessor))
        }, {
            label: AJS.I18n.getText('gh.rapid.chart.controlchart.dialog.cycletime.min'),
            value: durationFormatter(d3.min(issues, durationAccessor))
        }];
    }

    function renderIssue(issue) {
        return GH.tpl.reports.controlChart.renderDialogIssueContent({
            issue: formatIssue(issue),
            url: GH.Ajax.CONTEXT_PATH + "/browse/" + encodeURIComponent(issue.key)
        });
    }

    function renderCluster(issues) {
        var issueKeys = _.pluck(issues, 'key');
        return GH.tpl.reports.controlChart.renderDialogClusterContent({
            issueCount: issues.length,
            stats: getClusterStats(issues),
            url: GH.Ajax.CONTEXT_PATH + "/secure/GHViewInNavigator.jspa?issueKeys=" + encodeURIComponent(issueKeys.join(","))
        });
    }

    function show(trigger, issues) {
        // use d3 to manipulate the classes since jQuery/ie don't support addClass/classList for svg elements.
        var triggerElement = d3.select(trigger);
        triggerElement.classed('active', true);

        var clustered = _.isArray(issues);
        dialog = GH.SideInlineDialog.create(DIALOG_ID, trigger, function ($content, $trigger, showPopup) {
            $content.closest('.aui-inline-dialog').addClass('ghx-inline-dialog');
            $content.html(clustered ? renderCluster(issues) : renderIssue(issues));
            showPopup();
        }, {
            addActiveClass: false,
            container: SCROLL_CONTAINER,
            fadeTime: 0,
            noBind: true,
            cacheContent: false, // don't cache the dialog content
            hideDelay: 60000, // set longer timeout (default is 10 seconds)
            persistent: true,
            hideCallback: function hideCallback() {
                triggerElement.classed('active', false);
            }
        });
        dialog.show();

        return dialog;
    }

    function isVisible() {
        return dialog ? dialog.is(":visible") : false;
    }

    return {
        /**
         * Create and show a dialog for an issue or cluster of issues
         *
         * @param {Element} trigger - element to position the dialog next to
         * @param {issue[]|issue} issues - renders different content depending on whether
         * a single issue or an array of issues are passed
         * @return {InlineDialog}
         */
        show: show,
        /**
         * Hides dialog if it is showing
         */
        hide: function hide() {
            if (dialog) {
                dialog.hide();
            }
        },
        /**
         * Test if there is an instance of this dialog already visible.
         */
        isVisible: isVisible
    };
}();