/**
 * A dialog which resembles Tipsy and binds to a scrubber line
 */
GH.Reports.ControlChartScrubberDialog = function () {
    /**
     * @typedef {object} DialogContents
     * @property {number} [time]
     * @property {number} [startTime]
     * @property {number} [endTime]
     * @property {number} [rollingAverage]
     * @property {number} [standardDeviation]
     */

    var dialog;
    var $scrubber;
    var isEnabled = true;

    var DIALOG_ID = 'scrubber-dialog';
    var SCROLL_CONTAINER = '#ghx-chart-group';
    var SCRUBBER_DIALOG_OPTS = {
        container: SCROLL_CONTAINER,
        fadeTime: 0,
        noBind: true,
        cacheContent: false,
        hideDelay: 60000,
        width: 230,
        closeOthers: false,
        persistent: true,
        addActiveClass: false
    };

    /**
     * Format a date appropriate for a short timeframe
     *
     * @param {Date} date The date
     * @returns {String} The formatted date
     */
    function dateFormatterShort(date) {
        return moment(date).format('LLL');
    }

    /**
     * Format a date appropriate for a long timeframe
     *
     * @param {Date} date The date
     * @returns {String} The formatted date
     */
    function dateFormatterLong(date) {
        return moment(date).format('LL');
    }

    /**
     * Initialise the dialog
     *
     * @param {jQuery} $scrubberElement the scrubber line which will be bound to by the dialog.
     */
    function init($scrubberElement) {
        $scrubber = $scrubberElement;
        dialog = GH.SideInlineDialog.create(DIALOG_ID, $scrubberElement, function initDialog($content, $trigger, showPopup) {
            $content.closest('.aui-inline-dialog').addClass('ghx-inline-dialog ghx-scrubber-dialog');
            $content.addClass('ghx-scrubber-contents');
            $content.next("#arrow-" + DIALOG_ID).addClass('ghx-arrow-scrubber-dialog');
            showPopup();
        }, SCRUBBER_DIALOG_OPTS);

        preventDialogHideOnClick();
    }

    /**
     * Disable the scrubber dialog
     */
    function disableScrubberDialog() {
        isEnabled = false;
    }

    /**
     * Enable the scrubber dialog
     */
    function enableScrubberDialog() {
        isEnabled = true;
    }

    /**
     * Show the dialog
     */
    function show() {
        if (!GH.Reports.ControlChartDialog.isVisible() && isEnabled) {
            dialog.show();
        }
    }

    /**
     * By default InlineDialogs will close when a region not enclosed by the dialog is clicked, this method disables
     * this behaviour.
     */
    function preventDialogHideOnClick() {
        GH.InlineDialog.registerInlineDialogHacks(DIALOG_ID, {
            validateClickToCloseFn: function validateClickFn() {
                return false;
            }
        });
    }

    /**
     * Hide the scrubber dialog.
     */
    function hide() {
        if (dialog) {
            dialog.hide();
        }
    }

    /**
     * Refresh the dialog after a mousemove. This will reposition the dialog relative to $scrubber,
     * update the content inside the dialog and unhide the dialog if it is currently invisible.
     *
     * @param {DialogContents} dialogContents
     */
    function refresh(dialogContents) {
        // reshow the dialog if it is hidden
        unhideDialog();

        // update the content
        updateContent(dialogContents);

        // reposition
        dialog.refresh();
    }

    /**
     * Updates the text inside the dialog to reflect the time the scrubber line indicates.
     *
     * @param {DialogContents} dialogContents
     */
    function updateContent(dialogContents) {
        var content;
        if (dialogContents.time) {
            content = GH.tpl.reports.controlChart.renderScrubberDialogContent({
                indicatedTime: formatDate(dialogContents.time),
                rollingAverage: GH.Util.roundToDecimalPlaces(dialogContents.rollingAverage, 1),
                standardDeviation: GH.Util.roundToDecimalPlaces(dialogContents.standardDeviation, 1)
            });
        } else {
            content = GH.tpl.reports.controlChart.renderScrubberBrushingDialogContent({
                startTime: formatDate(dialogContents.startTime),
                endTime: formatDate(dialogContents.endTime)
            });
        }
        getElement().find('.ghx-scrubber-contents').html(content);
    }

    /**
     * Formats the current time indicated by the scrubber line appropriately according to
     * the current timeframe of the chart
     *
     * @param {Date} time the current time to show on the dialog
     * @returns {String} The textual representation of time
     */
    function formatDate(time) {
        var timeFrame = GH.Reports.ControlChartController.getFilterOptions().timeFrame;
        var days = timeFrame.getTimeFrameDuration() / GH.ChartTimeFrames.day;

        if (days > 30) {
            time = dateFormatterLong(time);
        } else {
            time = dateFormatterShort(time);
        }

        return time;
    }

    /**
     * Gets the dialog element
     *
     * @returns {jQuery} the dialog
     */
    function getElement() {
        return AJS.$("#inline-dialog-" + DIALOG_ID);
    }

    /**
     * Unhides the dialog if it currently exists but is not shown
     */
    function unhideDialog() {
        if (!isVisible()) {
            show();
        }
    }

    function isVisible() {
        return dialog ? dialog.is(":visible") : false;
    }

    return {
        /**
         * Initialises the dialog.
         *
         * @param {jQuery} $scrubberElement The scrubber element to bind the dialog to.
         */
        init: init,

        /**
         * Shows the scrubber dialog and creates it if it does not yet exist.
         */
        show: show,

        /**
         * Hides the dialog if it exists.
         */
        hide: hide,

        isVisible: isVisible,

        /**
         * Repositions the scrubber dialog next to the scrubber element.
         *
         * @param {Date} time The current time indicated by the scrubber element.
         */
        refresh: refresh,
        disable: disableScrubberDialog,
        enable: enableScrubberDialog
    };
}();