define('jira-agile/rapid/ui/chart/v2/releaseburndown/release-burndown-report-controller', ['require'], function (require) {
    // REQUIRES
    var _ = require('underscore');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    var ReleaseBurndownReportView = require('jira-agile/rapid/ui/chart/v2/releaseburndown/release-burndown-report-view');
    var ReleaseBurndownChartDataService = require('jira-agile/rapid/ui/chart/v2/releaseburndown/release-burndown-chart-data-service');
    var ReleaseBurndownChartTableDataService = require('jira-agile/rapid/ui/chart/v2/releaseburndown/release-burndown-chart-table-data-service');

    var releaseBurndownChartDataService = new ReleaseBurndownChartDataService();
    var releaseBurndownChartTableDataService = new ReleaseBurndownChartTableDataService();

    // GLOBALS... FIX ME
    var PromiseKeeper;
    var RapidBoardState;
    var VersionsDataService;

    GlobalEvents.on("pre-initialization", function () {
        PromiseKeeper = GH.PromiseKeeper;
        RapidBoardState = GH.RapidBoard.State;
        VersionsDataService = new GH.VersionsDataService();
    });

    function ReleaseBurndownReportController() {};
    ReleaseBurndownReportController.id = 'releaseBurndown';

    ReleaseBurndownReportController.prototype = {
        SETTINGS_KEY: 'chart.config.releaseBurndown',
        displayed: false,

        _model: null,
        _rapidViewData: null,
        _tracker: null,

        setRapidView: function setRapidView(rapidViewData) {
            this._rapidViewData = rapidViewData || {};
        },

        show: function show() {
            this.displayed = true;
            this._tracker = new AnalyticsTracker('gh.report.releaseburndown');

            RapidBoardState.pushState();

            // Initialise view
            var view = this._view = new ReleaseBurndownReportView();
            view.on('versionSelected', _.bind(function (versionId) {
                this._model.versionId = versionId;
                this.update();
            }, this)).on('optionChanged', _.bind(function (changes) {
                this._model.zeroBaseline = changes.zeroBaseline;
                this._tracker.trigger('zeroBaseline.' + (changes.zeroBaseline ? 'on' : 'off'));
                this.update();
            }, this));
            view.init(this._model);
            view.showSpinner();

            this._tracker.trigger('show');

            this._dataPromiseKeeper = new PromiseKeeper();

            // Get versions and update the view
            var versionsPromise = VersionsDataService.getSimpleVersions(this._rapidViewData.id);
            this._dataPromiseKeeper.add(versionsPromise).done(_.bind(function (versions) {
                if (!versions.length) {
                    this._model.versionId = null;
                    view.showBlankState(true, true);
                    this._tracker.trigger('blank');
                    view.hideSpinner();
                } else {
                    versions.reverse();
                    if (this._model.versionId == null || !VersionsDataService.versionExists(versions, this._model.versionId)) {
                        this._model.versionId = versions[0].id;
                    }
                    view.setVersions(versions);
                    this.update(true);
                }
            }, this));
        },

        hide: function hide() {
            this.displayed = false;
            if (this._dataPromiseKeeper) {
                this._dataPromiseKeeper.reset();
            }
            if (this._view) {
                this._view.destroy();
                this._view = null;
            }
            releaseBurndownChartDataService.clearCache();
            releaseBurndownChartTableDataService.clearCache();
        },

        update: function update(isAfterLoadingVersions) {
            if (isAfterLoadingVersions) {
                RapidBoardState.replaceState();
            } else {
                RapidBoardState.pushState();
            }
            this.saveState();

            if (this._model.versionId == null) {
                return;
            }

            var controller = this;

            // Get data and update chart
            controller._view.showSpinner();
            var dataPromise = releaseBurndownChartDataService.get(controller._rapidViewData.id, controller._model.versionId, {
                zeroBaseline: controller._model.zeroBaseline
            });

            return controller._dataPromiseKeeper.add(dataPromise).done(function (data) {
                // Toggle blankstate
                if (data.snapshot.estimatedIssueCount === 0) {
                    controller._view.showBlankState(false, true);
                    controller._tracker.trigger('blank');
                } else {
                    controller._view.hideBlankState();
                }
                controller._view.update(data);

                // Get issue table data
                var tableDataPromise = releaseBurndownChartTableDataService.get(controller._rapidViewData.id, controller._model.versionId, data);
                controller._dataPromiseKeeper.add(tableDataPromise).done(function (issueData) {
                    controller._view.updateTable(issueData);
                });
            }).always(function (data) {
                if (data !== 'abort') {
                    controller._view.hideSpinner();
                }
            });
        },

        saveState: function saveState() {
            RapidBoardState.setBoardSetting(this.SETTINGS_KEY, this.getNormalizedFromInternal());
        },

        loadState: function loadState() {
            this._model = RapidBoardState.getBoardSetting(RapidBoardState.getRapidViewId(), this.SETTINGS_KEY, {
                versionId: null,
                zeroBaseline: false
            });
        },

        /**
         * Is the chart available for this type of board
         *
         * @param {Object} rapidViewConfig
         * @returns {boolean}
         */
        isApplicable: function isApplicable(rapidViewConfig) {
            return rapidViewConfig.sprintSupportEnabled;
        },

        /**
         * Return a normalized representation of the chart state.
         *
         * @returns {Object}
         */
        getNormalizedFromInternal: function getNormalizedFromInternal() {
            if (this._model === null) {
                this.loadState();
            }
            return _.defaults(_.clone(this._model), {
                zeroBaseline: false
            });
        },

        /**
         * Update the chart state from a normalized representation.
         * Saves the state to localStorage as part of the update.
         *
         * @param {Object} normalized
         */
        updateInternalFromNormalized: function updateInternalFromNormalized(normalized) {
            this._model = normalized;
        },

        /**
         * Convert a normalized representation of the chart state into url query params.
         *
         * @param {Object} normalized
         * @returns {Object}
         */
        toUrl: function toUrl(normalized) {
            return {
                version: normalized.versionId !== null ? normalized.versionId : undefined,
                zeroBaseline: normalized.zeroBaseline ? true : undefined
            };
        },

        /**
         * Convert url query params into a normalized representation of the chart state.
         * Types must be cast correctly from string otherwise the URL manager will get confused when comparing equality.
         *
         * @param {Object} params
         * @returns {Object}
         */
        getNormalizedFromUrl: function getNormalizedFromUrl(params) {
            return {
                versionId: isNaN(params.version) ? null : +params.version,
                zeroBaseline: params.zeroBaseline === 'true'
            };
        }
    };

    return ReleaseBurndownReportController;
});