/* globals GH */
define('jira-agile/rapid/ui/chart/v2/releaseburndown/release-burndown-report-view', ['require'], function(require) {
    // REQUIRES
    var $ = require('jquery');
    var _ = require('underscore');
    var d3 = require('jira-agile/d3');
    var Events = require('jira-agile/rapid/events');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var ChartView = require('jira-agile/rapid/ui/chart/chart-view');
    var ChartPicker = require('jira-agile/rapid/ui/component/chart-picker');

    // GLOBALS... FIX ME
    var ScopeBurndownBySprintChart;
    var ScopeBurndownBySprintChartLegend;
    var ScopeBurndownBySprintChartHowTo;
    var Layout;
    var Ajax;
    var RapidViewConfig;
    var RapidBoardViewController;

    GlobalEvents.on("pre-initialization", function() {
        ScopeBurndownBySprintChart = GH.Reports.ScopeBurndownBySprintChart;
        ScopeBurndownBySprintChartLegend = GH.Reports.ScopeBurndownBySprintChartLegend;
        ScopeBurndownBySprintChartHowTo = GH.Reports.ScopeBurndownBySprintChartHowTo;
        Layout = GH.Layout;
        Ajax = GH.Ajax;
        RapidViewConfig = GH.RapidViewConfig;
        RapidBoardViewController = GH.RapidBoard.ViewController;
    });

    function ReleaseBurndownReportView() {
        new Events(this);
    }

    ReleaseBurndownReportView.prototype = {
        _chart: null,
        _legend: null,
        _howTo: null,
        _firstRender: true,
        _model: null,

        init: function (model) {
            this._model = model;

            ChartView.getChartContentElem(true).html(GH.tpl.reports.releaseBurndownReport.renderChartBody());
            ChartView.getChartLegendD3(true).html(GH.tpl.reports.releaseBurndownReport.renderLegend()).show();
            this._renderOptions();

            this._versionPicker = new ChartPicker('releaseBurndown.selectedVersionId', AJS.I18n.getText('gh.rapid.charts.version.noversions'));
            $(this._versionPicker).on('selectedItemChanged', _.bind(this._handleVersionSelected, this));

            this._chart = new ScopeBurndownBySprintChart();
            this._legend = new ScopeBurndownBySprintChartLegend();
            this._howTo = new ScopeBurndownBySprintChartHowTo();

            this._renderLegend();
            this._renderHowTo();
        },

        destroy: function () {
            this._chart = null;
            this._legend = null;
            this._howTo = null;
            ChartView.getChartContentElem(true).empty();
            ChartView.getChartLegendD3(true).empty();
            $(GH).unbind(Layout.EVENT_DELAYED_RESIZE + '.releaseBurndownReport');
        },

        setVersions: function (versions) {
            this._versionPicker.setItems(versions);
            this._versionPicker.setDefaultItem(versions[0]);
        },

        update: function (data) {
            this._versionPicker.setSelectedItem(data.version.id, true);
            this._versionPicker.render(ChartView.getChartSelector(true));
            this._chart.data(data);
            this._renderChart();
            this._renderSnapshot(data);
            this._renderVersionMetadata(data);

            if (this._firstRender) {
                // Delay binding to window resize until there is a chart rendered
                Layout.bindDelayedWindowResize();
                $(GH).bind(Layout.EVENT_DELAYED_RESIZE + '.releaseBurndownReport', _.bind(this._handleWindowResized, this));
            } else {
                ChartView.showChartUpdatedToast();
            }
            this._firstRender = false;
        },

        updateTable: function (data) {
            this._renderTable(data);
        },

        showBlankState: function (hasNoVersion, hasNoEstimatedStories) {
            var $message = $(GH.tpl.reports.releaseBurndownReport.renderBlankStateMessage({
                planModeUrl: Ajax.CONTEXT_PATH + '/secure/RapidBoard.jspa?rapidView=' + RapidViewConfig.currentData.data.id + '&view=planning&versions=visible',
                hasNoVersion: hasNoVersion,
                hasNoEstimatedStories: hasNoEstimatedStories
            }));

            $message.find('.js-plan-mode').on('simpleClick', function (e) {
                e.preventDefault();
                RapidBoardViewController.setMode('plan');
            });

            ChartView.showBlankStateMessage($message);
        },

        hideBlankState: function () {
            ChartView.hideBlankStateMessage();
        },

        showSpinner: function () {
            ChartView.showSpinner();
        },

        hideSpinner: function () {
            ChartView.hideSpinner();
        },

        _renderSnapshot: function (data) {
            var UNESTIMATED_ISSUES_WARNING_THRESHOLD = 30;
            var unestimatedIssuesCount = data.snapshot.estimatableIssueCount - data.snapshot.estimatedIssueCount;
            var unestimatedIssuesPercentage = data.snapshot.estimatableIssueCount> 0 ? Math.round((unestimatedIssuesCount / data.snapshot.estimatableIssueCount) * 100) : 0;
            var snapshotHtml = data.snapshot.issueCount === 0 ? '' : GH.tpl.reports.releaseBurndownReport.renderSnapshot({
                estimatableIssuesCount: data.snapshot.estimatableIssueCount,
                unestimatedIssuesCount: unestimatedIssuesCount,
                unestimatedIssuesPercentage: unestimatedIssuesPercentage,
                unestimatedIssuesThresholdExceeded: unestimatedIssuesPercentage > UNESTIMATED_ISSUES_WARNING_THRESHOLD,
                workCompleted: data.estimationStatistic.formatter.formatFull(data.snapshot.workCompleted),
                workTotal: data.estimationStatistic.formatter.formatFull(data.snapshot.workCompleted + data.snapshot.workRemaining),
                estimationStatistic: data.estimationStatistic.name,
                shouldShowUnestimatedIssues : data.estimationStatistic.typeId !== 'issueCount'
            });
            $('#ghx-chart-snapshot').html(snapshotHtml);
            GH.Tooltip.tipsify({
                selector: '#ghx-chart-snapshot .ghx-unestimated-issues',
                html: true,
                className: 'ghx-tipsy-left'
            });
        },

        _renderTable: function (data) {
            ChartView.getChartContentElem(true).find('.ghx-report-issue-list')
                .html(GH.tpl.reports.scopeBurndownBySprint.renderIssueList(_.extend({
                    emptySprintMessage: AJS.I18n.getText('gh.rapid.charts.releaseburndown.no.work.completed')
                }, data)));
        },

        _renderLegend: function () {
            this._legend(d3.select('.ghx-chart-legend'));
        },

        _renderHowTo: function () {
            this._howTo(d3.select('.ghx-group-how'));
        },

        _renderChart: function () {
            this._chart(d3.select('#ghx-release-burndown-report'));
        },

        _renderVersionMetadata: function (data) {
            ChartView.getChartContentElem(true).find('.ghx-version-metadata')
                .html(GH.tpl.reports.releaseBurndownReport.renderVersionMetadata(data.version));
        },

        _renderOptions: function() {
            var view = this;
            $('#ghx-chart-options').html(GH.tpl.reports.releaseBurndownReport.renderChartViewingOptions(this._model));
            $('#js-option-zero-baseline').on('change', function() {
                view.trigger('optionChanged', { zeroBaseline: this.checked });
            });
        },

        _handleVersionSelected: function (event, data) {
            this.trigger('versionSelected', data.item.id);
        },

        _handleWindowResized: function () {
            this._renderChart();
        }
    };

    return ReleaseBurndownReportView;
});
