GH.StatusCategoryPicker = (function () {

    var DEFAULT_STATUS_CATEGORY_ID = 4; // The "In Progress" category

    /**
     * Default options for the StatusCategoryPicker objects.
     * @const
     */
    var PICKER_DEFAULT_OPTIONS = {
        width: 250,
        itemAttrDisplayed: "label",
        errorMessage: "",
        removeOnUnSelect: false,
        statusCategories: [],
        selectedStatusCategoryId: DEFAULT_STATUS_CATEGORY_ID
    };

    /**
     * Creates and returns a new instance of the StatusCategoryPicker
     * @param options
     * @returns {AJS.SingleSelect}
     */
    function buildPicker(options) {
        var picker = {
            statusCategoryPicker: null,
            options: _.defaults({}, options, PICKER_DEFAULT_OPTIONS)
        };

        configure(picker);
        defineComponent(picker);

        return createComponent(picker);
    }

    /**
     * Configures the StatusCategoryPicker and adds the options to the select field so the AJS.SingleSelect
     * can populate
     * @param pickerData
     */
    function configure(pickerData) {

        // update the options with the specified status categories
        pickerData.options.element.html(
            GH.tpl.component.statuscategorypicker.renderStatusCategoryPickerOptions({
                statusCategories: pickerData.options.statusCategories,
                selectedStatusCategoryId: pickerData.options.selectedStatusCategoryId
            })
        );

    }

    /**
     * Extends AJS.SingleSelect to behave like the Status Category Picker
     * @param pickerData
     */
    function defineComponent(pickerData) {

        // Extend SingleSelect to display our status category icons as well.
        pickerData.statusCategoryPicker = AJS.SingleSelect.extend({
            _hasIcon: function () {
                return true;
            },

            setSelection: function (descriptor) {
                AJS.SingleSelect.fn.setSelection.call(this, descriptor);

                this.$container.find(".fake-ss-icon")
                    .remove();

                this.$container.append(getStatusCategoryIcon(descriptor)
                    .css({
                        position: "absolute",
                        left: "6px",
                        top: "50%",
                        marginTop: "-8px",
                        boxSizing: "border-box"
                    })
                    .addClass("fake-ss-icon aui-ss-entity-icon")
                );
            },

            setSelectionByValue: function (value) {
                var matchingDescriptor = _.find(this.suggestionsHandler.model.getAllDescriptors(), function (descriptor) {
                    return descriptor.properties.value == value;
                });
                if (matchingDescriptor) {
                    this.setSelection(matchingDescriptor);
                }
            },

            disableWithValue: function (value) {
                this.setSelectionByValue(value);
                this.disable();
            },

            enableWithValueIfDisabled: function (value) {
                if (this.disabled) {
                    this.enable();
                    this.setSelectionByValue(value);
                }
            }
        });

        // Don't render any other AJS.SingleSelect icons
        pickerData.statusCategoryPicker.fn._renders.entityIcon = function (descriptor) {
            return AJS.$();
        };

    }

    /**
     * Given a AJS.SingleSelect ItemDescriptor representing a status category, return the corresponding status
     * category icon to display.
     * @param descriptor
     * @returns {jQuery}
     */
     function getStatusCategoryIcon(descriptor) {

        var statusCategory = _.omit(
            AJS.$.extend({}, descriptor.model().data()),
            "descriptor"
        );

        var $statusCategory = AJS.$(JIRA.Template.Util.Issue.Status.issueStatus({
            issueStatus: {
                name: descriptor.label(),
                statusCategory: statusCategory
            },
            isCompact: true
        }));

        $statusCategory
            .removeClass("jira-issue-status-lozenge-tooltip")
            .removeAttr("title")
            .removeAttr("data-tooltip");

        return $statusCategory;

    }

    /**
     * Creates a new instance of the extended AJS.SingleSelect
     * @param pickerData
     * @returns {AJS.SingleSelect}
     */
    function createComponent(pickerData) {

        var select = new pickerData.statusCategoryPicker(pickerData.options);
        var oldRenderer = select.listController._renders.suggestion;

        select.listController._renders.suggestion = function (descriptor) {
            var elem = oldRenderer.apply(this, arguments);

            elem.find('a')
                .prepend('&nbsp;')
                .prepend(getStatusCategoryIcon(descriptor));

            return elem;
        };

        return select;

    }

    return {
        /**
         * @lends GH.StatusCategoryPicker
         */
        build: buildPicker,
        defaultStatusCategoryId: DEFAULT_STATUS_CATEGORY_ID
    };

})();

