/**
 * A service for performing search requests for confluence pages for a specific epic.
 * When a search is requested from the service, it doesn't directly fire off an ajax call to perform the search, but waits X amount of time before launching the search.
 * If a new search request is performed before the scheduled search request is executed, the scheduled request will cleared out ie cancelled.
 * The intervening request will then be queued for execution and the delay timer is reset.
 * Once a scheduled request has been launched and another request comes in before the excecuting request has finished, the results of that executing search request will be discarded.
 * If a search request completes it's execution without being interrupted by another request, the promise returned by {@link GH.PageSearchRequestService.search} will be resolved successfully with the search results.
 * Interrupted search request's promise won't be resolved.
 */
GH.PageSearchRequestService = {};

GH.PageSearchRequestService.delay = 400;
GH.PageSearchRequestService.timeoutId = null;

GH.PageSearchRequestService.search = function(epicKey, query){
    GH.PageSearchRequestService.cancel();

    var searchDeferred = AJS.$.Deferred();
    GH.PageSearchRequestService.timeoutId = setTimeout(function() {
        GH.PageSearchRequestService._execute(searchDeferred, epicKey, query);
    }, GH.PageSearchRequestService.delay);

    return searchDeferred.promise();
};

/**
 * Cancels a running search query, if any
 *
 * @returns {*}
 */
GH.PageSearchRequestService.cancel = function(){
    if(GH.PageSearchRequestService.timeoutId != null){
        clearTimeout(GH.PageSearchRequestService.timeoutId);
        GH.PageSearchRequestService.timeoutId = null;
    }
};

GH.PageSearchRequestService._scheduleRequest = function(epicKey, query){
    GH.PageSearchRequestService.timeoutId = setTimeout(function() {
        GH.PageSearchRequestService._execute(epicKey, query);
    }, GH.PageSearchRequestService.delay);
};

GH.PageSearchRequestService._execute = function(searchDeferred, epicKey, query) {
    var timeoutId = GH.PageSearchRequestService.timeoutId;
    var deferred = GH.Ajax.get({
        url : "/integration/confluence/pages/search",
        data: { searchQuery: query },
        deferErrorHandling: true
    }, "searchConfluencePagesForEpic");

    deferred.done(function(result){
        // Check if the search request has been interrupted since we started the ajax call
        if(timeoutId === GH.PageSearchRequestService.timeoutId){
            GH.PageSearchRequestService._replaceHighlightTokensWithHtml(result.pages);
            searchDeferred.resolveWith(searchDeferred, arguments);
        }
    });
    deferred.fail(function(result){
        // Check if the search request has been interrupted since we started the ajax call
        if(timeoutId === GH.PageSearchRequestService.timeoutId){
            searchDeferred.rejectWith(searchDeferred, arguments);
        }
    });
};

GH.PageSearchRequestService.HIGHLIGHT_TOKEN_START = "@@@hl@@@";
GH.PageSearchRequestService.HIGHLIGHT_TOKEN_END = "@@@endhl@@@";

/**
 * Replaces an highlight tokens in a page's attributes with '<strong>' tags
 * @param pages
 */
GH.PageSearchRequestService._replaceHighlightTokensWithHtml = function(pages){
    var replaceTokens = function(string){
        return string.replace(new RegExp(GH.PageSearchRequestService.HIGHLIGHT_TOKEN_START, 'g'), "<strong>")
                     .replace(new RegExp(GH.PageSearchRequestService.HIGHLIGHT_TOKEN_END, 'g'), "</strong>");
    };

    var stripTokens = function(string){
        return string.replace(new RegExp(GH.PageSearchRequestService.HIGHLIGHT_TOKEN_START, 'g'), "")
                     .replace(new RegExp(GH.PageSearchRequestService.HIGHLIGHT_TOKEN_END, 'g'), "");
    };

    _.each(pages, function(page){
        if(page.title){
            page.titleHtml = replaceTokens(page.title);
            page.title = stripTokens(page.title);
        }
        if(page.space){
            page.spaceHtml = replaceTokens(page.space);
            page.space = stripTokens(page.space);
        }
    });
    return pages;
};
