/**
 * Details field edit controls
 */

/**
 * Defining the base class for an Edit Control (TextArea or TextInput)
 */
GH.DetailsFieldEditControl = function(){};

GH.DetailsFieldEditControl.events = new GH.Events();

(function() {
    if (!GH.DetailsObjectFactory.isInlineEditableEnabled()) {
        var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
        /**
         * @type module:jira-agile/rapid/analytics-tracker
         */
        GH.DetailsFieldEditControl.Analytics = new AnalyticsTracker("gh.issuedetail", "inlineedit");
    }
})();

GH.DetailsFieldEditControl.prototype = {

    generateControl : function(editInfo) {
        // implement this
        throw 'Unimplemented method';
    },

    keydownHandler : function(event, editInfo) {
        var self = this;
        if (event.keyCode == 27) { // escape key
            event.stopPropagation();
            self.cancelEdit(editInfo);
        }
        else if (event.keyCode == 13) { // return
            event.preventDefault();
            // Blur the edit control. This will in turn submit the changes.
            // We have this indirection so that pressing enter is consistent
            // with the blur event. See GHS-5856
            editInfo.editControl.blur();
        }
        else if (event.keyCode == 9) { // tab
            // 1: tab, -1: shift-tab
            var delta = event.shiftKey ? -1 : 1;
            var success = GH.DetailsFieldEdit.editRelativeField(editInfo, delta);
            if (success) {
                // prevent default
                //event.preventDefault();
                // trigger submitting the changes - the blur event is not properly fired in IE
                //GH.DetailsFieldTextArea.submitChanges(editInfo);
            }
        }
    },

    /**
     * Called to initiate editing using the provided editInfo
     */
    startEdit : function(editInfo) {
        var self = this;

        var editControl = self.generateControl(editInfo);

        // set the value inside the form element
        // if the entry has a 'value' attribute, we prefer that to 'text'
        var editValue = '';
        if (!_.isUndefined(editInfo.fieldData.value)) {
            editValue = editInfo.fieldData.value;
        } else if (!_.isUndefined(editInfo.fieldData.text)) {
            editValue = editInfo.fieldData.text;
        }
        editControl.val(editValue);

        // On blur return the field with new value
        editControl.bind('blur focusout', function() {
            self.submitChanges(editInfo);
        });

        // handle some keys specially
        editControl.keydown(function(event) {
            self.keydownHandler(event, editInfo);
        });

        // show the edit and focus the area
        GH.DetailsFieldEdit.showEdit(editInfo);

        // focus the text area
        self.focus(editInfo);

        // mark the field as enabled now
        editInfo.enabled = true;
    },

    /**
     * Cancels the edit operation
     */
    cancelEdit : function(editInfo) {
        // disable
        if (!editInfo.enabled) {
            return;
        }
        editInfo.enabled = false;

        // remove blur handler from textarea
        editInfo.editControl.unbind('blur');

        // then switch out the edit area
        GH.DetailsFieldEdit.hideEdit(editInfo);
    },

    /**
     * Submits a new value for the given edit field
     * @fires GH.DetailsFieldEditControl#before:inlineedit as soon as the edit is submitted, unless the edit is cancelled.
     * @fires GH.DetailsFieldEditControl#after:inlineedit after the ajax request returns and the details pane data is updated.
     */
    submitChanges : function(editInfo) {
        var self = this;
        // disable, ensure only one event can trigger the submit
        if (!editInfo.enabled) {
            return;
        }
        editInfo.enabled = false;

        // check whether we actually changed, otherwise just cancel the edit
        var newValue = editInfo.editControl.val();
        if (newValue == editInfo.fieldData.text) {
            editInfo.enabled = true;
            self.cancelEdit(editInfo);
            return;
        }
        GH.DetailsFieldEditControl.events.trigger('before:inlineedit');
        // change the ui into a processing state...
        editInfo.editControl.unbind('blur').attr("disabled", "disabled");
        editInfo.editDd.find('.ghx-spinner').show();

        // trigger the submitting of the value
        var errorCtxMap = {};
        errorCtxMap[editInfo.fieldData.id] = function(ctxError) {
            GH.DetailsFieldEdit.handleFieldUpdateFailure(editInfo, editInfo.fieldData.id, ctxError);
        };

        GH.Ajax.put({
            url: '/xboard/issue/update-field.json',
            data: {
                'issueIdOrKey' : editInfo.model.getIssueId(),
                'fieldId' : editInfo.fieldData.id,
                'newValue' : newValue
            },
            errorContextMap: errorCtxMap
        })
            .done(function(data) {
                GH.DetailsFieldEdit.handleFieldUpdateSuccess(editInfo, data.success);
                // analytics
                GH.DetailsFieldEditControl.events.trigger('after:inlineedit');

                if (GH.DetailsObjectFactory.isInlineEditableEnabled()) {
                    AJS.trigger('analytics', {name: 'gh.issueaction.issuedetail.inlineedit.submit', data: {changedField: editInfo.fieldData.id}});
                } else {
                    GH.DetailsFieldEditControl.Analytics.trigger("fieldId: " + editInfo.fieldData.id);
                }
            })
            .fail(function(result) {
                // on global error, return to edit to remove spinners
                // TODO: this needs to pass through the DetailsFieldEdit to allow it to not return to edit,
                // e.g. when the edited issue changed!
                editInfo.editor.returnToEdit(editInfo);
            });

        if (!GH.DetailsObjectFactory.isInlineEditableEnabled()) {
            // Trigger analytics event [SW-2194]
            AJS.trigger('analytics', {name: 'gh.issueaction.issuedetail.inlineedit.submit', data: {changedField: editInfo.fieldData.id}});
        }
    },

    setErrors : function(editInfo, errors) {
        var html = GH.tpl.detailview.renderFieldErrors({errors: errors});

        if (editInfo.fieldData.errorSelector) {
            // use the specified error selector (could be anywhere in the DOM)
            AJS.$(editInfo.fieldData.errorSelector).html(html);
        } else {
            // use the default error selector (near the input)
            editInfo.editDd.find('.js-fielderrors').html(html);
        }
    },

    /**
     * Called in case of a submit or validation error
     */
    returnToEdit : function(editInfo) {
        var self = this;
        // make sure we add the blur back in
        editInfo.editControl.blur(function(){
            self.submitChanges(editInfo);
        });

        // remove the spinner
        editInfo.editDd.find('.ghx-spinner').hide();

        // reenable the area
        editInfo.editControl.removeAttr('disabled');

        // ask for the focus back, but only if there are no other elements being edited (happens in case of tabbing through the elements)
        if (editInfo.model.getEditingCount() == 1) {
            self.focus(editInfo);
        }

        // re-enable
        editInfo.enabled = true;
    },

    /**
     * Focus the edited field
     */
    focus : function(editInfo) {
        var focus = function() {
            editInfo.editControl.focus();

            // hack! we want the story points field to be selected
            if (editInfo.fieldData.type == 'number') {
                editInfo.editControl.select();
            }
        };
        setTimeout(focus, 0);
    }
};

/**
 * TextArea edit support.
 * @param doesReturnSubmit set to true if typing 'return' should make the control submit; false if multi-line editing is
 * required.
 */
GH.DetailsFieldTextArea = function (doesReturnSubmit) {
    this.doesReturnSubmit = _.isUndefined(doesReturnSubmit) ? true : doesReturnSubmit;
};

GH.DetailsFieldTextArea.prototype = new GH.DetailsFieldEditControl();

GH.DetailsFieldTextArea.prototype.generateControl = function(editInfo) {
    // render the edit html and insert it into the editDd
    // hard-code the number of rows for this control for 'summary' field to 6.
    var numRows = editInfo.fieldData.id == 'summary' ? 6 : 10;
    var fieldContent = GH.tpl.detailview.renderTextareaEdit({numRows: numRows});
    editInfo.editDd.html(fieldContent);

    // append listeners
    var textarea = editInfo.editDd.find('textarea');
    editInfo.editControl = textarea;
    return textarea;
};

GH.DetailsFieldTextArea.prototype.keydownHandler = function(event, editInfo) {
    var self = this;
    if (event.keyCode == 27) { // escape key
        event.stopPropagation();
        self.cancelEdit(editInfo);
    }
    else if (self.doesReturnSubmit && event.keyCode == 13) { // return
        event.preventDefault();
        // Blur the edit control. This will in turn submit the changes.
        // We have this indirection so that pressing enter is consistent
        // with the blur event. See GHS-5856
        editInfo.editControl.blur();
    }
    else if (event.keyCode == 9) { // tab
        // 1: tab, -1: shift-tab
        var delta = event.shiftKey ? -1 : 1;
        var success = GH.DetailsFieldEdit.editRelativeField(editInfo, delta);
        if (success) {
            // prevent default
            //event.preventDefault();
            // trigger submitting the changes - the blur event is not properly fired in IE
            //GH.DetailsFieldTextArea.submitChanges(editInfo);
        }
    }
};

/**
 * Text edit support
 */
GH.DetailsFieldTextInput = function () {};
GH.DetailsFieldTextInput.prototype = new GH.DetailsFieldEditControl();

GH.DetailsFieldTextInput.prototype.generateControl = function(editInfo) {
    // render the edit html and insert it into the editDd
    var fieldContent = GH.tpl.detailview.renderTextEdit({field: editInfo.fieldData});
    editInfo.editDd.html(fieldContent);

    // append listeners
    var textControl = editInfo.editDd.find('input');
    editInfo.editControl = textControl;
    return textControl;
};
