define('jira-agile/rapid/ui/detail/inlineedit/issue-detail-view-inline-editor', ['require'], function (require) {
    'use strict';

    var EVENT_NAMESPACE = "agileDetailViewInlineEditableEventNamespace";

    var _ = require('underscore');
    var jQuery = require('jquery');
    var Events = require('jira/util/events');
    var EventTypes = require("jira/components/issueeditor/eventtypes");
    var DialogsOverrider = require('jira-agile/rapid/ui/detail/inlineedit/dialogsoverrider');
    var InlineEditUtils = require('jira/components/issueeditor/inlineeditutils');
    var IssueFieldUtil = require('jira/components/issueviewer/legacy/issuefieldutil');
    var IssueEditorWrapper = require('jira-agile/rapid/ui/detail/inlineedit/issue-editor-wrapper');
    var EventReasons = require('jira/util/events/reasons');
    var TimeTrackingFieldHelper = require('jira-agile/rapid/ui/detail/inlineedit/timetracking-fields-helper');
    var EditableDetailsViewReloadReason = require('jira-agile/rapid/ui/detail/inlineedit/details-view-reload-reason');
    var ATTACHMENT_ITEM_POSSIBLE_IDS = ['attachment_thumbnails', 'file_attachments']; // there are two display modes (thumbnail, list) for attachments, different mode has different item id.

    var $detailContainer = null;

    // callbacks
    var getRapidViewIdCallback = null;
    var loadIssueDetailCallback = null;
    var getIssueIdCallback = null;
    var getIssueKeyCallback = null;
    var getIssueSprintStatusCallback = null;

    var DetailViewInlineEditor = {

        /**
         * @param {Object} options
         * @param {Object} options.$detailContainer
         *      a jQuery object of the element wrapping the whole detail view.
         * @param {function} options.getRapidViewIdCallback
         *      a function that returns the id of the *current* board.
         * @param {function} options.loadIssueDetailCallback
         *      a function that load data and redraw the detail view.
         * @param {function} options.getInlineEditableFieldsCallback
         *      a function that returns a list of *FieldHtmlBean* which will be used to enable inline-editable mode on the fields.
         * @param {function} options.getIssueIdCallback
         *      a function that returns ID of the issue being display on the detail view.
         * @param {function} options.getIssueKeyCallback
         *      a function that returns key of the issue being display on the detail view.
         * @param {function} options.getIssueSprintStatusCallback
         *      a function that returns the status (string) of the current sprint of the issue that being displayed
         *      in the detail view.
         */
        init: function init(options) {
            IssueEditorWrapper.init(options);

            $detailContainer = options.$detailContainer;
            getRapidViewIdCallback = options.getRapidViewIdCallback;
            loadIssueDetailCallback = options.loadIssueDetailCallback;
            getIssueIdCallback = options.getIssueIdCallback;
            getIssueKeyCallback = options.getIssueKeyCallback;
            getIssueSprintStatusCallback = options.getIssueSprintStatusCallback;

            DialogsOverrider.init(loadIssueDetailCallback);
            registerListeners();
        },

        /**
         * @returns {object} a promise which will call all done callbacks without data
         */
        loadIssueToEditor: IssueEditorWrapper.loadIssueToEditor,

        triggerFieldUpdate: IssueEditorWrapper.triggerFieldUpdate,

        getFieldSelector: IssueFieldUtil.getFieldSelector,

        hasEditsInProgress: IssueEditorWrapper.hasEditsInProgress,

        getEditsInProgress: IssueEditorWrapper.getEditsInProgress,

        getIssueAttachmentQuery: IssueEditorWrapper.getIssueAttachmentQuery,
        /* visible for testing */
        _issueEditorWrapper: IssueEditorWrapper,

        fixUpDetailView: fixUpDetailViewForInlineEditableMode,

        destroy: function destroy() {
            getRapidViewIdCallback = null;
            loadIssueDetailCallback = null;
            getIssueIdCallback = null;
            getIssueKeyCallback = null;
            getIssueSprintStatusCallback = null;

            DialogsOverrider.destroy();
            IssueEditorWrapper.destroy();
        },

        registerListenersForTab: function registerListenersForTab(tabId, $detailsContents) {
            switch (tabId) {
                case 'comment':
                    fixUpDetailViewForInlineEditableMode($detailsContents);
                    registerCommentModuleListener();
                    break;
                case 'header':
                    renderSprintScopeChangeForPrimaryEstimateField();
                    registerBlurEditOnEstimateFields();
                    break;
                case 'attachment':
                    registerAttachmentListener();
                    renderAttachmentDropZone();
                    break;
                case 'details':
                    IssueEditorWrapper.loadIssueToEditor();
                    break;
                default:
                    break;
            }
        }
    };

    function registerListeners() {
        registerCommentModuleListener();
        renderSprintScopeChangeForPrimaryEstimateField();
        registerBlurEditOnEstimateFields();
        registerAttachmentListener();
        Events.unbind(eventName(JIRA.Events.ISSUE_REFRESHED));
        // this event is fired in IssueViewer.js#_onIssueLoaded
        // when reloading (only this case) a selected issue (re-click on the card), after finishing loading the issue data,
        // the function _onIssueLoaded will be called twice: one for "onLoadingDone" and one for "onUpdatingDone"
        // See the implementation of function IssueViewer.js#_onIssueLoaded: in case of reloading an issue, it will call to
        // function refreshIssue. This happens on issue nav as well.
        // we want to bootstrap the editor once. However, don't use _.once here as it will prevent future invocations.
        // It is the case of adding comments/adding subtasks. In such cases, the attachment drop zone will disappear after
        // adding a comment because the _.once prevents the function to be re-invoked
        Events.bind(eventName(JIRA.Events.ISSUE_REFRESHED), _.debounce(onIssueRefreshedHandler, 0));
    }

    function registerCommentModuleListener() {
        // hook the sort comment button
        $detailContainer.off(eventName('click')).on(eventName('click'), '#commentmodule .issue-activity-sort-link', function (e) {
            e.preventDefault();

            var sortLink = jQuery(e.currentTarget).attr('href');
            jQuery.get(sortLink).done(function () {
                loadIssueDetailCallback(null, EditableDetailsViewReloadReason.COMMENTS_CHANGED);
            });
        });

        // reload the ADV when new comment is added
        Events.unbind(eventName(JIRA.Events.UNLOCK_PANEL_REFRESHING));
        Events.bind(eventName(JIRA.Events.UNLOCK_PANEL_REFRESHING), function (event, trigger) {
            if (trigger === 'addcommentmodule') {
                loadIssueDetailCallback(null, EditableDetailsViewReloadReason.COMMENTS_CHANGED);
            }
        });
    }

    function renderSprintScopeChangeForPrimaryEstimateField() {
        // Show warning message when scope changes [SW-2992]
        $detailContainer.off(eventName('focus')).on(eventName('focus'), '.ghx-estimate input', function () {
            var $estimate = $detailContainer.find('.ghx-estimate');
            var $scopeChangeWarning = $estimate.find('.ghx-estimate-scope-warning');

            if (getIssueSprintStatusCallback() === 'ACTIVE' && $scopeChangeWarning.length === 0) {
                $scopeChangeWarning = jQuery('<div class="ghx-estimate-scope-warning"></div>');
                $scopeChangeWarning.text(AJS.I18n.getText('gh.sprint.issue.move.dialog.warning.scopechange'));

                // Get the input element to append the warning message
                var $editForm = $estimate.find('.ghx-editing'); // "Original Time Estimate" type
                if ($editForm.length === 0) {
                    $editForm = $estimate.find('.editable-field.active .field-group'); // number type
                }
                $editForm.append($scopeChangeWarning);
            }
        });
    }

    function registerBlurEditOnEstimateFields() {
        Events.bind(EventTypes.INLINE_EDIT_STARTED, function (e, fieldId, fieldType, $container) {
            if (TimeTrackingFieldHelper.isTimeTrackingField(fieldId)) {
                if (!$container.find('.error').length) {
                    InlineEditUtils.BlurTriggers.Default(fieldId, TimeTrackingFieldHelper.getFieldFromContainer(fieldId, $container));
                }
            }
        });
    }

    function registerAttachmentListener() {
        // reload the ADV when new file attached. both reason and context must be match to catch uploaded file event
        Events.unbind(eventName(JIRA.Events.NEW_CONTENT_ADDED));
        Events.bind(eventName(JIRA.Events.NEW_CONTENT_ADDED), function (e, context, reason) {
            if (reason === JIRA.CONTENT_ADDED_REASON.panelRefreshed && ATTACHMENT_ITEM_POSSIBLE_IDS.indexOf(context.attr('id')) >= 0) {

                var hasAnyRecentUploadedAttachment = _.any(context.find('.attachment-content'), function (attachment) {
                    return _.isNull(jQuery(attachment).data('issue-id'));
                });

                if (hasAnyRecentUploadedAttachment) {
                    /* global GH */
                    if (GH.RapidBoard.State.isPlanMode()) {
                        loadIssueDetailCallback(null, EditableDetailsViewReloadReason.ATTACHMENTS_CHANGED);
                    } else {
                        // If you add an attachment via the browse file menu, when it closes it will trigger a focus on the window which reloads the details view (Thanks to the work mode poller).
                        // However, iff we used loadIssueDetailCallback() here, we would load the details view twice (which is undesirable).
                        // Doing it this way, we're simulating the action which will be done after the browse file menu closes, which will trigger the work mode poller to start.

                        /* global GH */
                        GH.windowActivityNotifier.trigger('blur');
                        loadIssueDetailCallback(null, EditableDetailsViewReloadReason.ATTACHMENTS_CHANGED);
                    }
                }
            }
        });
    }

    function onIssueRefreshedHandler() {
        renderAttachmentDropZone();
    }

    function renderAttachmentDropZone() {
        // reference code: IssuePanelView.js#onRender (issue nav plugin). In the platform, after rendering a panel
        // (the attachment is a panel), the method IssuePanelView.js#onRender will be called. It then will fire a
        // NEW_CONTENT_ADDED event its panel ID as an event context. That event will be used to create the attachment
        // drop zone. See function initIssueDropZone.js#onready
        //
        //            JIRA.bind(JIRA.Events.NEW_CONTENT_ADDED, function (e, context, reason) {
        //                if (context.parents(triggerDialogIDs).size() > 0) {
        //                    createDropZoneInContext(context, 'CreateIssueDropZone');
        //                }
        //
        //                createAttachmentsDropZone(context);
        //            });
        //
        // In Agile detail view, attachment panel is draw by agile code, thus, we need to fire this event
        JIRA.trigger(JIRA.Events.NEW_CONTENT_ADDED, [jQuery('#attachmentmodule'), EventReasons.panelRefreshed]);
    }

    function fixUpDetailViewForInlineEditableMode($detailContents) {
        // hook the static link comment button to open the link in the new tab
        $detailContents.find('#commentmodule .activitymodule-link.issue-comment-action').attr('target', '_blank');
    }

    function eventName(eventType) {
        return eventType + '.' + EVENT_NAMESPACE;
    }

    return DetailViewInlineEditor;
});