/* global atl_token */
define('jira-agile/rapid/ui/detail/inlineedit/issue-editor-wrapper',[
    'underscore',
    'jquery',
    'require'
], function(
    _,
    jQuery,
    require
) {
    'use strict';

    var EVENT_NAMESPACE = "agileIssueEditorEventNamespace";

    var Meta = require('jira/util/data/meta');
    var IssueEditor = require('jira/components/issueeditor');
    var IssueFieldUtil = require('jira/components/issueviewer/legacy/issuefieldutil');
    var LinksCapturer = require('jira/components/issueviewer/linkscapturer');
    var TimeTrackingFieldsHelper = require('jira-agile/rapid/ui/detail/inlineedit/timetracking-fields-helper');
    var EditableDetailsViewReloadReason = require('jira-agile/rapid/ui/detail/inlineedit/details-view-reload-reason');

    var issueEditor = null;
    var linksCapturer = null;

    var $detailContainer = null;

    // callbacks
    var getRapidViewIdCallback = null;
    var loadIssueDetailCallback = null;
    var getInlineEditableFieldsCallback = null;
    var getIssueIdCallback = null;
    var getIssueKeyCallback = null;

    var currentIssueKey = null;
    var currentIssueData = null;
    var currentViewIssueQuery = null;

    var IssueEditorWrapper = {

        /**
         * @param {Object} options
         * @param {Object} options.$detailContainer
         *      a jQuery object of the element wrapping the whole detail view.
         * @param {function} options.getRapidViewIdCallback
         *      a function that returns the id of the *current* board.
         * @param {function} options.loadIssueDetailCallback
         *      a function that load data and redraw the detail view.
         * @param {function} options.getInlineEditableFieldsCallback
         *      a function that returns a list of *FieldHtmlBean* which will be used to enable inline-editable mode on the fields.
         * @param {function} options.getIssueIdCallback
         *      a function that returns ID of the issue being display on the detail view.
         * @param {function} options.getIssueKeyCallback
         *      a function that returns key of the issue being display on the detail view.
         */
        init: function(options) {
            $detailContainer = options.$detailContainer;

            getRapidViewIdCallback = options.getRapidViewIdCallback;
            loadIssueDetailCallback = options.loadIssueDetailCallback;
            getInlineEditableFieldsCallback = options.getInlineEditableFieldsCallback;
            getIssueIdCallback = options.getIssueIdCallback;
            getIssueKeyCallback = options.getIssueKeyCallback;

            initEditor();

            initLinksCapturer();
        },

        loadIssueToEditor: function() {
            // There is a race condition between
            // 1. destroyIssueEditor
            // 2. initEditor
            // 3. loadIssueEditor
            // If we request a page with issue editor before the previous request has completed, we might call (1)
            // after we have called (2). In effect (3) gets a null issueEditor. If this happens, just ignore this (3)
            // and the last request will eventually do (2) and (3) without an interleaving (1).
            if (!issueEditor) {
                return;
            }

            // if this flag is false, the editor will load issue data from DOM, not via AJAX call.
            // Loading issue from DOM is used we access the issue view page directly via url /browse/{issue-key}. In that
            // case, the web page is rendered from server side and returned as a normal HTTP request. The issue data is
            // embedded in the returned DOM. Loading issue from DOM means extracting that data from DOM
            //
            // Our case requires load issue data via an AJAX call.
            Meta.set("server-view-issue-is-editable", true);

            // at the end, the function loadIssueData (below) will be called
            issueEditor.loadIssue({
                id: getIssueIdCallback(),
                key: getIssueKeyCallback()
            });
        },

        triggerFieldUpdate: function(fieldId, $newIssueHtml) {
            var EditIssueController = issueEditor.editIssueController;
            var fieldModelToUpdate = _.find(EditIssueController.getFields().models, function(model) {
                return model.id === fieldId;
            });
            if (fieldModelToUpdate) {
                // Store the new view html into the field view.
                EditIssueController.getIssueEventBus().triggerPanelRendered(fieldModelToUpdate.id, $newIssueHtml);
                // Tell the model to update itself based on the new view.
                fieldModelToUpdate.triggerUpdateRequired();
            }
        },

        hasEditsInProgress: function() {
            return this.getEditsInProgress().length > 0;
        },

        getEditsInProgress: function() {
            var EditIssueController = issueEditor.editIssueController;
            return _.union(EditIssueController.getDirtyEditsInProgress(), EditIssueController.getEditsInProgress());
        },

        destroy: function() {
            currentIssueKey = null;
            currentIssueData = null;
            currentViewIssueQuery = null;
            getRapidViewIdCallback = null;
            loadIssueDetailCallback = null;
            getInlineEditableFieldsCallback = null;
            getIssueIdCallback = null;
            getIssueKeyCallback = null;

            destroyLinkCapture();
            destroyIssueEditor();
        },

        getWrappedIssueEditor : function() {
            return issueEditor;
        },

        /**
         * Visible for testing
         */
        _getFieldsData: getFieldsData,

        getIssueAttachmentQuery: function() {
            return currentViewIssueQuery;
        }
    };

    function initEditor() {
        if (issueEditor) {
            return;
        }
        // Like jira-issue-search-plugin's DetailsLayout.js, issue editor is created once and reused when switching
        // among issues in the search result list. It will only be destroyed when the detail view is closed (choosing
        // the view option "List view"
        issueEditor = new IssueEditor();

        // this function will be called in an event handler of "issueLoaded" event (see IssueEditor.js)
        // to load editable fields. Only fields returned by this call are editable AND the value of the field displayed
        // on the field editor (e.g. text box, dropdown...) is inside each of this returned fields.
        issueEditor.fieldsLoader._getFieldsData = getFieldsData;

        // calls to AjaxIssueAction!default.jspa to load issue data into the editor
        issueEditor.viewIssueData.fetch = loadIssueData;

        // reference code in SaveInProgressManager.js file in jira-issue-navigator-components-plugin
        issueEditor.saveInProgressManager.saveIssue = saveAField;

        // ADV doesn't need the issue editor/viewer to render anything, ADV will do the rendering. Thus,
        // letting the issue editor run rendering code is unnecessary and more importantly too many
        // NEW_CONTENT_ADDED events are fired by the rendering code which cause unexpected behavior
        // in third-party plugins like Tempo.
        // more details are in SW-3212
        issueEditor.issueController.show = function() {};

        // Dirty comment will only be checked if mistakenly cancel the comment, not when loading issue to editor.
        issueEditor.canDismissComment = function() {
            return true;
        };

        var fieldChanges = {};
        jQuery(GH)
            .unbind(eventName("QuickEdit.fieldChange"))
            .bind(eventName("QuickEdit.fieldChange"), function (event, data) {
                fieldChanges[data.fieldId] = data.fieldChangeData;
            });

        issueEditor.on("saveSuccess", function (event) {
            // Reload details view, jira platform inline-edit only returns the field id on success (event.savedFieldIds)
            // we need to reload the issue detail in order to:
            // - Redraw the detail view with the new value. NOTE that, unlike on the view issue page of the platform where
            //   after saving a field, no additional AJAX call made (e.g the one the load issue data). The reason is: the
            //   response of the save request contains all the panels with updated data and the IssueViewer just takes them
            //   to redraw the whole issue page. In agile view, we don't use these panels. Agile builds its own panels and
            //   return in a separate REST call.
            // - have the corresponding card in the board updated (info displayed on the card may be the one that has been updated)

            var savedFieldId = event.savedFieldIds[0];
            loadIssueDetailCallback(savedFieldId, EditableDetailsViewReloadReason.getReloadReasonForChangedField(savedFieldId));
            jQuery(GH).trigger('issueUpdated', {issueId: event.issueId, source: 'detailView', fieldId: savedFieldId, fieldChanges: fieldChanges});

            // [SW-2194] we want to monitor the trend of using inline editor
            AJS.trigger('analytics', {name: 'gh.issueaction.issuedetail.inlineedit.submit', data: {changedField: savedFieldId}});
        });
    }

    // arguments of this function are identical to the ViewIssueData#.fetch function
    function loadIssueData(issueKey, options) {
        var returnedData;

        if (shouldReloadIssueDataFromServer(options)) {
            returnedData = getNewIssueData();
        } else {
            returnedData = currentIssueData;
        }

        var deferred = new jQuery.Deferred();
        deferred.resolve(returnedData);
        return deferred.promise();
    }

    /*
     * This function is responsible for creating a new data
     * with a new/changed issue, then bootstrap inline-edibility for fields as a result.
     */
    function getNewIssueData() {
        var data = {};
        data.fields = [];
        data.issue = {
            id: getIssueIdCallback(),
            key: getIssueKeyCallback(),
            isEditable: true,
            operations: {
                linkGroups: []
            }
        };
        data.panels = {
            leftPanels: [],
            rightPanels: [],
            infoPanels: []
        };
        moderateAjaxIssueActionResponse(data);

        return data;
    }

    function shouldReloadIssueDataFromServer(loadOptions) {
        if (!loadOptions.issueEntity || !currentIssueData) {
            return true;
        }
        var issueEntity = loadOptions.issueEntity;

        // when options.issueEntity.viewIssueQuery is defined, we must always reload data from the server
        // to have load options (ex attachmentOrder : 'DESC') take effect
        return currentIssueKey !== issueEntity.key ||
            !_.isEqual(currentViewIssueQuery ? currentViewIssueQuery : null,
                        issueEntity.viewIssueQuery ? issueEntity.viewIssueQuery : null);
    }

    // arguments of this function are identical to the SaveInProgressManager.saveIssue function
    function saveAField(issueId, issueKey, fieldsToSave, data, ajaxProperties) {

        var fieldIdToSave = fieldsToSave[0];
        // The server has a different id to represent time tracking fields,
        // so we need to differentiate between them
        var serverIdToSave = fieldIdToSave;

        /* Time tracking fields have odd behaviour. The form (edit mode) has both original and remaining fields,
            and we hide the one we are not interested in. Therefore, when we save the form, we need
            to tell the server we only want to save the value for the field we have shown the user.
         */
        if (TimeTrackingFieldsHelper.isTimeTrackingField(fieldIdToSave)) {
            TimeTrackingFieldsHelper.specifyWhichFieldToSave(fieldIdToSave, data, IssueEditorWrapper.getEditsInProgress());
            serverIdToSave = TimeTrackingFieldsHelper.getServerFieldId();
        }

        var instance = issueEditor.saveInProgressManager;
        instance.triggerBeforeSaving();

        var saveInProgress;
        var responseData;

        var allParams = _.extend(data, {
            issueId: issueId,
            atl_token: atl_token(),
            singleFieldEdit: true,
            fieldsToForcePresent: [serverIdToSave],
            skipScreenCheck: true,
            rapidViewId: GH.DetailsView.rapidViewId
        });

        var ajaxOpts = _.extend({
            type: "POST",
            url: AJS.contextPath() + "/secure/DetailsViewAjaxIssueAction.jspa?decorator=none",
            headers: {'X-SITEMESH-OFF': true},
            error: function(xhr) {
                instance._handleSaveError(issueId, issueKey, fieldsToSave, xhr);
            },
            success: function(resp, statusText, xhr, smartAjaxResult) {
                responseData = smartAjaxResult.data;
                // Was the response HTML?
                if (typeof responseData === "string") {
                    instance._handleHtmlResponse(issueId, issueKey, fieldsToSave, responseData);
                } else {
                    responseData = moderateAjaxIssueActionResponse(responseData);
                    IssueFieldUtil.transformFieldHtml(responseData);

                    // When we tell the server to save only one tracking field,
                    // it copies the value to the other field in the response.
                    // We don't want this duplicated value, so remove it.
                    if (TimeTrackingFieldsHelper.isTimeTrackingField(fieldsToSave[0])) {
                        responseData.fields = TimeTrackingFieldsHelper.removeOtherTrackingField(fieldsToSave[0], responseData.fields);
                    }
                    instance.triggerSaveSuccess(issueId, issueKey, fieldsToSave, responseData);
                }
            },
            complete: function() {
                currentViewIssueQuery = null;
                instance.removeSaveInProgress(saveInProgress);
                JIRA.trigger(JIRA.Events.INLINE_EDIT_SAVE_COMPLETE);
            },
            data: allParams
        }, ajaxProperties);

        saveInProgress = JIRA.SmartAjax.makeRequest(ajaxOpts);
        instance.addSaveInProgress(saveInProgress);

        instance.triggerSavingStarted(issueId, fieldsToSave, allParams);
    }

    /**
     * Handles click actions on the option menu of the attachment block e.g Sort by Name, Sort by Date...
     */
    function initLinksCapturer() {
        if (linksCapturer) {
            return;
        }
        linksCapturer = new LinksCapturer();

        linksCapturer.on("refineViewer", function(event) {
            event.preventDefault();

            // when clicking on the [Options] menu of the attachment block (e: Sort Ascending by name)
            // after the sort is done in the server, we redraw the detail view.
            issueEditor.once("loadComplete", function (e, data) {
                if (data.issueId === getIssueIdCallback()) {
                    loadIssueDetailCallback(null, EditableDetailsViewReloadReason.ATTACHMENTS_CHANGED);
                }
            });

            // see IssueNavCreator.js (jira-issue-search-plugin)
            //          JIRA.Issues.Application.on("issueEditor:refineViewer", function(event) {
            //              ...
            //              JIRA.Issues.Application.execute("issueEditor:updateIssueWithQuery", event.query);
            //          });
            // this will reload issue with additional options in the request (ex: attachmentOrder = asc)
            currentViewIssueQuery = event.query;
            issueEditor.updateIssueWithQuery(event.query);
        });

        // the IssueViewer (the parent of IssueEditor) also initializes this LinksCapturer which call to the capture function
        // in IssueViewer#setContainer function. However, we don't call to IssueViewer#setContainer because we don't let the
        // IssueViewer draw issue content. Thus we have to initialize our own LinksCapturer

        linksCapturer.capture($detailContainer);
    }

    // Reference:  FieldsLoaderService.js file in jira-issue-navigator-components-plugin
    function getFieldsData(issueId, issueKey) {
        // The result of this call will be used to determine what fields are editable. The platform's endpoint
        // (AjaxIssueEditAction!default.jspa) returns fields defined in "field tabs". But in Agile detail view,
        // editable fields are defined in Board configuration > Issue Detail View.

        var deferred = new jQuery.Deferred();

        // this setTimeout looks weird (zero timeout) but needed to correctly initialize the inline editor.
        // without this, it's impossible to save an inline editing field: can enter the edit mode but can't save, clicking
        // the check button gives no effect.
        //
        // - The cause (of non-savable, why doesn't work): I'll try to explain it here, but it's best to place a breakpoint
        //      and find it yourself. Below is more or less a hint for debugging:
        //   1) When entering edit mode of a field: function FieldModel.js#edit is called, it set the flag "editing" to true.
        //          This is IMPORTANT, take note it.
        //   2) When clicking (✓) button to save and editing field, the following flow will occur
        //      2.1) Function FieldView.js#onSubmit gets called
        //      2.2) Function FieldModel.js#save gets called. It triggers the *save* event and passes itself as event's data
        //                  this.triggerSave(this);
        //      2.3) Function EditIssueController.js#save is the handler of the *onSave* event, now gets called
        //              This function has the following check:
        //                  if (!model.getEditing() || model.getSaving()) {
        //                      return;
        //                  }
        //              *model* is an instance of the FieldModel.js
        //              *editing* is the flag set in step 1)
        //      In our case (without setTimeout), the *editing* flas is undefined, thus never saved
        //
        //   Observation: when NOT using setTimeout, every field model (each field has one model) will be initialize TWICE
        //   e.g function FieldModel.js#initialize will be called twice. The consequence is: when clicking (✓) button to save an editing field:
        //      + FieldView.js#onSubmit (2.1) was called twice
        //      + FieldModel.js#save (2.2) was called twice on 2 different instances
        //          ==> the "save" event is trigger twice
        //      + But EditIssueController.js#save only called once (guessing backbone trip out one duplicated event)
        //         And the passed to this function is the one with "editing" flag undefined ==> not saved
        //
        // - Guess (hence, comes up with this setTimeout):
        //   The platform's code relies on the fact that getting field data is an asynchronous operation. Thus, after firing
        //   the request, it continues its after-issue-loaded tasks and the code handling field data will always be executed later on.
        //   Now, we since "resolve" the "promise" synchronously, it breaks the assumption of platform code.
        //
        //   Setting this timeout (with zero delay) is just to move the promise resolution out of the current browser event loop,
        //   makes it async
        //
        setTimeout(function() {
            // This might be executed after the IssueEditor has been destroyed.
            if (issueEditor) {
                // Will be undefined if a new issue has started loading before this callback was invoked.
                var fields = getInlineEditableFieldsCallback();
                if (fields) {
                    deferred.resolve(buildFieldDataResponse(fields, issueId, issueKey));
                }
            }
        }, 0);

        return deferred.promise();
    }

    function buildFieldDataResponse(fields, issueId, issueKey) {
        var resp = IssueFieldUtil.transformFieldHtml({ fields : fields });
        return {
            issueId: issueId,
            issueKey: issueKey,
            fields: resp.fields
        };
    }

    // By default, whenever a field is saved from a view issue page, on the server side, it will look in the screen scheme
    // to see if field is in there. If it's not, the field will not be saved.
    //
    // In Agile detail view, fields to be displayed (and editable) are not coming from any screen scheme. They are configured
    // in the board configuration.
    function moderateAjaxIssueActionResponse(result) {
        if (result && result.panels) {
            // the *result* is returned by platform's AjaxIssueAction!default.jspa. The following *panels* are plugable
            // panels. That means 3rd-party plugins can introduce new panels in any predefined sections (below).
            // when the platform's JS code receives this result, it will evaluate all the panels. If there is any embedded
            // JS code, the code will be execuated. However, since we don't request the whole context of view-issue page,
            // 3rd-party JS will not availble ==> JS errors (SW-2937).
            // Agile don't use these panel, thus, it's safe to clear everything before passing this result object to platform.
            result.panels.infoPanels = [];
            result.panels.leftPanels = [];
            result.panels.rightPanels = [];
        }

        // don't clear the result.fields as in case of validation error, the field (html) with error markup will be return

        currentIssueKey = result.issue.key;
        currentIssueData = result;
        return result;
    }

    function destroyLinkCapture() {
        if (linksCapturer) {
            linksCapturer.destroy();
        }
        linksCapturer = null;
    }

    function destroyIssueEditor() {
        if (issueEditor) {
            issueEditor.close();
        }
        jQuery(GH).unbind(eventName("QuickEdit.fieldChange"));
        issueEditor = null;
    }

    function eventName(eventType) {
        return eventType + '.' + EVENT_NAMESPACE;
    }

    return IssueEditorWrapper;
});
