GH.Dialogs.ConvertWorkflow = {};

GH.Dialogs.ConvertWorkflow.EVENT_WORKFLOW_CONVERTED = "workflowConverted";
GH.Dialogs.ConvertWorkflow.statusSelect = undefined;

(function() {
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    /**
     * @type module:jira-agile/rapid/analytics-tracker
     */
    GH.Dialogs.ConvertWorkflow.analytics = new AnalyticsTracker('gh.config.simplify.workflow.dialog.change');
})();

GH.Dialogs.ConvertWorkflow.showDialog = function(rapidViewId) {
    GH.Dialog.dim();

    GH.Ajax.get({
        url: '/workflow/convert.json',
        data : {
            rapidViewId : rapidViewId
        }
    }, 'convertWorkflow')
        .done(GH.Dialogs.ConvertWorkflow.renderDialog)
        .fail(GH.Dialog.undim);
    GH.Dialogs.ConvertWorkflow.analytics.trigger("start"); // SAFE
};

/**
 * Renders the "convert workflow" dialog
 */
GH.Dialogs.ConvertWorkflow.renderDialog = function(model) {
    // cancel callback
    var cancelFn = function() {
        GH.Dialogs.ConvertWorkflow.analytics.trigger("cancel"); // SAFE
    };

    // create the dialog
    var dialog = GH.Dialog.create({
        width : 575,
        height : 300,
        id : 'ghx-dialog-convert-workflow',
        onCancelFn: cancelFn
    });

    // initial page
    var needsIssueUpdatePage = (model.numIssuesToMigrate && model.numIssuesToMigrate > 0);
    GH.Dialogs.ConvertWorkflow.renderDialogIntroPage(dialog, model);
    GH.Dialogs.ConvertWorkflow.renderStatusResolutionDonePage(dialog, model, needsIssueUpdatePage);
    if (needsIssueUpdatePage) {
        GH.Dialogs.ConvertWorkflow.renderDialogIssuesUpdatePage(dialog, model);
    }
    GH.Dialogs.ConvertWorkflow.renderDialogProgressPage(dialog, model);

    AJS.$('.aui-dialog').addClass('ghx-dialog');

    dialog.gotoPage(0);
    dialog.gotoPanel(0);
    dialog.show();
    dialog.updateHeight();

    AJS.$('.aui-blanket').empty();
};

GH.Dialogs.ConvertWorkflow.renderDialogIntroPage = function(dialog, model) {
    dialog.addHeader(AJS.I18n.getText('gh.rapid.config.workflow.convert.workflow.dialog.title'));
    dialog.addPanel('', GH.tpl.dialogs.convertworkflow.renderConvertWorkflowDialogIntro({
        projectName: AJS.escapeHTML(String(model.projectName)),
        workflowSchemeName: AJS.escapeHTML(String(model.workflowSchemeName)),
        adminWorkflowUrl: GH.Ajax.CONTEXT_PATH + '/plugins/servlet/project-config/' + model.projectKey + '/workflows',
        simplifyWorkflowLearnMoreUrl: GH.HelpPaths.getHelpPath('simplified.workflow').url
    }));

    dialog.addButton(AJS.I18n.getText('gh.rapid.common.next'), function() {
        dialog.nextPage();
        _.each(model.statuses, function(status) {
            if(status.resolutionDoneCandidate) {
                GH.Dialogs.ConvertWorkflow.statusSelect.addItem(new AJS.ItemDescriptor({
                    value: status.id,
                    label: status.name
                }));
            }
        });

        dialog.updateHeight();
    }, 'ghx-dialog-convert-next aui-button');
    GH.Dialog.addCancelButton(dialog);
};

GH.Dialogs.ConvertWorkflow.renderStatusResolutionDonePage = function(dialog, model, needsIssueUpdatePage) {
    dialog.addPage();
    dialog.addHeader(AJS.I18n.getText('gh.rapid.config.workflow.convert.workflow.dialog.title'));
    dialog.addPanel('', GH.tpl.dialogs.convertworkflow.renderStatusResolution({
        urls: {
            'simplifiedWorkflow': GH.HelpPaths.getHelpPath("simplified.workflow").url
        }
    }));

    var $select = AJS.$("#ghx-resolution-done-statuses-select");
    GH.Dialogs.ConvertWorkflow.statusSelect = new AJS.MultiSelect({
        element: $select,
        width:350,
        itemAttrDisplayed: "label",
        suggestions: function() {
            return _.map(model.statuses, function(status) {
                return new AJS.ItemDescriptor({
                    value: status.id,
                    label: status.name
                });
            });
        }
    });

    AJS.$("#ghx-resolution-done-statuses-select-multi-select").keydown(function(event) {
        if (event.keyCode == 13 && !GH.Dialogs.ConvertWorkflow.statusSelect.suggestionsVisible) {
            event.preventDefault();
            return false;
        }
    });

    if(needsIssueUpdatePage) {
        dialog.addButton(AJS.I18n.getText('gh.rapid.common.next'), function() {
            GH.Dialogs.ConvertWorkflow.analytics.trigger("showWarning"); // SAFE

            dialog.nextPage();
            dialog.updateHeight();
        }, 'ghx-dialog-convert-next aui-button');
    } else {
        GH.Dialogs.ConvertWorkflow.addSimplifyButton(dialog, model);
    }
    GH.Dialog.addCancelButton(dialog);
};

GH.Dialogs.ConvertWorkflow.renderDialogIssuesUpdatePage = function(dialog, model) {
    dialog.addPage();
    dialog.addHeader(AJS.I18n.getText('gh.rapid.config.workflow.convert.workflow.dialog.title'));
    var page2 = GH.tpl.dialogs.convertworkflow.renderConvertWorkflowDialogIssues({
        projectName: AJS.escapeHTML(String(model.projectName)),
        workflowSchemeName: AJS.escapeHTML(String(model.workflowSchemeName)),
        numIssuesToMigrate: GH.NumberFormat.format(model.numIssuesToMigrate),
        isSysAdmin : model.isSysAdmin,
        backupUrl: GH.Ajax.CONTEXT_PATH + '/secure/admin/XmlBackup!default.jspa'
    });
    dialog.addPanel('', page2);

    GH.Dialogs.ConvertWorkflow.addSimplifyButton(dialog, model);
    GH.Dialog.addCancelButton(dialog);
};

GH.Dialogs.ConvertWorkflow.renderDialogProgressPage = function(dialog, model) {
    dialog.addPage();
    dialog.addHeader(AJS.I18n.getText('gh.rapid.config.workflow.convert.workflow.dialog.title'));
    var page3 = GH.tpl.dialogs.convertworkflow.renderConvertWorkflowDialogProgress({
    });
    dialog.addPanel('', page3);

    GH.Dialogs.ConvertWorkflow.addAcknowledgeButton(dialog, model);
};

GH.Dialogs.ConvertWorkflow.addSimplifyButton = function(dialog, model) {
    dialog.addButton(AJS.I18n.getText('gh.rapid.config.workflow.convert.workflow.dialog.submit.label'), function() {
        GH.Dialogs.ConvertWorkflow.analytics.trigger("simplify"); // SAFE

        dialog.nextPage();
        dialog.updateHeight();
        dialog.disableControls();
        GH.Dialogs.ConvertWorkflow.submit(dialog, model);
    },'aui-button');
};

GH.Dialogs.ConvertWorkflow.addAcknowledgeButton = function(dialog, model) {
    dialog.addButton(AJS.I18n.getText('common.words.acknowledge'), function() {
        dialog.disableControls();

        // close our dialog
        dialog.dispose();

        // tell parties about the success
        AJS.$(GH).trigger(GH.Dialogs.ConvertWorkflow.EVENT_WORKFLOW_CONVERTED);
    },'aui-button');
};

/**
 * Submit the dialog
 */
GH.Dialogs.ConvertWorkflow.submit = function(dialog, model) {
    var resolutionDoneStatusElement = AJS.$("#ghx-resolution-done-statuses-select");
    var resolutionDoneStatusIds = resolutionDoneStatusElement.val();

    // fire off two requests - firstly we need to initiate the workflow migration on our side by copying the workflow and scheme
    GH.Ajax.post({
        url: '/workflow/convert.json',
        data: {
            projectKey: model.projectKey,
            resolutionDoneStatusIds: resolutionDoneStatusIds
        }
    })
        .done(function(result) {
            // then we need to get JIRA to associate our project to this new scheme
            GH.Dialogs.ConvertWorkflow.handleConversionResult(dialog, result);
        })
        .fail(dialog.enableControls);
};

GH.Dialogs.ConvertWorkflow.handleConversionResult = function(dialog, result) {
    var numIssuesToMigrate = result.success.numIssuesToMigrate,
        taskId = result.success.taskId;

    var migrationComplete = function(migrationModel) {
        // analytics for state change and also meta data
        GH.Dialogs.ConvertWorkflow.analytics.trigger("complete"); // SAFE

        // push progress bar to 100%
        var progress = AJS.$('#ghx-convert-workflow-progress');
        progress.find('.ghx-current-progress').css({"width" : "100%"});

        // reenable controls
        dialog.enableControls();

        // add message near the progress bar
        if (migrationModel.successful) {
            progress.append(GH.tpl.rapid.notification.renderAuiMessage({
                title: AJS.I18n.getText('admin.selectworkflowscheme.progress.complete'),
                type: 'success',
                icon: true,
                messageHtml: false,
            }));
        } else {
            // convert JIRA error collection to one that we understand
            var errors = GH.Ajax._convertJiraErrors(migrationModel.errorCollection).getGlobalErrors();

            progress.append(GH.tpl.rapid.notification.renderAuiMessage({
                title: AJS.I18n.getText('gh.rapid.config.workflow.convert.workflow.error.migration.unsuccessful'),
                type: 'error',
                icon: true,
                messageHtml: true,
                message: GH.tpl.rapid.notification.renderErrorMessages({
                    errors: errors
                })
            }));
        }

        // update height of dialog to include new message
        dialog.updateHeight();
    };

    if (!numIssuesToMigrate || numIssuesToMigrate == 0) {
        // if there are no issues to migrate, switch success handler
        migrationComplete({
            finished: true,
            successful: true
        });
    } else {
        // default success handler if there are issues
        if (taskId) {
            GH.Dialogs.ConvertWorkflow.getTaskStatus(taskId, migrationComplete);
        }
    }
};

GH.Dialogs.ConvertWorkflow.getTaskStatus = function(taskId, migrationComplete) {
    GH.Ajax.get({
        url: '/configuration/jiraTask/' + taskId + '.json'
    }).done(function(model) {
        if (!model.finished) {
            // log progress
            console.log('Workflow migration progress: ' + model.progress);

            var $progress = AJS.$('#ghx-convert-workflow-progress');
            $progress.find('.ghx-current-progress').css({"width" : "" + model.progress + "%"});

            // run again
            setTimeout(function() {
                GH.Dialogs.ConvertWorkflow.getTaskStatus(taskId, migrationComplete);
            }, 2000);
        } else {
            console.log('Workflow migration finished. Successful? ' + model.successful);

            migrationComplete(model);
        }
    });
};
