(function () {
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    var FeatureFlagManager = require('jira/featureflags/feature-manager');
    var DARK_FEATURE_SPRINT_GOAL_KEY = 'com.atlassian.jira.agile.darkfeature.sprint.goal';

    GH.Dialogs.StartSprintDialog = {};

    GH.Dialogs.StartSprintDialog.EVENT_SPRINT_STARTED = "sprintStarted";
    GH.Dialogs.StartSprintDialog.dateTimePickerFormat = undefined;

    /**
     * @type module:jira-agile/rapid/analytics-tracker
     */
    GH.Dialogs.StartSprintDialog.analytics = new AnalyticsTracker('gh.start.sprint.dialog.change');

    /**
     * Analytics for start sprint meta data
     */
    GH.Dialogs.StartSprintDialog.analyticsMetaData = function (metaData) {
        AJS.trigger('analyticsEvent', { name: 'gh.sprint.start', data: metaData });
    };

    // TODO thanks to server persisted markers, we no longer need to pass the lastIssueInSprint along; we just need to pass
    // along the marker ID. when we make this change server side, change this to reflect that
    GH.Dialogs.StartSprintDialog.showDialog = function (rapidViewId, sprintId, numVisibleIssuesInSprint) {
        AJS.dim();
        AJS.$('.aui-blanket').html('<span class="ghx-wait"></span>');

        GH.Ajax.get({
            url: '/sprint/' + sprintId + '/start',
            data: {
                rapidViewId: rapidViewId,
                sprintId: sprintId
            }
        }, "showStartSprintDialog").done(function (model) {
            GH.Dialogs.StartSprintDialog.renderDialog(model, numVisibleIssuesInSprint < model.numIssues);
        }).fail(GH.Dialog.undim);

        GH.Dialogs.StartSprintDialog.analytics.trigger("start"); // SAFE
    };

    /**
     * Renders the "start sprint" dialog where the details of the new version can be specified
     *
     * @param model : see above for model
     */
    GH.Dialogs.StartSprintDialog.renderDialog = function (model, showFilteredIssuesWarning) {
        // cancel callback
        var cancelFn = function cancelFn() {
            GH.Dialogs.StartSprintDialog.analytics.trigger("cancel"); // SAFE
        };

        var isSprintGoalEnabled = FeatureFlagManager.isFeatureEnabled(DARK_FEATURE_SPRINT_GOAL_KEY);

        // create the dialog
        var dialog = GH.Dialog.create({
            width: 575,
            height: 390 + (showFilteredIssuesWarning ? 70 : 0) + (model.countNotEstimatedIssues > 0 ? 90 : 0) + (model.noTrackingEstimateIssueCount > 0 ? 90 : 0) + (isSprintGoalEnabled ? 70 : 0),
            id: 'ghx-dialog-start-sprint',
            onCancelFn: cancelFn
        });
        dialog.addHeader(AJS.I18n.getText('gh.sprint.start.dialog.title'));
        dialog.addPanel('', 'panel');

        var sprintDurations = [{ value: 7, text: AJS.I18n.getText('gh.time.weeks', 1) }, { value: 14, text: AJS.I18n.getText('gh.time.weeks', 2) }, { value: 21, text: AJS.I18n.getText('gh.time.weeks', 3) }, { value: 28, text: AJS.I18n.getText('gh.time.weeks', 4) }, { value: "custom", text: AJS.I18n.getText('gh.rapid.timeframes.custom') }];

        var linkIssue = function linkIssue(issue) {
            return issue ? GH.tpl.detailview.renderIssueLink({ issueKey: issue }) : issue;
        };

        dialog.getCurrentPanel().html(GH.tpl.dialogs.startsprint.renderStartSprintDialog({
            numIssues: model.numIssues,
            defaultName: model.defaultName,
            goal: model.goal || '',
            sprintGoalEnabled: isSprintGoalEnabled,
            defaultStartDate: model.defaultStartDate,
            defaultEndDate: model.defaultEndDate,
            showFilteredIssuesWarning: showFilteredIssuesWarning,
            numNotEstimatedIssues: model.countNotEstimatedIssues,
            numExtraNotEstimatedIssues: model.countNotEstimatedIssues - model.issuesNotEstimated.length,
            snapshotIssuesNotEstimated: _.flatten([_.first(model.issuesNotEstimated, model.issuesNotEstimated.length - 1)]).map(linkIssue).join(", "),
            lastIssueNotEstimated: linkIssue(_.last(model.issuesNotEstimated)),
            numNoTrackingEstimateIssues: model.noTrackingEstimateIssueCount,
            numExtraNoTrackingEstimateIssues: model.noTrackingEstimateIssueCount - model.noTrackingEstimateIssues.length,
            snapshotIssuesNoTrackingEstimate: _.flatten([_.first(model.noTrackingEstimateIssues, model.noTrackingEstimateIssues.length - 1)]).map(linkIssue).join(", "),
            lastIssueNoTrackingEstimate: linkIssue(_.last(model.noTrackingEstimateIssues)),
            listOfSprintDurations: sprintDurations,
            hasNoSubtaskDoneStatus: model.hasNoSubtaskDoneStatus,
            endSprintHelpUrl: GH.HelpPaths.getHelpPath("sprint.end").url
        }));

        dialog.addButton(AJS.I18n.getText('gh.sprint.start.dialog.ok.button.label'), function () {
            dialog.disableControls();
            GH.Dialogs.StartSprintDialog.submit(dialog, model);
        }, 'aui-button');
        GH.Dialog.addCancelButton(dialog);

        GH.Dialogs.StartSprintDialog.registerCalendarControls(model.dateTimePickerFormat);

        GH.Dialogs.StartSprintDialog.registerDurationDropdown(AJS.$(document.body), sprintDurations);

        AJS.$('.aui-dialog').addClass('ghx-dialog');

        dialog.show();
        AJS.$('.aui-blanket').empty();

        // give focus to first input element
        AJS.$('#ghx-sprint-name').focus();

        GH.Dialogs.StartSprintDialog.registerWorkdayCounter(model.rapidViewId, dialog);
    };

    /**
     * Submit the dialog
     */
    GH.Dialogs.StartSprintDialog.submit = function (dialog, model) {
        // fetch the dialog, we got some id clashes atm
        var dialogDiv = AJS.$('#ghx-dialog-start-sprint');

        // create the data we send to the server
        var data = {
            rapidViewId: model.rapidViewId,
            sprintId: model.sprintId,
            name: dialogDiv.find('#ghx-sprint-name').val(),
            goal: dialogDiv.find('#sprint-goal').val(),
            startDate: dialogDiv.find('#ghx-sprint-start-date').val(),
            endDate: dialogDiv.find('#ghx-sprint-end-date').val()
        };

        // clear previous errors
        GH.Validation.clearContextualErrors(dialogDiv);

        GH.Ajax.put({
            url: '/sprint/' + model.sprintId + '/start',
            data: data,
            errorContextMap: {
                'name': '#ghx-sprint-name',
                'startDate': '#ghx-sprint-start-date-error',
                'endDate': '#ghx-sprint-end-date-error',
                'dateFormatMismatch': GH.Notification.handleDateFormatMismatchError
            }
        }).done(function (result) {
            var numIssues = model.numIssues;
            var sprintDuration = GH.Dialogs.StartSprintDialog.calculateSprintDuration(data.startDate, data.endDate);
            var totalStatistic = model.totalStatistic;

            // analytics for state change and also meta data
            GH.Dialogs.StartSprintDialog.analytics.trigger("complete"); // SAFE
            GH.Dialogs.StartSprintDialog.analyticsMetaData({
                sprintId: model.sprintId,
                numIssues: numIssues,
                sprintDuration: sprintDuration,
                totalStatistic: model.totalStatistic
            }); // SAFE

            // close our dialog
            dialog.dispose();

            // show a success message
            GH.Notification.showSuccess(AJS.I18n.getText("gh.api.sprint.started.success", AJS.escapeHTML(String(result.success.name))));

            // tell parties about the success
            AJS.$(GH).trigger(GH.Dialogs.StartSprintDialog.EVENT_SPRINT_STARTED);
        }).fail(dialog.enableControls);
    };

    GH.Dialogs.StartSprintDialog.registerCalendarControls = function (dateTimePickerFormat) {
        GH.Dialogs.StartSprintDialog.dateTimePickerFormat = dateTimePickerFormat;
        var cal = new Calendar(); // GHS-5829 Need to call new calendar to set up short month names and day names when they are not defined

        Calendar.setup({
            firstDay: 0,
            inputField: 'ghx-sprint-start-date',
            button: 'ghx-sprint-start-date_trigger_c',
            align: 'Tl',
            singleClick: true,
            showsTime: true,
            timeFormat: 12,
            positionStyle: 'fixed',
            ifFormat: GH.Dialogs.StartSprintDialog.dateTimePickerFormat
        });

        Calendar.setup({
            firstDay: 0,
            inputField: 'ghx-sprint-end-date',
            button: 'ghx-sprint-end-date_trigger_c',
            align: 'Tl',
            singleClick: true,
            showsTime: true,
            timeFormat: 12,
            positionStyle: 'fixed',
            ifFormat: GH.Dialogs.StartSprintDialog.dateTimePickerFormat
        });
    };

    GH.Dialogs.StartSprintDialog.calculateWorkDaysRemaining = function (startDate, endDate, rapidViewId, dialog) {
        var startMillis = Date.parseDate(startDate, GH.Dialogs.StartSprintDialog.dateTimePickerFormat).getTime();
        var endMillis = Date.parseDate(endDate, GH.Dialogs.StartSprintDialog.dateTimePickerFormat).getTime();

        GH.Ajax.get({
            url: '/rapidviewconfig/workingdays/period',
            data: {
                rapidViewId: rapidViewId,
                startDate: startMillis,
                endDate: endMillis
            }
        }, "workingdaysperiod").done(function (workDaysRemaining) {
            AJS.$('#ghx-sprint-work-days-remaining').remove();
            dialog.getCurrentPanel().body.append(GH.tpl.dialogs.startsprint.renderWorkDaysInSprint({
                numWorkDaysInSprint: workDaysRemaining.days,
                helpLinkUrl: GH.HelpPaths.getHelpPath("configuring.working.days").url
            }));
            AJS.$('#ghx-sprint-work-days-loading').hide();
        }).fail(function () {
            AJS.$('#ghx-sprint-work-days-remaining').remove();
        });
    };

    GH.Dialogs.StartSprintDialog.calculateSprintDuration = function (startDate, endDate) {
        return Math.ceil(GH.Dialogs.StartSprintDialog.calculateSprintDurationExact(startDate, endDate)).toString();
    };

    GH.Dialogs.StartSprintDialog.calculateSprintDurationExact = function (startDate, endDate) {
        var start = moment(Date.parseDate(startDate, GH.Dialogs.StartSprintDialog.dateTimePickerFormat));
        var end = moment(Date.parseDate(endDate, GH.Dialogs.StartSprintDialog.dateTimePickerFormat));
        var ret = end.diff(start, 'days', true);
        return ret;
    };

    GH.Dialogs.StartSprintDialog.registerDurationDropdown = function (document, sprintDurations) {

        var singleSelect = document.find('#ghx-sprint-duration');
        var startDateField = document.find('#ghx-sprint-start-date');
        var endDateField = document.find('#ghx-sprint-end-date');
        var endDateButton = document.find('#ghx-sprint-end-date_trigger_c');

        var setEndDateField = function setEndDateField() {
            if (singleSelect.val() !== "custom") {
                var d = moment(Date.parseDate(startDateField.val(), GH.Dialogs.StartSprintDialog.dateTimePickerFormat));
                d.add(parseInt(singleSelect.val()), 'days');
                endDateField.val(d.toDate().print(GH.Dialogs.StartSprintDialog.dateTimePickerFormat));
            }
        };

        // Duration dropdown event handler
        singleSelect.change(function () {
            var isFixedDateInterval = singleSelect.val() !== "custom";
            endDateField.prop('disabled', isFixedDateInterval);
            setEndDateField();
        });

        startDateField.change(setEndDateField);

        // Set the selected value for the duration dropdown
        var finder = function finder(choices) {
            var prevSprintDays = GH.Dialogs.StartSprintDialog.calculateSprintDurationExact(startDateField.val(), endDateField.val());

            var result = _.find(choices, function (opt) {
                return prevSprintDays === opt.value;
            });
            return result ? result.value : "custom";
        };

        // if the start/end dates match a interval which exists in the dropdown, the dropdown will select it.
        var setSelectionOfDurationDropdown = function setSelectionOfDurationDropdown() {
            singleSelect.val(finder(sprintDurations)).trigger('change');
        };

        setSelectionOfDurationDropdown();

        // Switch to custom when forced to
        endDateButton.click(function () {
            if (endDateField.prop('disabled')) {
                singleSelect.val('custom').trigger('change');
                endDateField.prop('disabled', false).focus();
                endDateButton.trigger('click');
            }
        });

        startDateField.focusout(setSelectionOfDurationDropdown);
        endDateField.focusout(setSelectionOfDurationDropdown);
    };

    GH.Dialogs.StartSprintDialog.registerWorkdayCounter = function (rapidViewId, dialog) {
        var startDateField = AJS.$('#ghx-sprint-start-date');
        var endDateField = AJS.$('#ghx-sprint-end-date');
        var singleSelect = AJS.$('#ghx-sprint-duration');

        var updateWorkdays = function updateWorkdays() {
            GH.Dialogs.StartSprintDialog.calculateWorkDaysRemaining(startDateField.val(), endDateField.val(), rapidViewId, dialog);
        };

        updateWorkdays();

        startDateField.change(updateWorkdays);
        endDateField.change(updateWorkdays);
        singleSelect.change(function () {
            if (singleSelect.val() !== "custom") {
                updateWorkdays();
            }
        });
    };
})();