AJS.test.require(["com.pyxis.greenhopper.jira:gh-test-common-base", "com.pyxis.greenhopper.jira:gh-test-dates", "com.pyxis.greenhopper.jira:gh-rapid"], function () {
    var BacklogController = require('jira-agile/rapid/ui/plan/backlog-controller');
    var BacklogSelectionController = require('jira-agile/rapid/ui/plan/backlog-selection-controller');
    var PlanControls = require('jira-agile/rapid/ui/plan/plan-controls');
    var PlanController = require('jira-agile/rapid/ui/plan/plan-controller');
    var BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');
    var BacklogView = require('jira-agile/rapid/ui/plan/BacklogView');
    var KanPlanFeatureService = require('jira-agile/rapid/ui/kanplan/kan-plan-feature-service');
    var _ = require('underscore');
    var $ = require('jquery');

    var GlobalEvents = require('jira-agile/rapid/global-events');
    GlobalEvents.trigger('pre-initialization');

    module("initialization", {
        setup: function setup() {
            // data for ScrumBase and KanPlanBase decide if it can handle the data or not
            GH.RapidViewConfig.currentData.data = {
                sprintSupportEnabled: true
            };
        },
        teardown: function teardown() {
            GH.Notification.clear();
            GH.Test.clearEvents();
        }
    });

    test("module 'jira-agile/rapid/ui/plan/backlog-controller' exists", function () {
        ok(BacklogController);
        ok(GH.BacklogController);
    });

    test("initializes the SprintBacklogController and BacklogSelectionController", sinon.test(function () //noinspection JSValidateTypes
    {

        this.stub(BacklogSelectionController, "init");
        this.stub(GH.SprintBacklogController, "init");
        this.stub(GH.RapidBoard.Util.InlineEditable, "register");

        BacklogController.init();

        ok(BacklogSelectionController.init.called);
        ok(GH.SprintBacklogController.init.called);
    }));

    module("filters", {
        setup: function setup() {
            // data for ScrumBase and KanPlanBase decide if it can handle the data or not
            GH.RapidViewConfig.currentData.data = {
                sprintSupportEnabled: true
            };
            sinon.stub(GH.EpicController, "setFilteredEpicKey");
        },
        teardown: function teardown() {
            GH.Notification.clear();
            GH.Test.clearEvents();
            GH.EpicController.setFilteredEpicKey.restore();
        }
    });

    test("clearing the filters does not reload the backlog if there are no filters set", sinon.test(function () {

        PlanControls.quickfilters = PlanControls.quickfilters || {};
        PlanControls.quickfilters.clearFilters = this.stub().returns(false);
        PlanControls.searchFilter = PlanControls.searchFilter || {};
        PlanControls.searchFilter.clearSearchBox = sinon.stub();

        this.spy(BacklogController, "loadData");

        BacklogController.clearFilters();

        ok(!BacklogController.loadData.called);
    }));

    test("epic filters are cleared and URL state is updated if epics panel is disabled", sinon.test(function () {
        sinon.stub(PlanController, "isEpicsPanelEnabled").returns(false);
        this.spy(PlanController, "setEpicsColumnVisible");
        this.spy(GH.RapidBoard.State, "replaceState");

        BacklogController.handleEpicFiltering();

        ok(GH.EpicController.setFilteredEpicKey.calledWith(null));
        ok(PlanController.setEpicsColumnVisible.calledWith(false));
        ok(GH.RapidBoard.State.replaceState.called);
        PlanController.isEpicsPanelEnabled.restore();
    }));

    test("versions filters are cleared and URL state is updated if epics and versions feature flag is disabled", sinon.test(function () {
        sinon.stub(KanPlanFeatureService, "isEpicsAndVersionsEnabled").returns(false);
        this.stub(GH.VersionController, "setFilteredVersionId");
        this.stub(PlanController, "setVersionsColumnVisible");
        this.spy(GH.RapidBoard.State, "replaceState");

        BacklogController.handleVersionFiltering();

        ok(GH.VersionController.setFilteredVersionId.calledWith(null));
        ok(PlanController.setVersionsColumnVisible.calledWith(false));
        ok(GH.RapidBoard.State.replaceState.called);
        KanPlanFeatureService.isEpicsAndVersionsEnabled.restore();
    }));

    module("Ajax tests", {
        setup: function setup() {
            // data for ScrumBase and KanPlanBase decide if it can handle the data or not
            GH.RapidViewConfig.currentData.data = {
                sprintSupportEnabled: true
            };

            GH.Test.setUpFakeServer(this);
            BacklogController.rapidViewData = {
                id: 1
            };
            PlanControls.quickfilters = {
                getActiveQuickFilters: function getActiveQuickFilters() {}
            };
            BacklogController.visible = true;
        },
        teardown: function teardown() {
            GH.Notification.clear();
            GH.Test.restoreServer(this);
        }
    });

    test("Successful Ajax request renders the dialog", sinon.test(function () {
        this.stub(BacklogController, "processData");
        this.stub(GH.PlanView, "hideLoadingBacklog");

        GH.Test.respondToGetWith200(this, '/xboard/plan/backlog/data.json', {});

        BacklogController.loadData();

        this.server.respond();

        ok(BacklogController.processData.called, "Called success method");
        ok(!GH.PlanView.hideLoadingBacklog.called);
    }));

    test("Failed request hides the loading backlog", sinon.test(function () {
        this.stub(BacklogController, "processData");
        this.stub(GH.PlanView, "hideLoadingBacklog");

        GH.Test.respondToGetWith500(this, '/xboard/plan/backlog/data.json', {});

        BacklogController.loadData();

        this.server.respond();

        ok(!BacklogController.processData.called, "Does not call process function on failure");
        ok(GH.PlanView.hideLoadingBacklog.called);
    }));

    test("reloadSingleIssue: Successful Ajax request updates the issue", sinon.test(function () {
        this.stub(BacklogController, "updateIssues");
        this.stub(BacklogModel, 'getIssueDataForId');

        GH.Test.respondToGetWith200(this, '/xboard/plan/backlog/issue.json', {});

        BacklogController.reloadSingleIssue();

        this.server.respond();

        ok(BacklogController.updateIssues.called, "Called success method");
    }));

    test("reloadSingleIssue: Failed request does nothing (no specific error handling)", sinon.test(function () {
        this.stub(BacklogController, "updateIssues");

        GH.Test.respondToGetWith500(this, '/xboard/plan/backlog/issue.json', {});

        BacklogController.reloadSingleIssue();

        this.server.respond();

        ok(!BacklogController.updateIssues.called, "Does not call process function on failure");
    }));

    test("updateEpicData: Successful Ajax request sets the data for the epic", sinon.test(function () {
        this.stub(BacklogController, "setEpicData");

        GH.Test.respondToGetWith200(this, '/xboard/plan/backlog/epics.json', {});

        BacklogController.updateEpicData();

        this.server.respond();

        ok(BacklogController.setEpicData.called, "Called success method");
    }));

    test("updateEpicData: Failed request does nothing (no specific error handling)", sinon.test(function () {
        this.stub(BacklogController, "setEpicData");

        GH.Test.respondToGetWith500(this, '/xboard/plan/backlog/epics.json', {});

        BacklogController.updateEpicData();

        this.server.respond();

        ok(!BacklogController.setEpicData.called, "Does not call function on failure");
    }));

    test("handleIssueCreatedImpl: Successful Ajax request sets the data for the epic", sinon.test(function () {
        this.stub(BacklogController, "issuesAndEpicsCreated");
        this.stub(GH.EpicConfig, "getEpicIssueTypeId").returns(1);

        GH.Test.respondToBareUrlGetWith200(this, '/rest/api/2/search', {});

        BacklogController.handleIssueCreatedImpl({});

        this.server.respond();

        ok(BacklogController.issuesAndEpicsCreated.called, "Called success method");
    }));

    test("handleIssueCreatedImpl: Failed request does nothing (no specific error handling)", sinon.test(function () {
        this.stub(BacklogController, "issuesAndEpicsCreated");
        this.stub(GH.EpicConfig, "getEpicIssueTypeId").returns(1);

        GH.Test.respondToBareUrlGetWith500(this, '/rest/api/2/search', {});

        BacklogController.handleIssueCreatedImpl({});

        this.server.respond();

        ok(!BacklogController.issuesAndEpicsCreated.called, "Does not call function on failure");
    }));

    module('updateIssues', {
        setup: function setup() {
            this.sandbox = sinon.sandbox.create();

            this.EPIC_ISSUE_TYPE_ID = 5;
            this.STORY_ISSUE_TYPE_ID = 6;

            this.mockBacklogModel();
            this.mockBacklogController();
            this.mockBacklogSelectionController();
            this.mockGH();
            this.mockPlanController();

            this.sandbox.stub(BacklogView, 'redrawChangedModels');
        },

        teardown: function teardown() {
            GH.Notification.clear();
            GH.EpicConfig.setEpicConfig(); // clear EpicConfig.epicConfig
            this.sandbox.restore();
        },

        /**
         * Mocking
         */

        mockBacklogModel: function mockBacklogModel() {
            this.BacklogModel = {
                model: {
                    issueList: {
                        removeIssues: sinon.stub(),
                        updateIssue: sinon.stub()
                    },

                    getIssueList: function () {
                        return this.BacklogModel.model.issueList;
                    }.bind(this),

                    getType: function getType() {
                        return 'sprint';
                    },

                    getSprintId: function getSprintId() {
                        return 1;
                    }
                }
            };

            this.BacklogModel.findModelWithIssue = this.sandbox.stub(BacklogModel, 'findModelWithIssue');
            this.BacklogModel.findModelWithIssue.withArgs('KEY-1').returns(this.BacklogModel.model).withArgs('KEY-2').returns(this.BacklogModel.model).withArgs('KEY-3').returns(this.BacklogModel.model);

            this.BacklogModel.findModelsWithFakeParent = this.sandbox.stub(BacklogModel, 'findModelsWithFakeParent');
            this.BacklogModel.updateFiltering = this.sandbox.stub(BacklogModel, 'updateFiltering');
            this.BacklogModel.afterIssueUpdate = this.sandbox.stub(BacklogModel, 'afterIssueUpdate');
        },

        mockBacklogController: function mockBacklogController() {
            this.BacklogController = {};
            this.BacklogController.updateEpicData = this.sandbox.stub(BacklogController, 'updateEpicData');
        },

        mockBacklogSelectionController: function mockBacklogSelectionController() {
            this.BacklogSelectionController = {};
            this.sandbox.stub(BacklogSelectionController, 'validateCurrentSelection');
            this.sandbox.stub(BacklogSelectionController, 'getSelectedIssueKey');
        },

        mockGH: function mockGH() {
            this.isScrumBoard = this.sandbox.stub(GH.RapidBoard.State, 'isScrumBoard');
            this.isKanbanBoard = this.sandbox.stub(GH.RapidBoard.State, 'isKanbanBoard');
            GH.EpicConfig.setEpicConfig({
                epicIssueTypeId: this.EPIC_ISSUE_TYPE_ID
            });
        },

        mockPlanController: function mockPlanController() {
            // makes tests resilient to details view opened in a different tab
            this.sandbox.stub(PlanController, 'isDetailsViewOpened');
            this.isEpicsPanelEnabled = this.sandbox.stub(PlanController, 'isEpicsPanelEnabled');
        },

        assertEpicsAsCards: function assertEpicsAsCards() {
            this.assertEpicsOnBacklogVisibility(true);
        },

        assertEpicsPanel: function assertEpicsPanel() {
            this.assertEpicsOnBacklogVisibility(false);
        },

        assertEpicsOnBacklogVisibility: function assertEpicsOnBacklogVisibility(shouldBeVisible) {
            var issue1 = { 'key': 'KEY-1', 'typeId': this.EPIC_ISSUE_TYPE_ID };
            var issue2 = { 'key': 'KEY-2', 'typeId': this.STORY_ISSUE_TYPE_ID };
            var shouldBeVisibleFn = shouldBeVisible ? ok : notOk;

            BacklogController.updateIssues([issue1, issue2]);

            shouldBeVisibleFn(!this.BacklogModel.model.issueList.removeIssues.calledWith(issue1.key), "Should call removeIssues for epic issue " + issue1.key);
            ok(!this.BacklogModel.model.issueList.removeIssues.calledWith(issue2.key), "Should not call removeIssues for non-epic issue " + issue2.key);

            shouldBeVisibleFn(this.BacklogModel.model.issueList.updateIssue.calledWith(issue1), "Should not call updateIssue for epic issue " + issue1.key);
            ok(this.BacklogModel.model.issueList.updateIssue.calledWith(issue2), "Should call updateIssue for non-epic issue " + issue2.key);

            ok(this.BacklogModel.updateFiltering.called, "Should call updateFiltering");
            shouldBeVisibleFn(!this.BacklogController.updateEpicData.calledOnce, "Should call updateEpicData");
        }
    });

    test("updateIssues: epics should be removed from the model for scrum board", sinon.test(function () {
        this.isScrumBoard.returns(true);
        this.isKanbanBoard.returns(false);
        this.isEpicsPanelEnabled.returns(true);

        this.assertEpicsPanel();
    }));

    test("updateIssues: epics should be removed from the model for kanplan with epics panel enabled", sinon.test(function () {
        this.isScrumBoard.returns(false);
        this.isKanbanBoard.returns(true);
        this.isEpicsPanelEnabled.returns(true);

        this.assertEpicsPanel();
    }));

    test("updateIssues: epics should not be removed from the model for kanban board", sinon.test(function () {
        this.isScrumBoard.returns(false);
        this.isKanbanBoard.returns(true);
        this.isEpicsPanelEnabled.returns(false);

        this.assertEpicsAsCards();
    }));

    test("updateIssues: should call redraw changed models for unique models", sinon.test(function () {
        var issue1 = { 'key': 'KEY-1', 'typeId': this.STORY_ISSUE_TYPE_ID };
        var issue2 = { 'key': 'KEY-2', 'typeId': this.STORY_ISSUE_TYPE_ID };

        BacklogController.updateIssues([issue1, issue2]);

        ok(BacklogView.redrawChangedModels.called, "Should call redraw changed models");
        deepEqual(BacklogView.redrawChangedModels.getCall(0).args[0], [this.BacklogModel.model], "Should call redrawChangedModels only with unique models");
    }));

    test('updateIssues: should notify BacklogModel', sinon.test(function () {
        var expectedIssue = { key: 'KEY-1' };
        BacklogController.updateIssues([expectedIssue]);
        ok(BacklogModel.afterIssueUpdate.calledWith(expectedIssue));
    }));

    module('handleIssueCreatedImpl', {
        setup: function setup() {
            this.sandbox = sinon.sandbox.create();

            this.EPIC_ISSUE_TYPE_ID = 5;
            this.STORY_ISSUE_TYPE_ID = 6;

            this.issue1 = { 'key': 'KEY-1', 'fields': { 'issuetype': { 'id': this.EPIC_ISSUE_TYPE_ID } } };
            this.issue2 = { 'key': 'KEY-2', 'fields': { 'issuetype': { 'id': this.STORY_ISSUE_TYPE_ID } } };

            this.issueDetailsResult = $.Deferred();

            this.mockBacklogController();
            this.mockGH();
        },

        teardown: function teardown() {
            GH.Notification.clear();
            GH.EpicConfig.setEpicConfig(); // clear EpicConfig.epicConfig
            this.sandbox.restore();
        },

        /**
         * Mocking
         */

        mockBacklogController: function mockBacklogController() {
            this.BacklogController = {};
            this.BacklogController.updateEpicData = this.sandbox.stub(BacklogController, 'updateEpicData');
            this.BacklogController.subtasksCreated = this.sandbox.stub(BacklogController, 'subtasksCreated');
            this.BacklogController.getIssueDetails = this.sandbox.stub(BacklogController, 'getIssueDetails');
            this.BacklogController.getIssueDetails.withArgs([this.issue1, this.issue2], 'issuetype').returns(this.issueDetailsResult);
            this.BacklogController.issuesAndEpicsCreated = this.sandbox.stub(BacklogController, 'issuesAndEpicsCreated');
        },

        mockGH: function mockGH() {
            this.isKanbanBoard = this.sandbox.stub(GH.RapidBoard.State, 'isKanbanBoard');
            GH.EpicConfig.setEpicConfig({
                epicIssueTypeId: this.EPIC_ISSUE_TYPE_ID
            });
        }
    });

    test("handleIssueCreatedImpl: subtasks should go to subtasksCreated", sinon.test(function () {
        var data = { isSubtask: true };

        BacklogController.handleIssueCreatedImpl(data);

        ok(this.BacklogController.subtasksCreated.calledWith(data), "Should call subtasksCreated for subtask creation");

        ok(!this.BacklogController.getIssueDetails.called, "Should not call getIssueDetails for subtask creation");
    }));

    test("handleIssueCreatedImpl: issues should go to issuesAndEpicsCreated", sinon.test(function () {
        function flattenIssue(issue) {
            return {
                'issueKey': issue.key,
                'issueId': issue.id,
                'issueType': issue.fields.issuetype.id
            };
        }

        var callbacks = [];

        BacklogController.handleIssueCreatedImpl({ 'issues': [this.issue1, this.issue2] }, callbacks);
        this.issueDetailsResult.resolve({ 'issues': [this.issue1, this.issue2] });

        ok(!this.BacklogController.subtasksCreated.called, "Should not call subtasksCreated for non-subtask creation");

        ok(this.BacklogController.issuesAndEpicsCreated.calledWith([flattenIssue(this.issue2)], [flattenIssue(this.issue1)], callbacks), "Should call issuesAndEpicsCreated with issues and epics split");
    }));

    module('issuesAndEpicsCreated', {
        setup: function setup() {
            this.sandbox = sinon.sandbox.create();

            this.EPIC_ISSUE_TYPE_ID = 5;
            this.issue1 = { 'key': 'KEY-1', 'fields': { 'issuetype': { 'id': this.EPIC_ISSUE_TYPE_ID } } };

            this.mockBacklogController();
            this.mockPlanController();
            this.mockGH();
        },
        teardown: function teardown() {
            GH.Notification.clear();
            GH.EpicConfig.setEpicConfig(); // clear EpicConfig.epicConfig
            this.sandbox.restore();
        },

        mockBacklogController: function mockBacklogController() {
            this.BacklogController = {};
            this.BacklogController.updateEpicData = this.sandbox.stub(BacklogController, 'updateEpicData').returns(AJS.$.Deferred().resolve().promise());
        },
        mockPlanController: function mockPlanController() {
            this.isEpicsPanelEnabled = this.sandbox.stub(PlanController, 'isEpicsPanelEnabled');
        },
        mockGH: function mockGH() {
            this.isCreatedEpicVisible = sinon.stub();
            this.isKanbanBoard = this.sandbox.stub(GH.RapidBoard.State, 'isKanbanBoard');

            this.sandbox.stub(GH.EpicView, 'renderNewEpics');
            this.sandbox.stub(GH.EpicView, 'scrollToBottom');
            this.sandbox.stub(GH.EpicController, 'getEpicModel').returns({
                getEpicList: sinon.stub().returns({
                    isIssueVisible: this.isCreatedEpicVisible
                })
            });

            GH.EpicConfig.setEpicConfig({
                epicIssueTypeId: this.EPIC_ISSUE_TYPE_ID
            });

            this.showCreatedEpicsMessageSpy = this.sandbox.spy(GH.RapidBoard.QuickCreate, 'showCreatedEpicsMessage');
        }
    });

    test("issuesAndEpicsCreated: epic specific message should be shown when creating epic and epics panel enabled", sinon.test(function () {
        var ISSUES = [];
        var CALLBACKS = [];
        var EPICS = [this.issue1];
        var IS_EPIC_VISIBLE = true;
        var IS_EPIC_NOT_VISIBLE = false;

        this.isKanbanBoard.returns(true);
        this.isEpicsPanelEnabled.returns(true);

        this.isCreatedEpicVisible.returns(IS_EPIC_VISIBLE);
        BacklogController.issuesAndEpicsCreated(ISSUES, EPICS, CALLBACKS);
        sinon.assert.calledOnce(this.showCreatedEpicsMessageSpy);
        sinon.assert.calledWith(this.showCreatedEpicsMessageSpy, { issues: EPICS }, IS_EPIC_VISIBLE);

        this.isCreatedEpicVisible.returns(IS_EPIC_NOT_VISIBLE);
        BacklogController.issuesAndEpicsCreated(ISSUES, EPICS, CALLBACKS);
        sinon.assert.calledTwice(this.showCreatedEpicsMessageSpy);
        sinon.assert.calledWith(this.showCreatedEpicsMessageSpy, { issues: EPICS }, IS_EPIC_NOT_VISIBLE);
    }));

    module('Search', {
        setup: function setup() {
            // data for ScrumBase and KanPlanBase decide if it can handle the data or not
            GH.RapidViewConfig.currentData.data = {
                sprintSupportEnabled: true
            };
            sinon.stub(BacklogSelectionController, "init");
            sinon.stub(GH.SprintBacklogController, "init");
            sinon.stub(GH.RapidBoard.Util.InlineEditable, "register");
            sinon.stub(BacklogModel, "updateFiltering");

            BacklogController.init();
        },
        teardown: function teardown() {
            GH.Notification.clear();
            GH.Test.clearEvents();
            BacklogSelectionController.init.restore();
            GH.SprintBacklogController.init.restore();
            GH.RapidBoard.Util.InlineEditable.register.restore();
            BacklogModel.updateFiltering.restore();
        }
    });

    test('executeSearch: if pattern is null it should clear the search filter', sinon.test(function () {
        // arrange
        BacklogController.lastExecutedSearchFilter = true;

        // act
        BacklogController.executeSearch(null);

        // assert
        ok(!BacklogController.lastExecutedSearchFilter, 'filter should be null');
    }));

    test('executeSearch: search for assigneeName should return true', sinon.test(function () {
        //arrange
        var issue = { key: 'MEM-1', summary: 'Create a meme', typeName: 'Story', assignee: 'MemeLord', assigneeName: 'Meme Lord' };
        var rawQuery = 'Meme Lord';
        var pattern = GH.Components.SearchFilter.prototype._queryToRegex(rawQuery);
        // act
        BacklogController.executeSearch(pattern);
        // assert
        ok(BacklogController.lastExecutedSearchFilter(issue), 'executeSearch should return true');
    }));

    test('executeSearch: search for assignee should return true', sinon.test(function () {
        //arrange
        var issue = { key: 'MEM-1', summary: 'Create a meme', typeName: 'Story', assignee: 'MemeLord', assigneeName: 'Meme Lord' };
        var rawQuery = 'MemeLord';
        var pattern = GH.Components.SearchFilter.prototype._queryToRegex(rawQuery);
        // act
        BacklogController.executeSearch(pattern);
        // assert
        ok(BacklogController.lastExecutedSearchFilter(issue), 'executeSearch should return true');
    }));

    test('executeSearch: search for typeName should return true', sinon.test(function () {
        //arrange
        var issue = { key: 'MEM-1', summary: 'Create a meme', typeName: 'Story', assignee: 'MemeLord', assigneeName: 'Meme Lord' };
        var rawQuery = "Story";
        var pattern = GH.Components.SearchFilter.prototype._queryToRegex(rawQuery);
        // act
        BacklogController.executeSearch(pattern);
        // assert
        ok(BacklogController.lastExecutedSearchFilter(issue), 'executeSearch should return true');
    }));

    test('executeSearch: search for summary should return true', sinon.test(function () {
        //arrange
        var issue = { key: 'MEM-1', summary: 'Create a meme', typeName: 'Story', assignee: 'MemeLord', assigneeName: 'Meme Lord' };
        var rawQuery = 'Create a meme';
        var pattern = GH.Components.SearchFilter.prototype._queryToRegex(rawQuery);
        // act
        BacklogController.executeSearch(pattern);
        // assert
        ok(BacklogController.lastExecutedSearchFilter(issue), 'executeSearch should return true');
    }));

    test('executeSearch: search for key should return true', sinon.test(function () {
        //arrange
        var issue = { key: 'MEM-1', summary: 'Create a meme', typeName: 'Story', assignee: 'MemeLord', assigneeName: 'Meme Lord' };
        var rawQuery = "MEM-1";
        var pattern = GH.Components.SearchFilter.prototype._queryToRegex(rawQuery);
        // act
        BacklogController.executeSearch(pattern);
        // assert
        ok(BacklogController.lastExecutedSearchFilter(issue), 'executeSearch should return true');
    }));

    test('executeSearch: search for part of a word should return true', sinon.test(function () {
        //arrange
        var issue = { key: 'MEM-1', summary: 'Create a meme', typeName: 'Story', assignee: 'MemeLord', assigneeName: 'Meme Lord' };
        var rawQuery = "sto";
        var pattern = GH.Components.SearchFilter.prototype._queryToRegex(rawQuery);
        // act
        BacklogController.executeSearch(pattern);
        // assert
        ok(BacklogController.lastExecutedSearchFilter(issue), 'executeSearch should return true');
    }));

    test('executeSearch: should return false for no match', sinon.test(function () {
        //arrange
        var issue = { key: 'MEM-1', summary: 'Create a meme', typeName: 'Story', assignee: 'MemeLord', assigneeName: 'Meme Lord' };
        var rawQuery = 'dank';
        var pattern = GH.Components.SearchFilter.prototype._queryToRegex(rawQuery);
        // act
        BacklogController.executeSearch(pattern);
        // assert
        ok(!BacklogController.lastExecutedSearchFilter(issue), 'executeSearch should return false');
    }));

    test('executeSearch: ignore case positive. Should find the issue (return true) with uppercase raw query', sinon.test(function () {
        //arrange
        var issue = { key: 'MEM-1', summary: 'Create a meme', typeName: 'Story', assignee: 'MemeLord', assigneeName: 'Meme Lord' };
        var rawQuery = "LORD";
        var pattern = GH.Components.SearchFilter.prototype._queryToRegex(rawQuery);
        // act
        BacklogController.executeSearch(pattern);
        // assert
        ok(BacklogController.lastExecutedSearchFilter(issue), 'executeSearch should return true');
    }));

    test('executeSearch: ignore case negative. Should not find the issue (return false) with uppercase raw query', sinon.test(function () {
        //arrange
        var issue = { key: 'MEM-1', summary: 'Create a meme', typeName: 'Story', assignee: 'MemeLord', assigneeName: 'Meme Lord' };
        var rawQuery = "RaREST";
        var pattern = GH.Components.SearchFilter.prototype._queryToRegex(rawQuery);
        // act
        BacklogController.executeSearch(pattern);
        // assert
        ok(!BacklogController.lastExecutedSearchFilter(issue), 'executeSearch should return false');
    }));

    test('executeSearch: null assignee positive. Search should still work with null values for some issue values', sinon.test(function () {
        //arrange
        var issue = { key: 'MEM-1', summary: 'Create a meme', typeName: 'Story', assignee: null, assigneeName: null };
        var rawQuery = "create";
        var pattern = GH.Components.SearchFilter.prototype._queryToRegex(rawQuery);
        // act
        BacklogController.executeSearch(pattern);
        // assert
        ok(BacklogController.lastExecutedSearchFilter(issue), 'executeSearch should return true');
    }));

    test('executeSearch: null assignee negative. Search should still work with null values for some issue values', sinon.test(function () {
        //arrange
        var issue = { key: 'MEM-1', summary: 'Create a meme', typeName: 'Story', assignee: null, assigneeName: null };
        var rawQuery = "rare";
        var pattern = GH.Components.SearchFilter.prototype._queryToRegex(rawQuery);
        // act
        BacklogController.executeSearch(pattern);
        // assert
        ok(!BacklogController.lastExecutedSearchFilter(issue), 'executeSearch should return false');
    }));

    test('executeSearch: null summary positive. Search should still work with null values for some issue values', sinon.test(function () {
        //arrange
        var issue = { key: 'MEM-1', summary: null, typeName: 'Story', assignee: null, assigneeName: null };
        var rawQuery = "MEM";
        var pattern = GH.Components.SearchFilter.prototype._queryToRegex(rawQuery);
        // act
        BacklogController.executeSearch(pattern);
        // assert
        ok(BacklogController.lastExecutedSearchFilter(issue), 'executeSearch should return true');
    }));

    test('executeSearch: null summary negative. Search should still work with null values for some issue values', sinon.test(function () {
        //arrange
        var issue = { key: 'MEM-1', summary: null, typeName: 'Story', assignee: null, assigneeName: null };
        var rawQuery = "pepe";
        var pattern = GH.Components.SearchFilter.prototype._queryToRegex(rawQuery);
        // act
        BacklogController.executeSearch(pattern);
        // assert
        ok(!BacklogController.lastExecutedSearchFilter(issue), 'executeSearch should return false');
    }));

    test('executeSearch: null typeName negative. Search should still work with null values for some issue values', sinon.test(function () {
        //arrange
        var issue = { key: 'MEM-1', summary: 'Create a meme', typeName: null, assignee: null, assigneeName: null };
        var rawQuery = "dank";
        var pattern = GH.Components.SearchFilter.prototype._queryToRegex(rawQuery);
        // act
        BacklogController.executeSearch(pattern);
        // assert
        ok(!BacklogController.lastExecutedSearchFilter(issue), 'executeSearch should return false');
    }));

    test('executeSearch: null typeName positive. Search should still work with null values for some issue values', sinon.test(function () {
        //arrange
        var issue = { key: 'MEM-1', summary: 'Create a meme', typeName: null, assignee: null, assigneeName: null };
        var rawQuery = "create";
        var pattern = GH.Components.SearchFilter.prototype._queryToRegex(rawQuery);
        // act
        BacklogController.executeSearch(pattern);
        // assert
        ok(BacklogController.lastExecutedSearchFilter(issue), 'executeSearch should return true');
    }));

    module('subtasksCreated', {
        setup: function setup() {
            var deferred = new AJS.$.Deferred();
            this.sandbox = sinon.sandbox.create();

            this.sandbox.stub(BacklogController, "loadData").returns(deferred.resolve());
            this.sandbox.stub(GH.RapidBoard.QuickCreate, "showSubtaskCreatedMessage");
            this.sandbox.stub(BacklogController, "selectCreatedIssues").returns(true);
            this.sandbox.stub(BacklogController, "expandParents").returns(true);
        },
        teardown: function teardown() {
            GH.Notification.clear();
            this.sandbox.restore();
        }
    });

    test('subtasksCreated() always calls showSubtaskCreatedMessage for Scrum board', function () {
        GH.RapidViewConfig.currentData.data = {
            sprintSupportEnabled: true
        };
        BacklogController.subtasksCreated({});
        ok(GH.RapidBoard.QuickCreate.showSubtaskCreatedMessage.called);
    });

    test('subtasksCreated() does not call showSubtaskCreatedMessage for visible created subtasks', function () {
        GH.RapidViewConfig.currentData.data = {
            sprintSupportEnabled: false
        };
        sinon.stub(BacklogModel, 'hasAnyIssueInvisible').returns(false);
        BacklogController.subtasksCreated({});
        ok(GH.RapidBoard.QuickCreate.showSubtaskCreatedMessage.notCalled);
        BacklogModel.hasAnyIssueInvisible.restore();
    });

    test('subtasksCreated() should call showSubtaskCreatedMessage for invisible created subtasks', function () {
        GH.RapidViewConfig.currentData.data = {
            sprintSupportEnabled: false
        };
        sinon.stub(BacklogModel, 'hasAnyIssueInvisible').returns(true);
        BacklogController.subtasksCreated({});
        ok(GH.RapidBoard.QuickCreate.showSubtaskCreatedMessage.called);
        BacklogModel.hasAnyIssueInvisible.restore();
    });

    module('moveIssues', {
        setup: function setup() {
            this.issueKeys = ['KEY-1', 'KEY-2'];
            this.model = 'model';
            this.sandbox = sinon.sandbox.create();
            this.sandbox.stub(BacklogModel, 'hasAnySubtasks', function (issueKeys) {
                return _.contains(issueKeys, 'SUB-1');
            });
            this.sandbox.stub(BacklogModel, 'moveIssuesNew').returns(this.model);
            this.sandbox.stub(BacklogView, 'redrawChangedModels');
        },
        teardown: function teardown() {
            GH.Notification.clear();
            this.sandbox.restore();
        }
    });

    test('redraws changed model', sinon.test(function () {
        var shouldReorder = BacklogController.moveIssues(this.issueKeys, this.sprintId, 1000, 1001);

        ok(BacklogView.redrawChangedModels.calledWith(this.model), 'Expected BacklogView.redrawChangedModel to be called with the changed model');
        ok(!shouldReorder, 'Should not suggest a reorder');
    }));

    test('redraws changed model and suggests backlog re-ordered when selection includes subtask', sinon.test(function () {
        this.issueKeys.push('SUB-1');

        var shouldReorder = BacklogController.moveIssues(this.issueKeys, this.sprintId, 1000, 1001);

        ok(BacklogView.redrawChangedModels.calledWith(this.model), 'Expected BacklogView.redrawChangedModel to be called with the changed model');
        ok(shouldReorder, 'Should suggest a reorder');
    }));

    module('reorderIssuesInSprint', {
        setup: function setup() {
            this.model = 'model';
            sinon.stub(BacklogModel, 'reorderIssuesInModel').returns(this.model);
            sinon.stub(BacklogView, 'redrawChangedModels');
        },
        teardown: function teardown() {
            BacklogModel.reorderIssuesInModel.restore();
            BacklogView.redrawChangedModels.restore();
        }
    });

    test('redraws changed model', sinon.test(function () {
        BacklogController.reorderIssuesInSprint(this.issueKeys, this.sprintId);

        ok(BacklogView.redrawChangedModels.calledWith(this.model), 'Expected BacklogView.redrawChangedModel to be called with the changed model');
    }));

    module("Issue updated tests", {
        setup: function setup() {
            this.issueId = 101;

            BacklogController.reloadSingleIssue = sinon.spy();
            BacklogController.updateVersionData = sinon.spy();
            BacklogController.loadData = sinon.spy();
            BacklogModel.findModelWithIssue = sinon.stub().withArgs(this.issueId).returns({});

            BacklogController.visible = true;
        }
    });

    test('handleIssueUpdated: should not reload versions when fieldId is undefined', sinon.test(function () {
        var event = {};
        var data = {
            issueId: this.issueId,
            fieldId: undefined
        };

        BacklogController.handleIssueUpdated(event, data);

        sinon.assert.calledOnce(BacklogController.reloadSingleIssue);
        sinon.assert.calledWith(BacklogController.reloadSingleIssue, this.issueId);
        sinon.assert.notCalled(BacklogController.updateVersionData);
        sinon.assert.notCalled(BacklogController.loadData);
    }));
});