AJS.test.require(["com.pyxis.greenhopper.jira:gh-test-common", "com.pyxis.greenhopper.jira:gh-rapid"], function () {
    var BacklogSelectionController = require('jira-agile/rapid/ui/plan/backlog-selection-controller');

    test("module 'jira-agile/rapid/ui/plan/backlog-selection-controller' exists", function () {
        ok(BacklogSelectionController);
        ok(GH.BacklogSelectionController);
    });

    var SUBTASK_KEY = 'subtask';
    var PARENT_KEY = 'parent-key';
    var ISSUE_KEY = 'issue';
    var HIDDEN_ISSUE_KEY = 'hidden-issue-key';

    var SPRINT_ID = 'sprint-id';
    var ISSUE_1 = "issue-1";
    var ISSUE_2 = "issue-2";

    module('BacklogSelectionController', {
        setup: function setup() {
            this.sandbox = sinon.sandbox.create();

            this.context = AJS.test.mockableModuleContext();

            this.mockBacklogModel();
            this.mockSubtasksExpandingController();

            this.backlogSelectionController = this.context.require('jira-agile/rapid/ui/plan/backlog-selection-controller');
            this.backlogSelectionController.updateUIAndState = sinon.stub();
            this.backlogSelectionController.getSelectedIssueKeys = sinon.stub();
            this.backlogSelectionController.selectionManager = {
                removeSelectedIssueKeys: sinon.stub()
            };

            this.mockInitialData();
        },

        teardown: function teardown() {
            this.sandbox.restore();
        },

        /**
         * Mocking
         */
        mockSubtasksExpandingController: function mockSubtasksExpandingController() {
            this.SubtasksExpandingController = {
                isIssueCollapsed: sinon.stub()
            };

            this.context.mock('jira-agile/rapid/ui/plan/subtasks-expanding-controller', this.SubtasksExpandingController);
        },

        mockBacklogModel: function mockBacklogModel() {
            this.IssueList = {
                isSubtask: sinon.stub(),
                getIssueData: sinon.stub(),
                isIssueValid: sinon.stub()
            };
            this.BacklogModel = {
                getHiddenBySearchIssues: sinon.stub(),
                findModelWithIssue: function () {
                    return {
                        getIssueList: function () {
                            return this.IssueList;
                        }.bind(this)
                    };
                }.bind(this),
                findTargetModel: sinon.stub()
            };

            this.context.mock('jira-agile/rapid/ui/plan/backlog-model', this.BacklogModel);
            this.BacklogModel.getHiddenBySearchIssues.returns({});
            this.BacklogModel.findTargetModel.withArgs(SPRINT_ID).returns(this.BacklogModel.findModelWithIssue());
        },

        mockInitialData: function mockInitialData() {
            this.mockHiddenBySearchIssuesResponse(SUBTASK_KEY);
            this.IssueList.isSubtask.withArgs(SUBTASK_KEY).returns(true);
            this.IssueList.getIssueData.withArgs(SUBTASK_KEY).returns({
                parentKey: PARENT_KEY,
                key: SUBTASK_KEY
            });
            this.IssueList.getIssueData.withArgs(PARENT_KEY).returns({
                key: PARENT_KEY
            });
            this.IssueList.getIssueData.withArgs(ISSUE_KEY).returns({
                key: ISSUE_KEY
            });
            this.IssueList.getIssueData.withArgs(HIDDEN_ISSUE_KEY).returns({
                key: HIDDEN_ISSUE_KEY,
                hidden: true
            });

            this.IssueList.isIssueValid.withArgs(PARENT_KEY).returns(true);

            this.SubtasksExpandingController.isIssueCollapsed.withArgs(PARENT_KEY).returns(true);

            this.backlogSelectionController.getSelectedIssueKeys.returns([ISSUE_1, ISSUE_2]);
            this.IssueList.isIssueValid.withArgs(ISSUE_1).returns(true);
            this.IssueList.isIssueValid.withArgs(ISSUE_2).returns(true);
        },

        mockHiddenBySearchIssuesResponse: function mockHiddenBySearchIssuesResponse() {
            var args = Array.prototype.slice.call(arguments);

            var getHiddenBySearchIssues = {};
            args.forEach(function (issueKey) {
                getHiddenBySearchIssues[issueKey] = true;
            });

            this.BacklogModel.getHiddenBySearchIssues.returns(getHiddenBySearchIssues);
        }

    });

    test('#filterIssuesThatCanBeRanked model is null so all issues should be filtered', function () {
        this.BacklogModel.findModelWithIssue = function () {
            return null;
        };

        var result = this.backlogSelectionController.filterIssuesThatCanBeRanked([ISSUE_KEY]);

        deepEqual(result, []);
    });

    test('#filterIssuesThatCanBeRanked filtering not subtasks - should filter hidden issues', function () {
        this.mockHiddenBySearchIssuesResponse(PARENT_KEY);

        var result = this.backlogSelectionController.filterIssuesThatCanBeRanked([ISSUE_KEY, HIDDEN_ISSUE_KEY, PARENT_KEY]);

        deepEqual(result, [ISSUE_KEY]);
    });

    test('#filterIssuesThatCanBeRanked issue is subtask, parent is collapsed, not hidden and in the same column - subtask should be transitioned', function () {
        var result = this.backlogSelectionController.filterIssuesThatCanBeRanked([SUBTASK_KEY]);

        deepEqual(result, [SUBTASK_KEY]);
    });

    test('#filterIssuesThatCanBeRanked when parent is not valid (in different column) subtask should not be transitioned', function () {
        this.IssueList.isIssueValid.withArgs(PARENT_KEY).returns(false);

        var result = this.backlogSelectionController.filterIssuesThatCanBeRanked([SUBTASK_KEY]);

        deepEqual(result, []);
    });

    test('#filterIssuesThatCanBeRanked when parent is expanded subtask should not be transitioned', function () {
        this.IssueList.isIssueValid.withArgs(PARENT_KEY).returns(false);

        var result = this.backlogSelectionController.filterIssuesThatCanBeRanked([SUBTASK_KEY]);

        deepEqual(result, []);
    });

    test('#filterIssuesThatCanBeRanked when parent is expanded subtask should not be transitioned', function () {
        this.SubtasksExpandingController.isIssueCollapsed.withArgs(PARENT_KEY).returns(false);

        var result = this.backlogSelectionController.filterIssuesThatCanBeRanked([SUBTASK_KEY]);

        deepEqual(result, []);
    });

    test('#filterIssuesThatCanBeRanked when parent and children are hidden subtask should not be transitioned', function () {
        this.mockHiddenBySearchIssuesResponse(SUBTASK_KEY, PARENT_KEY);

        var result = this.backlogSelectionController.filterIssuesThatCanBeRanked([SUBTASK_KEY]);

        deepEqual(result, []);
    });

    test('#filterIssuesThatCanBeRanked when parent is hidden by quick search and subtask is hidden - subtask should not be transitioned', function () {
        this.IssueList.getIssueData.withArgs(PARENT_KEY).returns({
            key: PARENT_KEY,
            hidden: true
        });

        var result = this.backlogSelectionController.filterIssuesThatCanBeRanked([SUBTASK_KEY]);

        deepEqual(result, []);
    });

    test('#removeFromSelectionIssuesInDifferentModel should do nothing if model can\'t be found', function () {
        this.backlogSelectionController.removeFromSelectionIssuesInDifferentModel('unknown-sprint-id');

        ok(!this.backlogSelectionController.selectionManager.removeSelectedIssueKeys.called);
        ok(!this.backlogSelectionController.updateUIAndState.called);
    });

    test('#removeFromSelectionIssuesInDifferentModel should filter only issues in different model', function () {
        this.IssueList.isIssueValid.withArgs(ISSUE_1).returns(false);

        this.backlogSelectionController.removeFromSelectionIssuesInDifferentModel(SPRINT_ID);

        ok(this.backlogSelectionController.selectionManager.removeSelectedIssueKeys.calledOnce);
        deepEqual(this.backlogSelectionController.selectionManager.removeSelectedIssueKeys.getCall(0).args[0], [ISSUE_1]);
        ok(this.backlogSelectionController.updateUIAndState.calledOnce);
    });

    test('#removeFromSelectionIssuesInDifferentModel should not update ui if all selected issues are in current sprint', function () {
        this.backlogSelectionController.removeFromSelectionIssuesInDifferentModel(SPRINT_ID);

        ok(!this.backlogSelectionController.selectionManager.removeSelectedIssueKeys.called);
        ok(!this.backlogSelectionController.updateUIAndState.called);
    });
});