/**
 * Utility functions for issue lists
 * @module jira-agile/rapid/ui/plan/issue-list-util
 * @requires module:underscore
 */
define('jira-agile/rapid/ui/plan/issue-list-util', ['underscore', 'jquery'], function (_, $) {

    var issueListUtil = {};

    /**
     * Convert flat issue list to tree structure where subtask issues are positioned under their parents
     * @param {Array} issues issue list to convert
     * @param {Object} allLoadedIssues map of all loaded issues (as some of subtasks parents may be not visible on boards)
     * @returns {Array}
     */
    issueListUtil.issueListToTree = function (issues, allLoadedIssues) {
        var issueList = [];

        //deep copy array to avoid children data mutation
        issues = issues.map(function (issue) {
            return $.extend(true, {}, issue);
        });

        //add all parent tasks
        _.each(issues, function (issue, key) {
            if (!issue.parentKey) {
                issueList.push(issue);
            } else {
                //check if parent issue exist in list
                var parent = _.find(issueList, function (parent) {
                    return parent.key === issue.parentKey;
                });

                // check if parent exist, but ranked lower then child
                // e.g. ordering by updated date
                if (!parent) {
                    parent = _.find(issues, function (parent) {
                        return parent.key === issue.parentKey;
                    });

                    // insert fake parent
                    // add issue summary (if available) from allLoadedIssues map
                    if (!parent) {
                        parent = _.extend({}, allLoadedIssues[issue.parentKey], {
                            key: issue.parentKey,
                            fakeParent: true
                        });

                        issueList.push(parent);
                    }
                }

                if (!parent.subTasks) {
                    parent.subTasks = [];
                }

                parent.subTasks.push(issue);
            }
        });
        return issueList;
    };

    /**
     * Apply subtasks count to the issue - include hidden issues
     * @param {Array} issueList list of issues in tree structure (so subtasks are positioned under their parents)
     * @param {IssueListModel} issueListModel list model representing current issues scope
     */
    issueListUtil.applySubtaskCount = function (issueList, issueListModel, hiddenIssues) {
        // mark fake parent as filtered if all subtasks are filtered - will hide parent stub
        issueList.forEach(function (issue) {
            issue.subtasksCount = issueListModel.getSubtasksKeys(issue.key).length;

            var subtasks = issue.subTasks ? issue.subTasks : [];

            var hiddenBySearchSubtasksCount = subtasks.filter(function (subtask) {
                return hiddenIssues[subtask.key];
            }).length;

            issue.subtasksFiltered = subtasks.length === hiddenBySearchSubtasksCount;

            // hidden only by quick filters
            issue.hiddenByQuickFiltersSubtasks = issue.subtasksCount - subtasks.length;
            // hidden by both quick filters and search
            issue.hiddenSubtasks = issue.hiddenByQuickFiltersSubtasks + hiddenBySearchSubtasksCount;
        });
    };

    return issueListUtil;
});