AJS.test.require(["com.pyxis.greenhopper.jira:gh-test-common", "com.pyxis.greenhopper.jira:gh-rapid"], function () {
    var SprintController = require('jira-agile/rapid/ui/plan/sprint-controller');
    var BacklogController = require('jira-agile/rapid/ui/plan/backlog-controller');
    var BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');
    var BacklogView = require('jira-agile/rapid/ui/plan/BacklogView');
    var $ = require('jquery');

    var GlobalEvents = require('jira-agile/rapid/global-events');
    GlobalEvents.trigger('pre-initialization');

    module("Ajax tests", {
        setup: function setup() {
            this.sandbox = sinon.sandbox.create();
            GH.Test.setUpFakeServer(this);
            BacklogController.rapidViewData = {
                id: 1
            };
        },
        teardown: function teardown() {
            GH.Test.restoreServer(this);
            this.sandbox.restore();
        }
    });

    test('SprintController module exists', function () {
        ok(SprintController);
        ok(GH.SprintController);
    });

    test("addSprint: Successful AJAX call adds a new sprint in the backlog model and updates the view", sinon.test(function () {

        this.stub(BacklogModel, "addNewSprint");
        this.stub(BacklogView, "updateSprintsAfterAddRemove");

        var url = new RegExp("^" + GH.Ajax.buildRestUrl("/sprint/1"));
        this.server.respondWith("POST", url, [200, { "Content-Type": "application/json" }, JSON.stringify({})]);

        SprintController.addSprint();

        this.server.respond();

        ok(BacklogModel.addNewSprint.called, "Called success method");
        ok(BacklogView.updateSprintsAfterAddRemove.called, "Called success method");
    }));

    test("addSprint: Failed AJAX call does nothing (no error handling)", sinon.test(function () {

        this.stub(BacklogModel, "addNewSprint");
        this.stub(BacklogView, "updateSprintsAfterAddRemove");

        var url = new RegExp("^" + GH.Ajax.buildRestUrl("/sprint/1"));
        this.server.respondWith("POST", url, [500, { "Content-Type": "application/json" }, JSON.stringify({})]);

        SprintController.addSprint();

        this.server.respond();

        ok(!BacklogModel.addNewSprint.called, "Nothing called when ajax fails");
        ok(!BacklogView.updateSprintsAfterAddRemove.called, "Nothing called when ajax fails");
    }));

    test("doDeleteSprint: Sends correct Ajax call", sinon.test(function () {

        this.stub(BacklogModel, "removeSprintAndMoveIssues");
        this.stub(BacklogView, "updateSprintsAfterAddRemove");

        var url = new RegExp("^" + GH.Ajax.buildRestUrl("/sprint/6"));
        this.server.respondWith("DELETE", url, [200, { "Content-Type": "application/json" }, JSON.stringify({})]);

        SprintController.doDeleteSprint(5, 6);

        this.server.respond();

        ok(BacklogModel.removeSprintAndMoveIssues.called);
        ok(BacklogView.updateSprintsAfterAddRemove.called);
    }));

    test("moveSprintDown: Successful AJAX call updates the backlog model and updates the view", sinon.test(function () {

        var sprintId = 42,
            otherSprintId = 54;
        this.sandbox.stub(BacklogModel, "moveSprintDown").withArgs(sprintId).returns(otherSprintId);
        this.sandbox.stub(BacklogView, "updateSprintsAfterAddRemove");

        var url = new RegExp("^" + GH.Ajax.buildRestUrl("/sprint/" + sprintId + "/swap"));
        this.server.respondWith("PUT", url, [204, { "Content-Type": "application/json" }, JSON.stringify({})]);

        SprintController.moveSprintDown(sprintId);

        this.server.respond();

        ok(BacklogModel.moveSprintDown.called, "Called mutating method");
        ok(BacklogView.updateSprintsAfterAddRemove.called, "Called success method");
    }));

    test("moveSprintDown: Failed AJAX call updates the backlog model twice and updates the view twice", sinon.test(function () {

        var sprintId = 42,
            otherSprintId = 54,
            moveSprintDownExpectation = this.sandbox.mock(BacklogModel);

        moveSprintDownExpectation.expects("moveSprintDown").once().withArgs(sprintId).returns(otherSprintId);
        moveSprintDownExpectation.expects("moveSprintDown").once().withArgs(otherSprintId).returns(sprintId);

        this.sandbox.stub(BacklogView, "updateSprintsAfterAddRemove");

        var url = new RegExp("^" + GH.Ajax.buildRestUrl("/sprint/" + sprintId + "/swap"));
        this.server.respondWith("PUT", url, [401, { "Content-Type": "application/json" }, JSON.stringify({})]);

        SprintController.moveSprintDown(sprintId);

        this.server.respond();

        moveSprintDownExpectation.verify();
        sinon.assert.callCount(BacklogView.updateSprintsAfterAddRemove, 2);
    }));

    test("moveSprintUp: Successful AJAX call updates the backlog model and updates the view", sinon.test(function () {

        var sprintId = 42,
            otherSprintId = 54;
        this.sandbox.stub(BacklogModel, "moveSprintUp").withArgs(sprintId).returns(otherSprintId);
        this.sandbox.stub(BacklogView, "updateSprintsAfterAddRemove");

        var url = new RegExp("^" + GH.Ajax.buildRestUrl("/sprint/" + sprintId + "/swap"));
        this.server.respondWith("PUT", url, [204, { "Content-Type": "application/json" }, JSON.stringify({})]);

        SprintController.moveSprintUp(sprintId);

        this.server.respond();

        ok(BacklogModel.moveSprintUp.called, "Called mutating method");
        ok(BacklogView.updateSprintsAfterAddRemove.called, "Called success method");
    }));

    test("moveSprintUp: Failed AJAX call updates the backlog model twice and updates the view twice", sinon.test(function () {

        var sprintId = 42,
            otherSprintId = 54,
            moveSprintUpExpectation = this.sandbox.mock(BacklogModel);

        moveSprintUpExpectation.expects("moveSprintUp").once().withArgs(sprintId).returns(otherSprintId);
        moveSprintUpExpectation.expects("moveSprintUp").once().withArgs(otherSprintId).returns(sprintId);

        this.sandbox.stub(BacklogView, "updateSprintsAfterAddRemove");

        var url = new RegExp("^" + GH.Ajax.buildRestUrl("/sprint/" + sprintId + "/swap"));
        this.server.respondWith("PUT", url, [401, { "Content-Type": "application/json" }, JSON.stringify({})]);

        SprintController.moveSprintUp(sprintId);

        this.server.respond();

        moveSprintUpExpectation.verify();
        sinon.assert.callCount(BacklogView.updateSprintsAfterAddRemove, 2);
    }));

    var activeSprint = { state: "ACTIVE" };
    var futureSprint = { state: "FUTURE" };

    module("trigger analytics events");

    test("Move between active sprints", sinon.test(function () {

        this.stub(SprintController.analytics, "trigger");

        var moveBetweenActiveSprintsModel = {
            changes: [{
                currentSprint: activeSprint,
                targetSprint: activeSprint
            }]
        };

        SprintController.triggerAnalytics(moveBetweenActiveSprintsModel);

        ok(SprintController.analytics.trigger.calledWith("movebetweenactivesprints"));
    }));

    test("Move between future sprints", sinon.test(function () {

        this.stub(SprintController.analytics, "trigger");

        var moveBetweenFutureSprintsModel = {
            changes: [{
                currentSprint: futureSprint,
                targetSprint: futureSprint
            }]
        };

        SprintController.triggerAnalytics(moveBetweenFutureSprintsModel);

        ok(SprintController.analytics.trigger.calledWith("movebetweenfuturesprints"));
    }));

    test("Move from future to active sprint", sinon.test(function () {

        this.stub(SprintController.analytics, "trigger");

        var moveFromFutureToActiveSprintModel = {
            changes: [{
                currentSprint: futureSprint,
                targetSprint: activeSprint
            }]
        };

        SprintController.triggerAnalytics(moveFromFutureToActiveSprintModel);

        ok(SprintController.analytics.trigger.calledWith("movefromfuturetoactivesprint"));
    }));

    test("Move from active to future sprint", sinon.test(function () {

        this.stub(SprintController.analytics, "trigger");

        var moveFromActiveToFutureSprintModel = {
            changes: [{
                currentSprint: activeSprint,
                targetSprint: futureSprint
            }]
        };

        SprintController.triggerAnalytics(moveFromActiveToFutureSprintModel);

        ok(SprintController.analytics.trigger.calledWith("movefromactivetofuturesprint"));
    }));

    test("Move in backlog", sinon.test(function () {

        this.stub(SprintController.analytics, "trigger");

        var noSprintsModel = {
            changes: [{
                currentSprint: null,
                targetSprint: null
            }]
        };

        SprintController.triggerAnalytics(noSprintsModel);

        ok(!SprintController.analytics.trigger.called);
    }));

    test("Add to future sprint", sinon.test(function () {

        this.stub(SprintController.analytics, "trigger");

        var addToFutureSprintModel = {
            changes: [{
                currentSprint: null,
                targetSprint: futureSprint
            }]
        };

        SprintController.triggerAnalytics(addToFutureSprintModel);

        ok(SprintController.analytics.trigger.calledWith("addtofuturesprint"));
    }));

    test("Add to active sprint", sinon.test(function () {

        this.stub(SprintController.analytics, "trigger");

        var addToActiveSprintModel = {
            changes: [{
                currentSprint: null,
                targetSprint: activeSprint
            }]
        };

        SprintController.triggerAnalytics(addToActiveSprintModel);

        ok(SprintController.analytics.trigger.calledWith("addtoactivesprint"));
    }));

    test("Remove from future sprint", sinon.test(function () {

        this.stub(SprintController.analytics, "trigger");

        var removeFromFutureSprintModel = {
            changes: [{
                currentSprint: futureSprint,
                targetSprint: null
            }]
        };

        SprintController.triggerAnalytics(removeFromFutureSprintModel);

        ok(SprintController.analytics.trigger.calledWith("removefromfuturesprint"));
    }));

    test("Remove from active sprint", sinon.test(function () {

        this.stub(SprintController.analytics, "trigger");

        var removeFromActiveSprintModel = {
            changes: [{
                currentSprint: activeSprint,
                targetSprint: null
            }]
        };

        SprintController.triggerAnalytics(removeFromActiveSprintModel);

        ok(SprintController.analytics.trigger.calledWith("removefromactivesprint"));
    }));

    module("ColumnConfigAnalytics tests", {
        setup: function setup() {
            this.context = AJS.test.mockableModuleContext();

            this.jiraAnalytics = {
                send: sinon.spy()
            };
            this.RAPID_VIEW_ID = 23;
            this.BacklogController = {
                rapidViewData: {
                    id: this.RAPID_VIEW_ID
                }
            };

            this.context.mock('jira/analytics', this.jiraAnalytics);
            this.context.mock('jira-agile/rapid/ui/plan/backlog-controller', this.BacklogController);
            this.context.mock('jira-agile/rapid/global-events', GlobalEvents);
            this.sprintController = this.context.require("jira-agile/rapid/ui/plan/sprint-controller");

            GlobalEvents.trigger('pre-initialization');
        },
        teardown: function teardown() {},

        assertTrigger: function assertTrigger(properties) {
            ok("Spy should be called once", this.jiraAnalytics.send.calledOnce);

            var options = this.jiraAnalytics.send.getCall(0).args[0];
            equal(options.name, "jira-software.scrum.plan.issuecard.dnd", "Name of event should be valid");
            deepEqual(options.properties, properties, "Properties should be equal");
        }
    });

    test('should send target and current sprint id', function () {
        var issueMoveModel = {
            changes: [{
                currentSprint: {
                    id: 'currentSprintId'
                },
                targetSprint: {
                    id: 'targetSprintId'
                },
                issueKeys: ['a'],
                doneIssueKeys: []
            }]
        };

        this.sprintController.triggerAnalyticsIssuesDnD(issueMoveModel);

        this.assertTrigger({
            issues: 1,
            currentSprintId: 'currentSprintId',
            targetSprintId: 'targetSprintId',
            boardId: this.RAPID_VIEW_ID
        });
    });

    test('should send -1 when sprint is null', function () {
        var issueMoveModel = {
            changes: [{
                targetSprint: null,
                issueKeys: ['a'],
                doneIssueKeys: ['b']
            }]
        };

        this.sprintController.triggerAnalyticsIssuesDnD(issueMoveModel);

        this.assertTrigger({
            issues: 2,
            currentSprintId: -1,
            targetSprintId: -1,
            boardId: this.RAPID_VIEW_ID
        });
    });
});