AJS.test.require(["com.pyxis.greenhopper.jira:gh-test-common-base", "com.pyxis.greenhopper.jira:gh-rapid"], function () {
    var $ = require('jquery');
    var SprintView = require('jira-agile/rapid/ui/plan/sprint-view');
    var BacklogController = require('jira-agile/rapid/ui/plan/backlog-controller');
    var BacklogStatistics = require('jira-agile/rapid/ui/plan/backlog-statistics');
    var IssueListModel = require('jira-agile/rapid/ui/plan/issue-list-model');
    var SprintModel = require('jira-agile/rapid/ui/plan/sprint-model');
    var BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');
    var FeatureFlagManager = require('jira/featureflags/feature-manager');

    var GlobalEvents = require('jira-agile/rapid/global-events');
    GlobalEvents.trigger('pre-initialization');

    module("Sprint View Statistics", {
        setup: function() {
            sinon.stub(BacklogStatistics, "getIssueListStats").returns({issueListStats: "issueListStats"});
            sinon.stub(BacklogStatistics, "getIssueListEstimateStatistics").returns({estimateStats: "estimateStats"});
            sinon.stub(BacklogStatistics, "getAssignedWorkStats").returns("assignedWorkStats");
            sinon.stub(BacklogModel, "getAllIssuesWithMissingParents").returns({});
        },
        teardown: function() {
            BacklogStatistics.getIssueListStats.restore();
            BacklogStatistics.getIssueListEstimateStatistics.restore();
            BacklogStatistics.getAssignedWorkStats.restore();
            BacklogModel.getAllIssuesWithMissingParents.restore();
        }
    });

    test('SprintView module exists', function () {
        ok(SprintView);
        ok(GH.SprintView);
    });

    test("gets issue stats plus estimation stats for a future sprint", function () {
        var sprintModel = givenSprintModel({isFutureSprint: true, isActiveSprint: false});

        var data = SprintView.calculateSprintIssueListStats(sprintModel, {});

        var expectedData = {
            issueListStats: "issueListStats",
            estimateStats: "estimateStats",
            assignedWorkStats: "assignedWorkStats"
        };

        deepEqual(data, expectedData);

    });

    test("calculates progress statistics for active sprint based", function () {
        var sprintModel = givenSprintModel({isFutureSprint: false, isActiveSprint: true});

        BacklogStatistics.calculateSprintProgress = function() { return {progress: "progress"}; };

        BacklogController.rapidViewConfig = {
            columns: []
        };

        var data = SprintView.calculateSprintIssueListStats(sprintModel, {});

        var expectedData = {
            issueListStats: "issueListStats",
            progressIndicators: {progress: "progress"},
            assignedWorkStats: "assignedWorkStats"
        };

        deepEqual(data, expectedData);

    });

    test('Epic label should show on sprint issues', function() {
        QUnit.GhFixtures.addFixture("<div id='ghx-backlog'><div class='ghx-sprint-group'></div></div>");

        BacklogController.rapidViewConfig = {
            columns: []
        };

        var epicKey = 'TEST';

        var issues = GH.IssueTest.createArrayOfIssues(2);
        var issueKey1 = issues[0].key;
        var issueKey2 = issues[1].key;
        issues[0].epic = epicKey;

        var sprintModel = new SprintModel({}, issues);
        this.stub(BacklogModel, 'getSprintModels').returns([sprintModel]);

        var epics = [GH.IssueTest.createIssueObject({key: epicKey, 'epicLabel': 'TEST Epic'})];
        var mockEpicModel = new GH.EpicModel();
        mockEpicModel.epicsList = new IssueListModel('test', epics, null);
        this.stub(GH.EpicController, 'getEpicModel').returns(mockEpicModel);

        var issueRenderData = {hiddenIssues: {}, selectedIssueKeys: {}};
        SprintView.renderAllSprints(issueRenderData);

        ok($('.js-issue[data-issue-key=' + issueKey1 + ']').find('span[data-epickey=' + epicKey + ']').length > 0);
        ok($('.js-issue[data-issue-key=' + issueKey2 + ']').find('span[data-epickey=' + epicKey + ']').length === 0);
    });


    module('calculateSprintIssueListStats', {
        setup: function () {
            sinon.stub(BacklogStatistics, 'calculateStatisticsFieldValue');
            sinon.stub(BacklogStatistics, 'calculateSprintProgress');
            sinon.stub(BacklogStatistics, 'getIssueListStats').returns({});
            BacklogController.rapidViewConfig = {
                columns: []
            };
            this.sprintModel = givenSprintModel({isFutureSprint: false, isActiveSprint: false});
        },
        teardown: function () {
            BacklogStatistics.calculateStatisticsFieldValue.restore();
            BacklogStatistics.calculateSprintProgress.restore();
            BacklogStatistics.getIssueListStats.restore();
            this.sprintModel = null;
        }
    });

    test('calculateSprintIssueListStats() should call calculateStatisticsFieldValue for Kanban board', function() {
        GH.RapidViewConfig.currentData.data = {sprintSupportEnabled: false};
        var issueList = new IssueListModel("model", [], new GH.VersionsModel([]));
        SprintView.calculateSprintIssueListStats(this.sprintModel, issueList);
        ok(BacklogStatistics.calculateStatisticsFieldValue.called);
    });

    test('calculateSprintIssueListStats() should not call calculateStatisticsFieldValue for Scrum board', function() {
        GH.RapidViewConfig.currentData.data = {sprintSupportEnabled: true};
        var issueList = new IssueListModel("model", [], new GH.VersionsModel([]));
        SprintView.calculateSprintIssueListStats(this.sprintModel, issueList);
        ok(!BacklogStatistics.calculateStatisticsFieldValue.called);
    });

    module('Sprint goal onboarding', {
        setup () {
            this.sandbox = sinon.sandbox.create();
            this.sandbox.stub(FeatureFlagManager, 'isFeatureEnabled').withArgs('com.atlassian.jira.agile.darkfeature.sprint.goal').returns(true);
            SprintView.sprintGoalOnboardingState = {isSprintGoalOnboardingClosed: false};
            this.sandbox.sprintModel = givenSprintModel({id: 1, sprintGoal: null});
        },
        teardown () {
            this.sandbox.sprintModel = null;
            this.sandbox.restore();
        }
    });

    test('should display sprint goal onboarding correctly', function () {
        ok(SprintView.shouldDisplaySprintGoalOnboarding(this.sandbox.sprintModel));
    });

    test('should display sprint goal only for the first sprint', function () {
        this.sandbox.stub(BacklogModel,'getSprintModels').returns([this.sandbox.sprintModel, givenSprintModel({id: 2, sprintGoal: null}) ]);
        this.sandbox.stub(SprintView,'_renderSprintContent');
        this.sandbox.spy(SprintView,'displaySprintGoalOnboarding');
        SprintView.renderAllSprints({});
        ok(SprintView.displaySprintGoalOnboarding.called);
        equal(SprintView.displaySprintGoalOnboarding.callCount, 1);
        ok(SprintView.displaySprintGoalOnboarding.calledWith(this.sandbox.sprintModel.getSprintId()));
    });

    test('should hide sprint goal onboarding when receiving "before:navigate" event', function () {
        this.sandbox.stub(BacklogModel, 'getSprintModels').returns([this.sandbox.sprintModel, givenSprintModel({id: 2, sprintGoal: null})]);
        this.sandbox.stub(SprintView, '_renderSprintContent');
        this.sandbox.spy(SprintView, 'displaySprintGoalOnboarding');
        this.sandbox.spy(SprintView, '_hideSprintGoalOnboardingDialog');

        SprintView.renderAllSprints({});
        sinon.assert.calledOnce(SprintView.displaySprintGoalOnboarding);

        GH.RapidBoard.ViewController.trigger('before:navigate');
        sinon.assert.calledOnce(SprintView._hideSprintGoalOnboardingDialog);
    });

    test('should not display sprint goal onboarding when there are no sprints', function () {
        this.sandbox.stub(BacklogModel,'getSprintModels').returns([]);
        this.sandbox.spy(SprintView,'displaySprintGoalOnboarding');
        SprintView.renderAllSprints({});
        ok(SprintView.displaySprintGoalOnboarding.notCalled);
    });

    test('should not display sprint goal onboarding when goal is set', function () {
        var sprintModel = this.sandbox.sprintModel;
        sprintModel.sprintGoal = "Sprint goal";
        notOk(SprintView.shouldDisplaySprintGoalOnboarding(sprintModel));
    });

    test('should not display sprint goal onboarding when its dismissed previously', function () {
        SprintView.sprintGoalOnboardingState.isSprintGoalOnboardingClosed = true;
        var sprintModel = this.sandbox.sprintModel;
        notOk(SprintView.shouldDisplaySprintGoalOnboarding(sprintModel));
    });

    test('should not display sprint goal onboarding when feature is disabled', function () {
        FeatureFlagManager.isFeatureEnabled.withArgs('com.atlassian.jira.agile.darkfeature.sprint.goal').returns(false);
        var sprintModel = this.sandbox.sprintModel;
        notOk(SprintView.shouldDisplaySprintGoalOnboarding(sprintModel));
    });

    function givenSprintModel (data) {
        return {
            isFutureSprint () {
                return data.isFutureSprint;
            },
            isActiveSprint () {
                return data.isActiveSprint;
            },
            getSprintId() {
                return data.id;
            },
            getSprintData () {
                return {goal: this.sprintGoal}
            },
            getIssueList () {
                return data.issueList || [];
            }
        };
    }
});