define('jira-agile/rapid/ui/work/grid-columns-data', [
    'underscore'
], function (
    _
) {
    "use strict";

    /**
     * Contains the logic for preprocessing swimlane information
     */
    const GridColumnsData = {};

    /**
     * Calculates column data.
     * @param issuesData
     * @param swimlanesData
     */
    GridColumnsData.calculateColumnsData = function (issuesData, columnsData, statistics) {
        var columns = columnsData.columns;

        // do the busted stuff
        var anyColumnBusted = GridColumnsData.computeBusted(columns, statistics);

        // issue mapping
        var columnIssueMapping = GridColumnsData.getColumnIssueMapping(issuesData, columns);

        var data = {
            columns: columns,
            columnIssueMapping: columnIssueMapping,
            anyColumnBusted: anyColumnBusted
        };
        return data;
    };

    /**
     * Get the column information and columnIssueMapping
     */
    GridColumnsData.getColumnIssueMapping = function (issuesData, columns) {
        // define the columns
        var columnIssueMapping = {};
        var statusToColumn = {};
        var columnStats = {};
        _.each(columns, function (column) {
            columnIssueMapping[column.id] = [];
            columnStats[column.id] = {
                'visible': 0,
                'total': column.statisticsFieldValue || 0
            };
            _.each(column.statusIds, function (statusId) {
                statusToColumn[statusId] = column.id;
            });
        });

        // calculate the mapping from each issue to the associated column
        // accumulate the total of visible stats for the column
        _.each(issuesData.issues, function (issue) {
            var columnId = statusToColumn[issue.statusId];
            if (columnIssueMapping[columnId]) {
                columnIssueMapping[columnId].push(issue.key);
                if (issue.columnStatistic && issue.columnStatistic.statFieldValue) {
                    columnStats[columnId].visible += issue.columnStatistic.statFieldValue.value;
                }
            }
        });

        _.each(columns, function (column) {
            column.stats = columnStats[column.id];
        });
        return columnIssueMapping;
    };

    /**
     * Compute information about column busted state, based on the statistics limits
     */
    GridColumnsData.computeBusted = function (columns, statistics) {
        // do we have to calculate the busted state?
        // skip if no statistics has been configured
        if (!statistics.fieldConfigured) {
            return false;
        }

        // min/max busted for columns
        var anyColumnBusted = false;
        _.each(columns, function (column) {
            column.minBusted = (typeof column.min != 'undefined') && column.min > column.statisticsFieldValue;
            column.maxBusted = (typeof column.max != 'undefined') && column.max < column.statisticsFieldValue;

            // any column busted
            anyColumnBusted = anyColumnBusted || column.minBusted || column.maxBusted;
        });
        return anyColumnBusted;
    };

    GridColumnsData.getColumnIdForStatus = function (columns, status) {
        var column = _.find(columns, function (column) {
            return _.find(column.statusIds, function (statusId) {
                if (statusId == status) {
                    return true;
                }
            });
        });
        return column ? column.id : false;
    };

    return GridColumnsData;
});

AJS.namespace('GH.GridColumnsData', null, require('jira-agile/rapid/ui/work/grid-columns-data'));