define('jira-agile/rapid/ui/work/grid-data-controller', [
    'underscore',
    'jira-agile/rapid/ui/work/grid-data-model',
    'jira-agile/rapid/ui/work/grid-columns-data',
    'jira-agile/rapid/ui/work/grid-swimlanes-data',
    'jira-agile/rapid/ui/work/ranking-model'
], function (
    _,
    GridDataModel,
    GridColumnsData,
    GridSwimlanesData,
    RankingModel
) {
    "use strict";

    /**
     * Controller for the grid data model
     */
    const GridDataController = {};

    GridDataController.model = undefined;
    GridDataController.swimlaneStrategy = 'custom';
    GridDataController.statistics = {};
    GridDataController.canRelease = false;

    /**
     * Compare the passed data with data currently stored in the controller
     * This function is used to validate whether data retrieved from the server is already reflecting
     * client state, and if so discard the update as a no-op.
     *
     * @return true if the data is different, false otherwise
     */
    GridDataController.hasDataChanged = function(data) {
        var model = GridDataController.calculateGridDataModel(data);

        // compare to the current model
        if (GridDataController.model) {
            return GridDataController.model.isModelDifferent(model);
        } else {
            return true;
        }
    };

    GridDataController.setData = function(data) {
        // set the statistics
        GridDataController.statistics = data.statistics;

        // set the epics data
        GridDataController.epics = data.epicData;

        // can the user release
        GridDataController.canRelease = data.issuesData.canRelease;

        // set the order data (defines ranking capability)
        RankingModel.setOrderData(data.orderData);

        GridDataController.hasBulkChangePermission = data.issuesData.hasBulkChangePermission;

        // calculate a GridDataModel off the input
        var model = GridDataController.calculateGridDataModel(data);
        GridDataController.model = model;
    };

    GridDataController.calculateGridDataModel = function(data) {
        // calculate the swimlanesData
        var swimlanesData = GridSwimlanesData.calculateSwimlanesData(data.issuesData, data.swimlanesData);

        // calculate the issue lookup and order array
        var order = [];
        var issues = {};

        // add the issues to order and issues
        _.each(data.issuesData.issues, function(issue) {
            order.push(issue.key);
            issues[issue.key] = issue;
        });
        // add the hidden parents to issues
        _.each(data.issuesData.missingParents, function(issue) {
            issues[issue.key] = issue;
        });

        // calulate the column data
        var columnsData = GridColumnsData.calculateColumnsData(data.issuesData, data.columnsData, data.statistics);
        var columns = columnsData.columns;
        var columnIssueMapping = columnsData.columnIssueMapping;

        var variables = {};
        variables.anyColumnBusted = columnsData.anyColumnBusted;

        // put together the swimlanes and swimlaneIssueMapping
        var swimlanes = [];
        var swimlaneIssueMapping = {};
        _.each(swimlanesData.swimlanes, function(swimlane) {
            swimlaneIssueMapping[swimlane.id] = swimlane.issueKeys;
            delete swimlane.issueKeys; // don't keep the issueKeys in the swimlane
            swimlanes.push(swimlane);
        });
        variables.swimlanesType = swimlanesData.type;

        // create the model
        return new GridDataModel(columns, columnIssueMapping, swimlanes, swimlaneIssueMapping, issues, order, variables, data.swimlanesData.swimlaneStrategy);
    };

    /**
     * Get the statistics field configuration
     */
    GridDataController.getStatistics = function() {
        return GridDataController.statistics;
    };

    /**
     * Get the epics count
     */
    GridDataController.getEpicsCount = function() {
        return GridDataController.epics.epicsCount;
    };

    /**
     * Get the current model
     */
    GridDataController.getModel = function() {
        return GridDataController.model;
    };

    /**
     * Is at least one column busted?
     */
    GridDataController.hasBustedColumns = function() {
        return GridDataController.model.data.variables.anyColumnBusted;
    };

    /**
     * Is at least one column busted?
     */
    GridDataController.getSwimlanesType = function() {
        return GridDataController.model.data.variables.swimlanesType;
    };

    GridDataController.collectPrintableIssues = function() {
        var model = GridDataController.getModel();

        // issues are printed in order: columns: left to right; on each column: top to bottom
        var issueKeys = [];
        _.each(model.getSwimlanes(), function(swimlane) {
            _.each(model.getColumns(), function(column) {
                var arr = model.getCells()[swimlane.id][column.id];
                if (arr) {
                    issueKeys.push.apply(issueKeys, arr);
                }
            });
        });

        return _.map(issueKeys, function(issueKey) {
            return model.getIssues()[issueKey];
        });
    };

    return GridDataController;
});

AJS.namespace('GH.GridDataController', null, require('jira-agile/rapid/ui/work/grid-data-controller'));
