define('jira-agile/rapid/ui/work/work-kb-navigation', [
    "jira-agile/rapid/ui/work/work-controller",
    "jira-agile/rapid/ui/work/swimlane-view",
    "jira-agile/rapid/ui/work/grid-data-controller",
    "jira-agile/rapid/logger",
    "jquery"
], function(WorkController, SwimlaneView, GridDataController, Logger, $) {
    "use strict";

    /**
     * Navigation keyboard shortcuts
     */
    const WorkKBNavigation = {};

    /**
     * Holds the position of the currently selected issue
     *
     * Used for column traversing to not reset the selected index every time we pass through a column containing less items,
     * plus for j/k traversing with parent swimlanes
     */
    WorkKBNavigation.currentIndex = 0;
    WorkKBNavigation.currentColumn = 0;
    WorkKBNavigation.currentSwimlane = 0;

    /**
     * True whenever a column change has been performed, in which case the index/swimlane should not be updated
     */
    WorkKBNavigation.columnChange = false;

    /**
     * Called on page load.
     */
    WorkKBNavigation.init = function() {
        // update the magic index on issue selection
        $(GH).bind('issueSelected', function(event, issueKey) {
            Logger.log(event.type + " handled", Logger.Contexts.ui);
            WorkKBNavigation.adaptCurrentPosition(issueKey);
        });
    };


    /**
     * Initializes the magic index according to the currently selected issue.
     */
    WorkKBNavigation.initCurrentPosition = function() {
        var issueKey = GH.WorkSelectionController.getSelectedIssueKey();
        if (issueKey) {
            WorkKBNavigation.setCurrentPosition(issueKey, true, false);
        } else {
            WorkKBNavigation.resetCurrentPosition();
        }
    };

    /**
     * Updates the selected issue index - if t. */
    WorkKBNavigation.adaptCurrentPosition = function(issueKey) {
        // skip if this is a column change
        if (WorkKBNavigation.columnChange) {
            WorkKBNavigation.columnChange = false;
            WorkKBNavigation.setCurrentPosition(issueKey, false, true);
        } else {
            WorkKBNavigation.setCurrentPosition(issueKey, false, false);
        }
    };

    /**
     * Sets the magic index to the index of the provided issue
     */
    WorkKBNavigation.setCurrentPosition = function(issueKey, forceSetColumn, dontSetIndex) {
        if (issueKey) {
            var data = GridDataController.getModel()._getIssuePositionAndDataByKey(issueKey);
            if (data) {
                if(!dontSetIndex) {
                    WorkKBNavigation.currentIndex = data.issueIndex;
                }
                if (forceSetColumn || !data.isSwimlane) {
                    WorkKBNavigation.currentColumn = data.columnId;
                }
                WorkKBNavigation.currentSwimlane = data.swimlaneId;
                return;
            }
        }

        WorkKBNavigation.resetCurrentPosition();
    };

    /**
     * Sets the magic index to the index of the provided issue
     */
    WorkKBNavigation.resetCurrentPosition = function() {
        WorkKBNavigation.currentIndex = 0;
        WorkKBNavigation.currentColumn = 0;
        WorkKBNavigation.currentSwimlane = 0;
    };

    /**
     * Select the next issue in a current column, or the first issue in the first column if none selected.
     */
    WorkKBNavigation.selectNextIssue = function() {
        if(!WorkController.isActive()) {
            return;
        }

        var issueKey = GH.WorkSelectionController.getSelectedIssueKey();
        var model = GridDataController.getModel();
        if (!issueKey) {
            // try selecting the first issue in the view
            var firstIssueKey = model.getFirstIssueKey();
            if (firstIssueKey) {
                GH.WorkSelectionController.focusOnIssue(firstIssueKey);
            }
        } else if (!SwimlaneView.ensureSwimlaneExpandedForIssueKey(issueKey)) {
            // only select next issue if there is one
            var nextIssueKey = model.getNextIssueKey(issueKey, WorkKBNavigation.currentColumn);
            if (nextIssueKey) {
                GH.WorkSelectionController.focusOnIssue(nextIssueKey);
            }
        }
    };

    /**
     * Select the previous issue in the current, or the last issue in the first column if none selected.
     */
    WorkKBNavigation.selectPreviousIssue = function() {
        if(!WorkController.isActive()) {
            return;
        }

        var issueKey = GH.WorkSelectionController.getSelectedIssueKey();
        var model = GridDataController.getModel();
        if (!issueKey) {
            // try selecting the last issue in the view
            var lastIssueKey = model.getLastIssueKey();
            if (lastIssueKey) {
                GH.WorkSelectionController.focusOnIssue(lastIssueKey);
            }
        } else if (!SwimlaneView.ensureSwimlaneExpandedForIssueKey(issueKey)) {
            // only select previous issue if there is one
            var prevIssueKey = model.getPreviousIssueKey(issueKey, WorkKBNavigation.currentColumn);
            if (prevIssueKey) {
                GH.WorkSelectionController.focusOnIssue(prevIssueKey);
            }
        }
    };

    /**
     * Select an issue in the next column or the first issue in the first column if none selected.
     * The function attempts to keep the selected index.
     */
    WorkKBNavigation.selectNextColumn = function() {
        if (! WorkController.isActive()) {
            return;
        }

        var issueKey = GH.WorkSelectionController.getSelectedIssueKey();

        // simply select the first issue if we haven't got one
        if (!issueKey) {
            WorkKBNavigation.selectNextIssue();
            return;
        }

        // fetch the data for the issue, which will give us the swimlane id
        var model = GridDataController.getModel();
        var issueValid = model.isIssueValid(issueKey);
        if (!issueValid) {
            WorkKBNavigation.selectNextIssue();
            return;
        }

        // expand swimlane if necessary
        if (SwimlaneView.ensureSwimlaneExpandedForIssueKey(issueKey)) {
            return;
        }

        var issueKeyToSelect = model.getIssueKeyInNextColumn(issueKey, WorkKBNavigation.currentIndex);
        if(issueKeyToSelect) {
            WorkKBNavigation.columnChange = true;
            GH.WorkSelectionController.focusOnIssue(issueKeyToSelect);
        }

    };

    /**
     * Select an issue in the previous column or the first issue in the last column if none selected.
     * The function attempts to keep the selected index.
     */
    WorkKBNavigation.selectPreviousColumn = function() {
        if (! WorkController.isActive()) {
            return;
        }

        var issueKey = GH.WorkSelectionController.getSelectedIssueKey();

        // simply select the first issue if we haven't got one
        if (!issueKey) {
            WorkKBNavigation.selectNextIssue();
            return;
        }

        // fetch the data for the issue, which will give us the swimlane id
        var model = GridDataController.getModel();
        var issueValid = model.isIssueValid(issueKey);
        if (!issueValid) {
            WorkKBNavigation.selectNextIssue();
            return;
        }

        // expand swimlane if necessary
        if (SwimlaneView.ensureSwimlaneExpandedForIssueKey(issueKey)) {
            return;
        }

        var issueKeyToSelect = model.getIssueKeyInPreviousColumn(issueKey, WorkKBNavigation.currentIndex);
        if(issueKeyToSelect) {
            WorkKBNavigation.columnChange = true;
            GH.WorkSelectionController.focusOnIssue(issueKeyToSelect);
        }
    };

    WorkKBNavigation.toggleSwimlanes = function() {
        // let swimlane view handle this
        SwimlaneView.toggleAllSwimlanes();
    };

    return WorkKBNavigation;
});

AJS.namespace('GH.WorkKBNavigation', null, require('jira-agile/rapid/ui/work/work-kb-navigation'));
