AJS.test.require(['com.pyxis.greenhopper.jira:gh-rapid', 'com.pyxis.greenhopper.jira:subnavigator-test'], function() {
    AJS.namespace('JIRA.Projects');

    var FeatureFlagManager = require('jira/featureflags/feature-manager');

    module('Subnavigator.tests', {
        setup: function () {
            this.$qunitFixture = AJS.$('#qunit-fixture');

            this.sandbox = sinon.sandbox.create();
            this.sandbox.stub(GH.RapidBoard.State, 'getMode');
            this.sandbox.stub(GH.RapidBoard.ReportSubnavigatorContextProvider, 'itemList');
            this.sandbox.stub(GH.RapidBoard.ReportSubnavigatorContextProvider, 'selectedItem');
            this.sandbox.stub(GH.RapidBoard.ReportSubnavigatorContextProvider, 'selectionHandler');
            this.sandbox.stub(GH.RapidBoard.ReportSubnavigatorContextProvider, 'onLoad');
            this.sandbox.stub(AJS, 'trigger');
            this.sandbox.stub(FeatureFlagManager, 'isFeatureEnabled').withArgs('com.atlassian.jira.agile.darkfeature.sprint.goal').returns(true);

            this.originalSubnavigatorConstructor = JIRA.Projects.Subnavigator;
            this.subnavigatorOnSpy = sinon.spy();
            this.subnavigatorShowSpy = sinon.spy();
            this.subnavigatorConstructorInvocationCount = 0;
            this.subnavigatorConstructorInvocationArguments = [];

            var self = this;
            JIRA.Projects.Subnavigator = function() {
                self.subnavigatorConstructorInvocationCount++;
                self.subnavigatorConstructorInvocationArguments.push(arguments[0]);

                this.on = self.subnavigatorOnSpy;
                this.show = self.subnavigatorShowSpy;
            };
        },
        teardown: function() {
            this.sandbox.restore();
            JIRA.Projects.Subnavigator = this.originalSubnavigatorConstructor;
        }
    });

    test('should invoke onLoad on context provider, set header, and render subnavigator when render is invoked', function() {
        GH.RapidBoard.State.getMode.returns('report');
        this.stub(GH.RapidBoard.SubnavigatorController, 'setHeader');
        this.stub(GH.RapidBoard.SubnavigatorController, 'renderSubnavigator');

        GH.RapidBoard.SubnavigatorController.render();

        sinon.assert.calledOnce(GH.RapidBoard.SubnavigatorController.setHeader);
        sinon.assert.calledOnce(GH.RapidBoard.SubnavigatorController.renderSubnavigator);
        sinon.assert.calledOnce(GH.RapidBoard.ReportSubnavigatorContextProvider.onLoad);

        GH.RapidBoard.SubnavigatorController.setHeader.restore();
        GH.RapidBoard.SubnavigatorController.renderSubnavigator.restore();
    });

    test('should configure context providers for all three modes', function() {
        notEqual(typeof(GH.RapidBoard.SubnavigatorController.contextProviders.plan), 'undefined');
        notEqual(typeof(GH.RapidBoard.SubnavigatorController.contextProviders.work), 'undefined');
        notEqual(typeof(GH.RapidBoard.SubnavigatorController.contextProviders.report), 'undefined');
    });

    test('should set header and wipe out old header', function() {
        this.stub(GH.RapidBoard.ReportSubnavigatorContextProvider, 'header');
        GH.RapidBoard.ReportSubnavigatorContextProvider.header.returns('Board name');
        GH.RapidBoard.State.getMode.returns('report');

        this.$qunitFixture.append('<div id="ghx-view-selector"><div id="old-header">This is the old header</div></div>');
        equal(AJS.$('#old-header').length, 1);
        equal(AJS.$('.subnav-page-header').length, 0);

        GH.RapidBoard.SubnavigatorController.setHeader();

        equal(AJS.$('#old-header').length, 0);
        equal(AJS.$('.subnav-page-header').length, 1);
        equal(AJS.$('.subnav-page-header').text(), GH.RapidBoard.ReportSubnavigatorContextProvider.header());
    });

    test('setting header should also set goal in work mode', function () {
        this.stub(GH.RapidBoard.WorkSubnavigatorContextProvider, 'getSprintGoal').returns('This is a goal');
        GH.RapidBoard.State.getMode.returns('work');

        this.$qunitFixture.append('<div id="ghx-view-selector"></div>');

        GH.RapidBoard.SubnavigatorController.setHeader();
        equal(AJS.$('#ghx-sprint-goal').length, 1);
        equal(AJS.$('#ghx-sprint-goal').text(), GH.RapidBoard.WorkSubnavigatorContextProvider.getSprintGoal());
    });

    test('should not render sprint goal when it is empty', function () {
        this.stub(GH.RapidBoard.WorkSubnavigatorContextProvider, 'getSprintGoal').returns('');
        GH.RapidBoard.State.getMode.returns('work');

        this.$qunitFixture.append('<div id="ghx-view-selector"></div>');

        GH.RapidBoard.SubnavigatorController.renderSprintGoal();
        equal(AJS.$('#ghx-sprint-goal').length, 0);
    });

    test('should set sprint goal and wipe out old sprint goal in work mode', function () {
        const oldGoal = 'Old goal';
        this.stub(GH.RapidBoard.WorkSubnavigatorContextProvider, 'getSprintGoal').returns('This is a goal');
        GH.RapidBoard.State.getMode.returns('work');

        this.$qunitFixture.append(`<div id="ghx-view-selector"><div id="ghx-sprint-goal">${oldGoal}</div></div>`);
        equal(AJS.$('#ghx-sprint-goal').length, 1);
        equal(AJS.$('#ghx-sprint-goal').text(), oldGoal);

        GH.RapidBoard.SubnavigatorController.renderSprintGoal();

        equal(AJS.$('#ghx-sprint-goal').length, 1);
        notEqual(AJS.$('#ghx-sprint-goal').text(), oldGoal);
        equal(AJS.$('#ghx-sprint-goal').text(), GH.RapidBoard.WorkSubnavigatorContextProvider.getSprintGoal());
    });

    test('should not set sprint goal in plan and report mode', function () {
        this.$qunitFixture.append('<div id="ghx-view-selector"></div>');

        GH.RapidBoard.State.getMode.returns('plan');

        GH.RapidBoard.SubnavigatorController.renderSprintGoal();
        equal(AJS.$('#ghx-view-selector').length, 1);
        equal(AJS.$('#ghx-sprint-goal').length, 0);

        GH.RapidBoard.State.getMode.returns('report');

        GH.RapidBoard.SubnavigatorController.renderSprintGoal();
        equal(AJS.$('#ghx-view-selector').length, 1);
        equal(AJS.$('#ghx-sprint-goal').length, 0);
    });

    test('should render subnavigator if mode requires subnavigator', function () {
        GH.RapidBoard.State.getMode.returns('report');
        GH.RapidBoard.ReportSubnavigatorContextProvider.itemList.returns([1, 2, 3]);
        GH.RapidBoard.ReportSubnavigatorContextProvider.selectedItem.returns(1);

        this.$qunitFixture.append('<div id="ghx-view-selector"><h1></h1></div>');

        equal(AJS.$('#subnav-trigger').length, 0);
        equal(AJS.$('#subnav-opts').length, 0);
        equal(this.subnavigatorConstructorInvocationCount, 0);

        GH.RapidBoard.SubnavigatorController.renderSubnavigator();

        equal(AJS.$('#subnav-trigger').length, 1);
        equal(AJS.$('#subnav-opts').length, 1);

        equal(this.subnavigatorConstructorInvocationCount, 1);
        notEqual(typeof this.subnavigatorConstructorInvocationArguments[0].triggerPlaceholder, 'undefined');
        notEqual(typeof this.subnavigatorConstructorInvocationArguments[0].contentPlaceholder, 'undefined');
        deepEqual(this.subnavigatorConstructorInvocationArguments[0].itemGroups, [1,2,3]);
        equal(this.subnavigatorConstructorInvocationArguments[0].selectedItem, 1);

        sinon.assert.calledOnce(this.subnavigatorOnSpy);
        sinon.assert.calledWith(this.subnavigatorOnSpy, 'itemSelected');

        var itemSelectedEvent = {
            preventDefault: function() {},
            item: {
                id: 5
            }
        };
        var preventDefaultSpy = this.spy(itemSelectedEvent, 'preventDefault');
        var itemSelectedCallback = this.subnavigatorOnSpy.args[0][1];

        itemSelectedCallback(itemSelectedEvent);

        sinon.assert.calledOnce(preventDefaultSpy);
        sinon.assert.calledOnce(GH.RapidBoard.ReportSubnavigatorContextProvider.selectionHandler);
        sinon.assert.calledWith(GH.RapidBoard.ReportSubnavigatorContextProvider.selectionHandler, 5);

        sinon.assert.calledOnce(this.subnavigatorShowSpy);
    });

    test("should emit an analytic event when subnavigator is loaded", function(){
        GH.RapidBoard.State.getMode.returns('report');
        GH.RapidBoard.ReportSubnavigatorContextProvider.itemList.returns([1, 2, 3]);
        GH.RapidBoard.ReportSubnavigatorContextProvider.selectedItem.returns(1);
        this.$qunitFixture.append('<div id="ghx-view-selector"><h1></h1></div>');

        GH.RapidBoard.SubnavigatorController.renderSubnavigator();

        sinon.assert.calledOnce(AJS.trigger);
        sinon.assert.calledWith(AJS.trigger, 'analyticsEvent', {
            name: 'jira.project.sidebar.report.select',
            data: {
                reportId: 1,
                initialLoad: true
            }
        });
    });

    test("should emit an analytic event when a subnavigator item is selected", function(){
        GH.RapidBoard.State.getMode.returns('report');
        GH.RapidBoard.ReportSubnavigatorContextProvider.itemList.returns([1, 2, 3]);
        GH.RapidBoard.ReportSubnavigatorContextProvider.selectedItem.returns(1);
        this.$qunitFixture.append('<div id="ghx-view-selector"><h1></h1></div>');

        GH.RapidBoard.SubnavigatorController.renderSubnavigator();

        var itemSelectedEvent = {
            preventDefault: function() {},
            item: {
                id: 5
            }
        };
        var itemSelectedCallback = this.subnavigatorOnSpy.args[0][1];
        itemSelectedCallback(itemSelectedEvent);

        sinon.assert.calledTwice(AJS.trigger); //Called once when subnavigator is loaded and again when an item is selected.

        var triggerArgs = AJS.trigger.getCall(1).args;
        equal(triggerArgs[0], 'analyticsEvent');
        deepEqual(triggerArgs[1], {
            name: 'jira.project.sidebar.report.select',
            data: {
                reportId: 5,
                initialLoad: false
            }
        });
    });

    test('should pass less settings when Subnavigator has old API', function(){
        var mode = 'report';
        GH.RapidBoard.State.getMode.returns(mode);
        var contextProvider = fakeContextProvider();
        var Subnavigator = function(){};

        var expectedResult = {
            id: mode,
            triggerPlaceholder: '#subnav-trigger',
            contentPlaceholder: '#subnav-opts',
            itemGroups: contextProvider.itemList(),
            selectedItem: contextProvider.selectedItem()
        };
        var actualResult = GH.RapidBoard.SubnavigatorController._getSubnavSettings(Subnavigator, contextProvider);

        deepEqual(actualResult, expectedResult, 'Settings do not match what is expected');
    });

    test('should pass more settings when Subnavigator has new API', function(){
        var mode = 'report';
        GH.RapidBoard.State.getMode.returns(mode);
        var contextProvider = fakeContextProvider();
        var Subnavigator = function(){};
        Subnavigator.prototype._shouldShowTriggerView = function(){};

        var expectedResult = {
            id: mode,
            triggerPlaceholder: '#subnav-trigger',
            contentPlaceholder: '#subnav-opts',
            itemGroups: contextProvider.itemList(),
            selectedItem: contextProvider.selectedItem(),
            titlePlaceholder: '#subnav-title',
            hideSelectedItem: true,
            changeViewText: contextProvider.getChangeViewText()
        };
        var actualResult = GH.RapidBoard.SubnavigatorController._getSubnavSettings(Subnavigator, contextProvider);

        deepEqual(actualResult, expectedResult, 'Settings do not match what is expected');
    });

    function fakeContextProvider() {
        return {
            itemList: sinon.spy(),
            selectedItem: sinon.spy(),
            getChangeViewText: sinon.spy()
        };
    }

});
