// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package ksp.com.intellij.psi.impl;

import ksp.com.intellij.ide.IconProvider;
import ksp.com.intellij.ide.TypePresentationService;
import ksp.com.intellij.lang.Language;
import ksp.com.intellij.openapi.project.Project;
import ksp.com.intellij.pom.*;
import ksp.com.intellij.psi.PsiElement;
import ksp.com.intellij.psi.PsiFile;
import ksp.com.intellij.psi.PsiTarget;
import ksp.com.intellij.util.IncorrectOperationException;
import ksp.org.jetbrains.annotations.NonNls;
import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;

import javax.swing.*;

public class PomTargetPsiElementImpl extends RenameableFakePsiElement implements PomTargetPsiElement {
  private final PomTarget myTarget;
  private final Project myProject;

  public PomTargetPsiElementImpl(@NotNull PsiTarget target) {
    this(target.getNavigationElement().getProject(), target);
  }

  public PomTargetPsiElementImpl(@NotNull Project project, @NotNull PomTarget target) {
    super(null);
    myProject = project;
    myTarget = target;
  }

  @Override
  public @NotNull PomTarget getTarget() {
    return myTarget;
  }

  @Override
  public String getName() {
    if (myTarget instanceof PomNamedTarget) {
      return ((PomNamedTarget)myTarget).getName();
    }
    return null;
  }

  @Override
  public boolean isWritable() {
    if (myTarget instanceof PomRenameableTarget) {
      return ((PomRenameableTarget<?>)myTarget).isWritable();
    }
    return false;
  }

  @Override
  public String getTypeName() {
    throw new UnsupportedOperationException("Method getTypeName is not yet implemented for " + myTarget.getClass().getName() + "; see PomDescriptionProvider");
  }

  @Override
  public @NotNull PsiElement getNavigationElement() {
    if (myTarget instanceof PsiTarget) {
      return ((PsiTarget)myTarget).getNavigationElement();
    }
    return super.getNavigationElement();
  }

  @Override
  public Icon getIcon() {
    for (IconProvider iconProvider : IconProvider.EXTENSION_POINT_NAME.getIterable()) {
      if (iconProvider instanceof PomIconProvider) {
        Icon icon = ((PomIconProvider)iconProvider).getIcon(myTarget, 0);
        if (icon != null) {
          return icon;
        }
      }
    }

    Icon icon = TypePresentationService.getService().getIcon(myTarget);
    if (icon != null) return icon;

    if (myTarget instanceof PsiTarget) {
      return ((PsiTarget)myTarget).getNavigationElement().getIcon(0);
    }
    return null;
  }

  @Override
  public boolean isValid() {
    if (myTarget instanceof PsiTarget) {
      return ((PsiTarget)myTarget).getNavigationElement().isValid();
    }

    return myTarget.isValid();
  }

  @Override
  public PsiElement setName(@NonNls @NotNull String name) throws IncorrectOperationException {
    if (myTarget instanceof PomRenameableTarget) {
      ((PomRenameableTarget<?>)myTarget).setName(name);
      return this;
    }
    throw new UnsupportedOperationException("Cannot rename " + myTarget);
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    PomTargetPsiElementImpl that = (PomTargetPsiElementImpl)o;
    return myTarget.equals(that.myTarget);
  }

  @Override
  public int hashCode() {
    return myTarget.hashCode();
  }

  @Override
  public boolean isEquivalentTo(PsiElement another) {
    return equals(another) ||
           (another != null && myTarget instanceof PsiTarget && another.isEquivalentTo(((PsiTarget)myTarget).getNavigationElement()));
  }

  @Override
  public PsiElement getContext() {
    if (myTarget instanceof PsiTarget) {
      return ((PsiTarget)myTarget).getNavigationElement();
    }
    return null;
  }

  @Override
  public @Nullable PsiElement getParent() {
    return null;
  }

  @Override
  public void navigate(boolean requestFocus) {
    myTarget.navigate(requestFocus);
  }

  @Override
  public boolean canNavigate() {
    return myTarget.canNavigate();
  }

  @Override
  public boolean canNavigateToSource() {
    return myTarget.canNavigateToSource();
  }

  @Override
  public @Nullable PsiFile getContainingFile() {
    if (myTarget instanceof PsiTarget) {
      return ((PsiTarget)myTarget).getNavigationElement().getContainingFile();
    }
    return null;
  }

  @Override
  public @NotNull Language getLanguage() {
    if (myTarget instanceof PsiTarget) {
      return ((PsiTarget)myTarget).getNavigationElement().getLanguage();
    }
    return Language.ANY;
  }

  @Override
  public @NotNull Project getProject() {
    return myProject;
  }

  @Override
  public String getLocationString() {
    if (myTarget instanceof PsiTarget) {
      PsiFile file = ((PsiTarget)myTarget).getNavigationElement().getContainingFile();
      if (file != null) {
        return "(" + file.getName() + ")";
      }
    }
    return super.getLocationString();
  }
}
